
import old_package.utils as utils


class GlossaryError(utils.ErrorObject):

    def __init__(self, response):
        """An Error object for the alerts endpoints.

        Attributes
        ----------
        type: str
            A URI reference (RFC3986) that identifies the problem type.
        title: str
            A short, human-readable summary of the problem type.
        status: int
            The HTTP status code (RFC7231, Section 6) generated by the origin server for this occurrence of the problem.
            Minimum: 100, Max 999
        detail: str
            A human-readable explanation specific to this occurrence of the problem.
        instance: string
            A URI reference that identifies the specific occurrence of the problem.
        correlationId: str
            A unique identifier for the request, used for NWS debugging purposes.
            Please include this identifier with any correspondence to help the API maintainers investigate your issue.
        """

        super().__init__(response)


class Glossary(utils.ObjectIterator):
    """Glossary object for ``/glossary``.

    Attributes
    ----------
    response_headers: requests.structures.CaseInsensitiveDict
        A dictionary containing the response headers.

    terms: dict
        A dictionary to hold the information regarding the glossary.
    """

    n_errors = 0
    has_any_request_errors = False

    def __init__(self, user_agent):
        response = utils.request("https://api.weather.gov/glossary", headers=user_agent)
        if not response.ok:
            self.terms = GlossaryError(response)
            self.n_errors += 1
            self.has_any_request_errors = True
        else:
            self.terms = {}
            for element in response.json()['glossary']:
                self.terms[element['term']] = element['definition']

        self.response_headers = response.headers  # requests.structures.CaseInsensitiveDict

    # Utilities iterator only works for lists/tuples, not dictionaries.
    def __iter__(self):
        """Allows for iteration though self.terms."""
        self._terms_index = 0
        return self

    def __next__(self):
        """Allows for iteration through self.alerts."""
        if self._terms_index < len(self.terms):
            val = self.terms[self._terms_index]
            self._terms_index += 1
            return val
        else:
            raise StopIteration

    def __getitem__(self, term):
        """Allows for alerts object to be directly indexable."""
        if isinstance(term, int):  # if they're tyring to index like a list
            raise KeyError("Glossary not list, index using a term (i.e. Tornado).")
        return self.terms[term]

