"""Module for configuring the data global attribute 

"""

import json

class configurations:
    """Global attributes definition
    
    This class defines all global attributes 
    that will be writen on the dataset
    
    Parameters
    ----------
    lst : object
        an instance of the lidarSuit package

    """
    
    def __init__(self, lst=None):
        
        self.loadVersion(lst)
        self.loadReference()
        self.loadInstitution()
        self.loadInstrument()
        self.loadSite()
        self.loadContact()
        self.loadEmail()
        self.loadComments()

        return None
    
    def loadVersion(self, lst):
        """
        It identifies the lidarSuit version
        and writes it to the configuration file
        
        Parameters
        ----------
        lst : object
            a instance of the lidarSuit package
        
        """
        
        if lst == None:
            self.lstVersion = 'temporary config file'
        else:
            self.lstVersion = lst.__version__
            
        return self

    def loadInstitution(self, institution='institution name'):
        """
        It defines the institution affiliation name
        
        Parameters
        ----------
        institution : str
            institution name
        
        """
        
        self.institution = institution

        return self

    def loadInstrument(self, instrument='instrument name'):
        """
        It defines the instrument name
        
        Parameters
        ----------
        instrument : str
            name of the instrument used during the experiment
        
        """

        self.instrument = instrument

        return self

    def loadSite(self, site='site name'):
        """
        It defines the name of the experimental site
        
        Parameters
        ----------
        site : str
            name of the experimental site
        
        """

        self.site = site

        return self

    def loadContact(self, contact='contact person'):
        """
        It defines the author's name
        
        Parameters
        ----------
        contact : str
            name of the contact person
            
        """

        self.contact = contact

        return self

    def loadEmail(self, email='contact email'):
        """
        It defines the contacting email
        
        Parameters
        ----------
        email : str
            contact email

        """
        self.email = email

        return self
    
    def loadReference(self, reference='Generated by lidarSuit version: {0}'):
        """
        It loads the lidarSuit's version used for
        processing the data
        
        Parameters
        ----------
        reference : str
            lidarSuit version used to process the data

        """

        self.references = reference.format(self.lstVersion)

        return self
    
    def loadComments(self, comments='General comments'):
        """
        It defines additional comments
        
        Parameters
        ----------
        comments : str
            additional comments
        
        """
        self.comments = comments
        
        return self

    def generateConf(self):
        """
        It writes and saves all defined global attributes.
        
        """

        configDic = {}
        
        configDic['references'] = self.references
        configDic['institution'] = self.institution
        configDic['instrument_name'] = self.instrument
        configDic['site_name'] = self.site
        configDic['comments'] = self.comments
        configDic['contact_person'] = self.contact
        configDic['email'] = self.email

        configJS = json.dumps(configDic)
        configFile = open('config.json','w')
        configFile.write(configJS)
        configFile.close()

        return None

    def loadConfFile(self, filePath='config.json'):
        """
        It loads the pre-defined global attributes
        from the config.json, if it exists. 
        
        Parameters
        ----------
        filePath : str
            the path to the configuration file (config.json)
        
        """
        
        try:
            configDic = json.load(open(filePath))
            
        except FileNotFoundError:
            
            print('You do not have a config file yet')
            print('a temporary config file was generated')
            print('See the documentation for generating it')
            self.generateConf()
            configDic = json.load(open(filePath))
        
        self.loadReference(configDic['references'])
        self.loadInstitution(configDic['institution'])
        self.loadInstrument(configDic['instrument_name'])
        self.loadComments(configDic['comments'])
        self.loadSite(configDic['site_name'])
        self.loadContact(configDic['contact_person'])
        self.loadEmail(configDic['email'])
        
        return self