"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqlServerSeeder = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const cr = require("@aws-cdk/custom-resources");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const tmp = require("tmp");
const fs = require("fs");
const path = require("path");
class SqlServerSeeder extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (!props.database.secret) {
            throw new Error("Database does not have secret value assigned");
        }
        if (!fs.existsSync(props.createScriptPath)) {
            throw new Error("Create script does not exist: " + props.createScriptPath);
        }
        if (props.deleteScriptPath && !fs.existsSync(props.deleteScriptPath)) {
            throw new Error("Create script does not exist: " + props.deleteScriptPath);
        }
        const destinationBucket = new s3.Bucket(this, 'bucket', { removalPolicy: cdk.RemovalPolicy.DESTROY });
        this.prepareSqlScripts(id, props, destinationBucket);
        const sqlSeederLambda = new lambda.Function(this, 'lambda', {
            code: new lambda.AssetCode('./lambda/sqlserver-seeder.zip'),
            handler: 'seed::seed.Bootstrap::ExecuteFunction',
            timeout: cdk.Duration.seconds(300),
            runtime: lambda.Runtime.DOTNET_CORE_3_1,
            memorySize: props.memorySize,
            vpc: props.vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PRIVATE
            },
            environment: {
                "DbEndpoint": props.database.dbInstanceEndpointAddress,
                "SecretArn": (_a = props.database.secret) === null || _a === void 0 ? void 0 : _a.secretArn,
                "ScriptsBucket": destinationBucket.bucketName,
                "RunOnDelete": `${!!props.deleteScriptPath}`
            }
        });
        const sqlSeederProvider = new cr.Provider(this, 'provider', {
            onEventHandler: sqlSeederLambda
        });
        const sqlSeederResource = new cdk.CustomResource(this, 'resource', {
            serviceToken: sqlSeederProvider.serviceToken,
            properties: {
                "IgnoreSqlErrors": !!props.ignoreSqlErrors
            }
        });
        sqlSeederResource.node.addDependency(props.database);
        // allow access
        destinationBucket.grantRead(sqlSeederLambda);
        (_b = props.database.secret) === null || _b === void 0 ? void 0 : _b.grantRead(sqlSeederLambda);
        // enable connection to RDS instance
        sqlSeederLambda.connections.allowTo(props.database, ec2.Port.tcp(props.port));
    }
    prepareSqlScripts(id, props, destinationBucket) {
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            fs.copyFileSync(props.createScriptPath, path.join(dir, 'create.sql'));
            if (props.deleteScriptPath) {
                fs.copyFileSync(props.deleteScriptPath, path.join(dir, 'delete.sql'));
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false
            });
        });
    }
}
exports.SqlServerSeeder = SqlServerSeeder;
//# sourceMappingURL=data:application/json;base64,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