"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.TaintEffect = exports.CapacityType = exports.NodegroupAmiType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cluster_1 = require("./cluster");
const eks_generated_1 = require("./eks.generated");
const instance_types_1 = require("./instance-types");
/**
 * The AMI type for your node group.
 *
 * GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 *
 * @stability stable
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
    NodegroupAmiType["AL2_ARM_64"] = "AL2_ARM_64";
    NodegroupAmiType["BOTTLEROCKET_ARM_64"] = "BOTTLEROCKET_ARM_64";
    NodegroupAmiType["BOTTLEROCKET_X86_64"] = "BOTTLEROCKET_x86_64";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * Capacity type of the managed node group.
 *
 * @stability stable
 */
var CapacityType;
(function (CapacityType) {
    CapacityType["SPOT"] = "SPOT";
    CapacityType["ON_DEMAND"] = "ON_DEMAND";
})(CapacityType = exports.CapacityType || (exports.CapacityType = {}));
/**
 * Effect types of kubernetes node taint.
 *
 * @stability stable
 */
var TaintEffect;
(function (TaintEffect) {
    TaintEffect["NO_SCHEDULE"] = "NO_SCHEDULE";
    TaintEffect["PREFER_NO_SCHEDULE"] = "PREFER_NO_SCHEDULE";
    TaintEffect["NO_EXECUTE"] = "NO_EXECUTE";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * The Nodegroup resource class.
 *
 * @stability stable
 */
class Nodegroup extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_eks_NodegroupProps(props);
        this.cluster = props.cluster;
        this.desiredSize = (_c = (_b = props.desiredSize) !== null && _b !== void 0 ? _b : props.minSize) !== null && _c !== void 0 ? _c : 2;
        this.maxSize = (_d = props.maxSize) !== null && _d !== void 0 ? _d : this.desiredSize;
        this.minSize = (_e = props.minSize) !== null && _e !== void 0 ? _e : 1;
        core_1.withResolved(this.desiredSize, this.maxSize, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (desired > max) {
                throw new Error(`Desired capacity ${desired} can't be greater than max size ${max}`);
            }
        });
        core_1.withResolved(this.desiredSize, this.minSize, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Minimum capacity ${min} can't be greater than desired size ${desired}`);
            }
        });
        if (props.launchTemplateSpec && props.diskSize) {
            // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
            // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
            throw new Error('diskSize must be specified within the launch template');
        }
        if (props.instanceType && props.instanceTypes) {
            throw new Error('"instanceType is deprecated, please use "instanceTypes" only.');
        }
        if (props.instanceType) {
            core_1.Annotations.of(this).addWarning('"instanceType" is deprecated and will be removed in the next major version. please use "instanceTypes" instead');
        }
        const instanceTypes = (_f = props.instanceTypes) !== null && _f !== void 0 ? _f : (props.instanceType ? [props.instanceType] : undefined);
        let expectedAmiType = undefined;
        if (instanceTypes && instanceTypes.length > 0) {
            // if the user explicitly configured instance types, we can calculate the expected ami type.
            expectedAmiType = getAmiType(instanceTypes);
            // if the user explicitly configured an ami type, make sure its the same as the expected one.
            if (props.amiType && props.amiType !== expectedAmiType) {
                throw new Error(`The specified AMI does not match the instance types architecture, either specify ${expectedAmiType} or dont specify any`);
            }
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            // if a launch template is configured, we cannot apply a default since it
            // might exist in the launch template as well, causing a deployment failure.
            amiType: props.launchTemplateSpec !== undefined ? props.amiType : ((_g = props.amiType) !== null && _g !== void 0 ? _g : expectedAmiType),
            capacityType: props.capacityType ? props.capacityType.valueOf() : undefined,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_h = props.forceUpdate) !== null && _h !== void 0 ? _h : true,
            // note that we don't check if a launch template is configured here (even though it might configure instance types as well)
            // because this doesn't have a default value, meaning the user had to explicitly configure this.
            instanceTypes: instanceTypes === null || instanceTypes === void 0 ? void 0 : instanceTypes.map(t => t.toString()),
            labels: props.labels,
            taints: props.taints,
            launchTemplate: props.launchTemplateSpec,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster instanceof cluster_1.Cluster) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
            // the controller runs on the worker nodes so they cannot
            // be deleted before the controller.
            (_j = this.cluster.albController) === null || _j === void 0 ? void 0 : _j.node.addDependency(this);
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes.
     *
     * @stability stable
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
_a = JSII_RTTI_SYMBOL_1;
Nodegroup[_a] = { fqn: "@aws-cdk/aws-eks.Nodegroup", version: "1.134.0" };
function getAmiTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? NodegroupAmiType.AL2_ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_ARM_64 :
            instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_X86_64_GPU :
                instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodegroupAmiType.AL2_X86_64_GPU :
                    NodegroupAmiType.AL2_X86_64;
}
// this function examines the CPU architecture of every instance type and determines
// what ami type is compatible for all of them. it either throws or produces a single value because
// instance types of different CPU architectures are not supported.
function getAmiType(instanceTypes) {
    const amiTypes = new Set(instanceTypes.map(i => getAmiTypeForInstanceType(i)));
    if (amiTypes.size == 0) { // protective code, the current implementation will never result in this.
        throw new Error(`Cannot determine any ami type comptaible with instance types: ${instanceTypes.map(i => i.toString).join(',')}`);
    }
    if (amiTypes.size > 1) {
        throw new Error('instanceTypes of different CPU architectures is not allowed');
    }
    return amiTypes.values().next().value;
}
//# sourceMappingURL=data:application/json;base64,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