"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDeploymentGroup = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambda'));
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.postHook && this.postHook.functionName }),
            },
        };
    }
    /**
     * Import an Lambda Deployment Group defined either outside the CDK app, or in a different AWS region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     * @param postHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param grantee to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
    }
}
//# sourceMappingURL=data:application/json;base64,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