"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BudgetNotifier = void 0;
const aws_budgets_1 = require("@aws-cdk/aws-budgets");
const core_1 = require("@aws-cdk/core");
const NotificationType_1 = require("./NotificationType");
const TimeUnit_1 = require("./TimeUnit");
class BudgetNotifier extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.threshold <= 0) {
            throw new Error("Thresholds less than or equal to 0 are not allowed.");
        }
        const costFilters = this.createCostFilters(props);
        const subscribers = this.createSubscribers(props);
        new aws_budgets_1.CfnBudget(this, "MonthlyBudget_" + id, {
            budget: {
                budgetType: "COST",
                timeUnit: props.timeUnit ? props.timeUnit : TimeUnit_1.TimeUnit.MONTHLY,
                budgetLimit: {
                    amount: props.limit,
                    unit: props.unit,
                },
                costFilters: costFilters,
            },
            notificationsWithSubscribers: [
                {
                    notification: {
                        comparisonOperator: "GREATER_THAN",
                        threshold: props.threshold,
                        thresholdType: "PERCENTAGE",
                        notificationType: props.notificationType
                            ? props.notificationType
                            : NotificationType_1.NotificationType.ACTUAL,
                    },
                    subscribers: subscribers,
                },
            ],
        });
    }
    createSubscribers(props) {
        const subscribers = new Array();
        for (const recipient of props.recipients) {
            subscribers.push({
                address: recipient,
                subscriptionType: "EMAIL",
            });
        }
        return subscribers;
    }
    createCostFilters(props) {
        const tags = [];
        if (props.application) {
            tags.push("user:Application$" + props.application);
        }
        if (props.costCenter) {
            tags.push("user:Cost Center$" + props.costCenter);
        }
        if (props.service) {
            tags.push("user:Service$" + props.service);
        }
        const costFilters = {};
        if (tags) {
            costFilters["TagKeyValue"] = tags;
        }
        const availabilityZones = [];
        if (props.availabilityZones) {
            for (const az of props.availabilityZones) {
                availabilityZones.push(az);
            }
            costFilters["AZ"] = availabilityZones;
        }
        return costFilters;
    }
}
exports.BudgetNotifier = BudgetNotifier;
//# sourceMappingURL=data:application/json;base64,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