import os
import sys
from copy import deepcopy
import tempfile
from typing import Optional

sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), "..")))

import numpy as np
import pytest

from seqlike import SeqLike, SequenceLike

from seqlike.codon_tables import yeast_codon_table, ecoli_codon_table, codon_table_to_codon_map
from seqlike.SeqLike import NT, AA, STANDARD_NT, STANDARD_AA

from . import test_path


from hypothesis import given, assume
from hypothesis.strategies import composite, text, integers, sampled_from, iterables, lists

CODONS = set([])
for first_letter in "ACUG":
    for second_letter in "ACUG":
        for third_letter in "ACUG":
            CODONS.add(first_letter + second_letter + third_letter)
CODONS = list(CODONS)

NUMBERS = [0, 1, 2, 3, 4, 5]

ALPHABETS = [NT, AA, STANDARD_NT, STANDARD_AA, CODONS, NUMBERS]


@composite
def string_sequences(draw, alphabet: Optional[str] = None, min_size: int = 2) -> tuple:
    """Composite Hypothesis strategy to generate strings from one of the alphabets.

    This test handles the cases where seq_type, sequence, and alphabet are all set.
    """
    max_size = draw(integers(min_value=min_size, max_value=1000))
    if alphabet is None:
        alphabet = draw(sampled_from(ALPHABETS))
    sequence = draw(lists(elements=sampled_from(alphabet), min_size=min_size, max_size=max_size))
    return sequence, alphabet


@given(string_sequences())
def test_init_hyp(sequence_and_alphabet: tuple):
    """Test initialization of SequenceLike objects from strings."""
    sequence, alphabet = sequence_and_alphabet
    seq = SequenceLike(sequence, alphabet=alphabet)
    assert seq.alphabet == sorted(alphabet)


def test_init_custom_alphabet():
    """Test that custom alphabets are passed right through."""
    sequence = "ATGC"
    seq = SequenceLike(sequence, alphabet="*-ACDEFG1234")
    assert seq.alphabet == sorted("*-ACDEFG1234")


@given(string_sequences())
def test_auto_alphabet(sequence_and_alphabet: tuple):
    """Test initialization of SeqLike objects from strings."""
    sequence, alphabet = sequence_and_alphabet
    seq1 = SequenceLike(sequence)
    seq2 = SequenceLike(sequence=sequence, alphabet=alphabet)

    # we know that the sequence is made from `alphabet`.

    # the alphabet must be as long as all the unique letters in `sequence`
    assert len(seq1.alphabet) == len(set(sequence))

    # the autogenerated alphabet must be a subset of the full potential alphabet
    assert set(seq1.alphabet).issubset(set(seq2.alphabet))


# @given(string_sequences())
def test_SequenceLike_interconversion():  # sequence_and_alphabet: tuple):
    """Test that SeqLike interconversion works properly.

    Suggested TODO: Split out SeqLike interconversion into a parametrized interconversion
    based on target types.
    """
    # sequence, alphabet = sequence_and_alphabet
    #  sequence = "".join(sequence)
    sequence, alphabet = ["-", "-"], "-ACGTUNBDHKMRSVWY"

    seq1 = SequenceLike(sequence, alphabet)

    d1 = seq1.to_onehot()
    assert isinstance(d1, np.ndarray)
    # raise ValueError("bingo!")
    assert d1.shape[0] == len(sequence)
    assert d1.shape[1] == len(alphabet)

    # e1 = SeqLike(d1, "dna", alphabet=seq1.alphabet).to_str()
    # e2 = SeqLike(d2, "aa", alphabet=seq2.alphabet).to_str()
    # assert e1 == a1
    # assert e2 == a2
    # # when interconverting, regenerate the letter_annotations as if starting from scratch
    # seqnums2 = [str(i + 1) for i in range(len(seq2))]
    # assert seq2.letter_annotations == SeqLike(seq2, "aa").letter_annotations == {"seqnums": seqnums2}
    # seq3 = SeqLike(seq2, "dna")
    # seqnums3 = [str(i + 1) for i in range(len(seq3))]
    # assert seq3.letter_annotations == SeqLike(seq3, "nt").letter_annotations == {"seqnums": seqnums3}
    # assert seq3[:2].letter_annotations["seqnums"] == seqnums3[:2]


# CODON_TABLES = {"ecoli_k12": yeast_codon_table, "Kazusa_yeast": yeast_codon_table}


# def assert_matched_properties(s1, s2):
#     """Test that SeqRecord-like sequence-independent attributes match."""
#     # retains SeqRecord-like metadata, e.g., `id` and `description`
#     assert s1.id == s2.id
#     assert s1.name == s2.name
#     assert s1.description == s2.description
#     # SeqRecord automatically sets the molecule_type annotation, so just check keys
#     assert s1.annotations.keys() == s2.annotations.keys()
#     assert s1.dbxrefs == s2.dbxrefs


# def assert_back_translate_properties(s):
#     """Test for back_translate properties.

#     This collection of assertions repeatedly comes up
#     so we have refactored them out into a separate function.
#     """
#     assert isinstance(s.back_translate(), SeqLike)
#     assert_matched_properties(s.back_translate(), s)
#     # but letter-specific metadata is changed
#     assert s.back_translate().letter_annotations != s.letter_annotations

#     # works, overrides self.codon_map
#     # assert isinstance(s2.back_translate(codon_map=codon_map2), SeqLike)

#     seqs = [s.back_translate().nt().to_str() for _ in range(10)]
#     assert len(seqs) == 10
#     assert len(set(seqs)) <= len(seqs)

#     # DNA sequences are also valid AA sequences,
#     # and we can only backtranslate a AA sequence.
#     with pytest.raises(TypeError):
#         s.nt().back_translate()


# def test_upper():
#     # TODO: Add docstring
#     # TODO: Parametrize with hypothesis.
#     s1 = SeqLike("TCgCAcActgcA", seq_type="nt")
#     s2 = SeqLike("GEgdatygKLTlkfiCTT", seq_type="aa")
#     assert isinstance(s1.upper(), SeqLike)
#     assert isinstance(s2.upper(), SeqLike)
#     assert str(s1.upper()) == str(s1).upper()
#     assert str(s2.upper()) == str(s2).upper()


# def test_count():
#     # TODO: Add docstring
#     # TODO: Parametrize with hypothesis.
#     s1 = SeqLike("TCGCACACTGCA", seq_type="nt")
#     s2 = SeqLike("GEGDRTYNKLTLIFQCTT", seq_type="aa")
#     assert isinstance(s1.count("T"), int)
#     assert isinstance(s2.count("Y"), int)
#     for letter in s1:
#         assert s1.count(letter) == str(s1).count(letter)
#     for letter in s2:
#         assert s2.count(letter) == str(s2).count(letter)


# def test_find():
#     # TODO: Add docstring
#     # TODO: Parametrize with hypothesis.
#     s1 = SeqLike("TCGCACACTGCATCGCACACTGCA", seq_type="dna")

#     sub1 = SeqLike("ACTG", seq_type="dna")
#     found = s1.find(sub1)
#     print(found)
#     assert isinstance(found, int)
#     assert found == 6
#     assert s1.find(sub1) == 6
#     assert s1.find(sub1, 1) == 6
#     assert s1.find(sub1, 7) == 18
#     assert s1.find("CCCCCCC") == -1

#     for substr in ["ACTG", "TCGCACA"]:
#         assert s1.find(SeqLike(substr, "dna")) == s1.find(substr)
#     for start, stop in [(0, 5), (8, 11), (5, 15)]:
#         assert s1.find(s1[start:stop]) == s1.find(str(s1)[start:stop])


# def test_apply():
#     # TODO: Add docstring
#     # TODO: Parametrize with hypothesis.
#     codon_map = codon_table_to_codon_map(ecoli_codon_table, deterministic=True)

#     s = SeqLike("GEGDATYGKLTLKFICTT", seq_type="aa", codon_map=codon_map)
#     s2 = SeqLike("GEGDATYGKLTLKFICTT", seq_type="aa")

#     # test an apply that returns a non-SeqLike
#     assert s.apply(count_residue) == 4

#     # test an apply that returns a SeqLike
#     s_split = s.apply(split_seq, split_location=5)
#     assert len(s_split) == 5
#     assert isinstance(s_split, SeqLike)

#     assert s.apply(split_seq, split_location=5).back_translate().codon_map is not None
#     assert s_split.codon_map is not None


def test__deepcopy__():
    for seqstr, seq_type in [("TCGCACACTGCA", "nt"), ("GEGDATYGKLTLKFICTT", "aa")]:
        s = SeqLike(seqstr, seq_type=seq_type, description="test")
        assert deepcopy(s).to_str() == seqstr
        assert deepcopy(s).id == s.id
        assert deepcopy(s).description == s.description == "test"
        # deepcopy returns a new object, so it should not match the original
        assert deepcopy(s) != s


# def split_seq(seq, split_location):
#     return seq[:split_location]


# def count_residue(seq, residue="T"):
#     return seq.to_str().count(residue)
