"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Topic = void 0;
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
const topic_base_1 = require("./topic-base");
/**
 * A new SNS topic.
 *
 * @stability stable
 */
class Topic extends topic_base_1.TopicBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.topicName,
        });
        /**
         * Controls automatic creation of policy objects.
         *
         * Set by subclasses.
         *
         * @stability stable
         */
        this.autoCreatePolicy = true;
        if (props.contentBasedDeduplication && !props.fifo) {
            throw new Error('Content based deduplication can only be enabled for FIFO SNS topics.');
        }
        const resource = new sns_generated_1.CfnTopic(this, 'Resource', {
            displayName: props.displayName,
            topicName: this.physicalName,
            kmsMasterKeyId: props.masterKey && props.masterKey.keyArn,
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoTopic: props.fifo,
        });
        this.topicArn = this.getResourceArnAttribute(resource.ref, {
            service: 'sns',
            resource: this.physicalName,
        });
        this.topicName = this.getResourceNameAttribute(resource.attrTopicName);
    }
    /**
     * Import an existing SNS topic provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param topicArn topic ARN (i.e. arn:aws:sns:us-east-2:444455556666:MyTopic).
     * @stability stable
     */
    static fromTopicArn(scope, id, topicArn) {
        class Import extends topic_base_1.TopicBase {
            constructor() {
                super(...arguments);
                this.topicArn = topicArn;
                this.topicName = core_1.Stack.of(scope).parseArn(topicArn).resource;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.Topic = Topic;
//# sourceMappingURL=data:application/json;base64,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