from http import HTTPStatus
from typing import Any, Optional, Union, cast

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.vulnerability import Vulnerability
from ...types import Response


def _get_kwargs(
    source: str,
    vuln: str,
) -> dict[str, Any]:
    _kwargs: dict[str, Any] = {
        "method": "get",
        "url": "/v1/vulnerability/source/{source}/vuln/{vuln}".format(
            source=source,
            vuln=vuln,
        ),
    }

    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, Vulnerability]]:
    if response.status_code == 200:
        response_200 = Vulnerability.from_dict(response.json())

        return response_200
    if response.status_code == 401:
        response_401 = cast(Any, None)
        return response_401
    if response.status_code == 404:
        response_404 = cast(Any, None)
        return response_404
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, Vulnerability]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    source: str,
    vuln: str,
    *,
    client: AuthenticatedClient,
) -> Response[Union[Any, Vulnerability]]:
    """Returns a specific vulnerability

     <p>Requires permission <strong>VIEW_PORTFOLIO</strong></p>

    Args:
        source (str):
        vuln (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, Vulnerability]]
    """

    kwargs = _get_kwargs(
        source=source,
        vuln=vuln,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    source: str,
    vuln: str,
    *,
    client: AuthenticatedClient,
) -> Optional[Union[Any, Vulnerability]]:
    """Returns a specific vulnerability

     <p>Requires permission <strong>VIEW_PORTFOLIO</strong></p>

    Args:
        source (str):
        vuln (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, Vulnerability]
    """

    return sync_detailed(
        source=source,
        vuln=vuln,
        client=client,
    ).parsed


async def asyncio_detailed(
    source: str,
    vuln: str,
    *,
    client: AuthenticatedClient,
) -> Response[Union[Any, Vulnerability]]:
    """Returns a specific vulnerability

     <p>Requires permission <strong>VIEW_PORTFOLIO</strong></p>

    Args:
        source (str):
        vuln (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, Vulnerability]]
    """

    kwargs = _get_kwargs(
        source=source,
        vuln=vuln,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    source: str,
    vuln: str,
    *,
    client: AuthenticatedClient,
) -> Optional[Union[Any, Vulnerability]]:
    """Returns a specific vulnerability

     <p>Requires permission <strong>VIEW_PORTFOLIO</strong></p>

    Args:
        source (str):
        vuln (str):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, Vulnerability]
    """

    return (
        await asyncio_detailed(
            source=source,
            vuln=vuln,
            client=client,
        )
    ).parsed
