"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LayerVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    /**
     * Add permission for this layer version to specific entities.
     *
     * Usage within
     * the same account where the layer is defined is always allowed and does not
     * require calling this method. Note that the principal that creates the
     * Lambda function using the layer (for example, a CloudFormation changeset
     * execution role) also needs to have the ``lambda:GetLayerVersion``
     * permission on the layer version.
     *
     * @stability stable
     */
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 *
 * @stability stable
 */
class LayerVersion extends LayerVersionBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * Imports a layer version by ARN.
     *
     * Assumes it is compatible with all Lambda runtimes.
     *
     * @stability stable
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     * @stability stable
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
_a = JSII_RTTI_SYMBOL_1;
LayerVersion[_a] = { fqn: "@aws-cdk/aws-lambda.LayerVersion", version: "1.108.0" };
//# sourceMappingURL=data:application/json;base64,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