"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCustomResource = void 0;
const hash_1 = require("./hash");
const reply_1 = require("./reply");
/* istanbul ignore file */
class SimpleCustomResource {
    constructor() {
        var _a;
        // Optionally suppress debugging statements.
        this.debugMode = ((_a = process.env.DEBUG) !== null && _a !== void 0 ? _a : 'false') === 'true';
        if (!this.debugMode) {
            console.debug = () => { };
        }
    }
    /**
     * Handler/engine for the CustomResource state machine. Users of this class should
     * instantiate the class, and then immediately call this function.
     * @param event The event passed to the lambda handler.
     * @param context The lambda context passed to the lambda handler.
     */
    async handler(event, context) {
        var _a;
        let status = reply_1.CfnResponseStatus.SUCCESS;
        let failReason;
        let cfnData;
        console.log(`Handling event: ${JSON.stringify(event)}`);
        const resourceProperties = (_a = event.ResourceProperties) !== null && _a !== void 0 ? _a : {};
        const physicalId = hash_1.calculateSha256Hash(resourceProperties);
        try {
            const timeout = (prom, time, exception) => {
                let timer;
                return Promise.race([
                    prom,
                    new Promise((_r, rej) => timer = setTimeout(rej, time, exception)),
                ]).finally(() => clearTimeout(timer));
            };
            // We want to always notify CloudFormation about the success/failure of the Lambda at all times.
            // If function execution time is longer than Lambda's timeout, then the function is just stopped
            // and CloudFormation is not notified at all. This would result in a hang-up during deployment.
            // Thus, we want to stop the execution by ourselves before the Lambda timeout and reserve some time
            // for notifying a CloudFormation about a failed deployment because of the timeout.
            // 3 seconds should be enough to resolve the request that signals success/failure of the custom resource,
            // but if Lambda timeout is too small, we would reserve 20% of the remaining time and still try to notify the CF.
            // Check the logs during the development to see if you allocated enough time for your Lambda.
            const defaultReserveTimeMs = 3000;
            const remainingTimeMs = context.getRemainingTimeInMillis();
            let reserveTimeMs = Math.min(0.2 * remainingTimeMs, defaultReserveTimeMs);
            if (reserveTimeMs < defaultReserveTimeMs) {
                console.debug(`The remaining Lambda execution time of ${reserveTimeMs} ` +
                    `ms might not be sufficient to send a CloudFormation response. At least ${defaultReserveTimeMs} ms is required. ` +
                    'Please increase the Lambda timeout.');
            }
            cfnData = await timeout(this.handleEvent(event, context, resourceProperties, physicalId), remainingTimeMs - reserveTimeMs, new Error('Timeout error'));
        }
        catch (e) {
            // We want to always catch the exception for a CfnCustomResource CloudFormation
            // must be notified about the success/failure of the lambda at all times;
            // failure to notify results in a stuck stack that takes at least an hour to
            // timeout.
            status = reply_1.CfnResponseStatus.FAILED;
            failReason = `${e.message}\n${e.stack}`;
        }
        finally {
            // Always send a response to CloudFormation, signal success or
            // failure based on whether or not we had an exception.
            await reply_1.sendCfnResponse({
                event,
                context,
                status,
                reason: failReason,
                physicalId,
                data: cfnData,
            });
        }
        const response = `${status}` + (failReason !== null && failReason !== void 0 ? failReason : '');
        console.log(`Result: ${response}`);
        return response;
    }
    async handleEvent(event, context, props, physicalId) {
        var _a;
        const requestType = event.RequestType;
        let cfnData;
        if (requestType === 'Create') {
            if (!this.validateInput(props)) {
                throw Error(`Input did not pass validation check. Check log group "${context.logGroupName}" ` +
                    `for log stream ${context.logStreamName} for additional information.`);
            }
            cfnData = await this.doCreate(physicalId, props);
            console.debug(`Create data: ${JSON.stringify(cfnData)}`);
        }
        else if (requestType === 'Update') {
            if (!this.validateInput(props)) {
                throw Error('Input did not pass validation check');
            }
            const oldResourceProperties = (_a = event.OldResourceProperties) !== null && _a !== void 0 ? _a : {};
            const oldPhysicalId = hash_1.calculateSha256Hash(oldResourceProperties);
            if (oldPhysicalId !== physicalId) {
                console.log('Doing Create -- ResourceProperties differ.');
                cfnData = await this.doCreate(physicalId, props);
                console.debug(`Update data: ${JSON.stringify(cfnData)}`);
            }
        }
        else {
            await this.doDelete(physicalId, props);
        }
        return cfnData;
    }
}
exports.SimpleCustomResource = SimpleCustomResource;
//# sourceMappingURL=data:application/json;base64,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