"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const AWS = require("aws-sdk");
const deadline_1 = require("../../../../deadline");
const handler_1 = require("../handler");
jest.mock('../../lib/secrets-manager/read-certificate');
const secretArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
// @ts-ignore
async function successRequestMock(request, returnValue) {
    return returnValue;
}
describe('SEPConfiguratorResource', () => {
    const validConnection = {
        hostname: 'internal-hostname.com',
        protocol: 'HTTPS',
        port: '4433',
        caCertificateArn: secretArn,
    };
    const validLaunchSpecification = {
        IamInstanceProfile: {
            Arn: 'iamInstanceProfileArn',
        },
        ImageId: 'any-ami',
        InstanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL).toString(),
        SecurityGroups: [{
                GroupId: 'sg-id',
            }],
        TagSpecifications: [{
                ResourceType: deadline_1.SpotFleetResourceType.INSTANCE,
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
        UserData: 'userdata',
        KeyName: 'keyname',
        SubnetId: 'subnet-id',
        BlockDeviceMappings: [{
                DeviceName: 'device',
                NoDevice: '',
                VirtualName: 'virtualname',
                Ebs: {
                    DeleteOnTermination: true,
                    Encrypted: true,
                    Iops: 10,
                    SnapshotId: 'snapshot-id',
                    VolumeSize: 10,
                    VolumeType: 'volume-type',
                },
            }],
    };
    const validSpotFleetRequestProps = {
        AllocationStrategy: deadline_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED,
        IamFleetRole: 'roleArn',
        LaunchSpecifications: [validLaunchSpecification],
        ReplaceUnhealthyInstances: true,
        TargetCapacity: 1,
        TerminateInstancesWithExpiration: true,
        Type: deadline_1.SpotFleetRequestType.MAINTAIN,
        TagSpecifications: [{
                ResourceType: deadline_1.SpotFleetResourceType.SPOT_FLEET_REQUEST,
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
        ValidUntil: core_1.Expiration.atDate(new Date(2022, 11, 17)).date.toISOString(),
    };
    const validConvertedLaunchSpecifications = {
        BlockDeviceMappings: [{
                DeviceName: 'device',
                Ebs: {
                    DeleteOnTermination: true,
                    Encrypted: true,
                    Iops: 10,
                    SnapshotId: 'snapshot-id',
                    VolumeSize: 10,
                    VolumeType: 'volume-type',
                },
                NoDevice: '',
                VirtualName: 'virtualname',
            }],
        IamInstanceProfile: {
            Arn: 'iamInstanceProfileArn',
        },
        ImageId: 'any-ami',
        KeyName: 'keyname',
        SecurityGroups: [{
                GroupId: 'sg-id',
            }],
        SubnetId: 'subnet-id',
        TagSpecifications: [{
                ResourceType: 'instance',
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
        UserData: 'userdata',
        InstanceType: 't2.small',
    };
    const validConvertedSpotFleetRequestProps = {
        AllocationStrategy: 'capacityOptimized',
        IamFleetRole: 'roleArn',
        LaunchSpecifications: [validConvertedLaunchSpecifications],
        ReplaceUnhealthyInstances: true,
        TargetCapacity: 1,
        TerminateInstancesWithExpiration: true,
        Type: 'maintain',
        ValidUntil: core_1.Expiration.atDate(new Date(2022, 11, 17)).date.toISOString(),
        TagSpecifications: [{
                ResourceType: 'spot-fleet-request',
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
    };
    const validSpotFleetRequestConfig = {
        group_name1: validSpotFleetRequestProps,
    };
    const validConvertedSpotFleetRequestConfig = {
        group_name1: validConvertedSpotFleetRequestProps,
    };
    const validSpotEventPluginConfig = {
        AWSInstanceStatus: deadline_1.SpotEventPluginDisplayInstanceStatus.DISABLED,
        DeleteInterruptedSlaves: true,
        DeleteTerminatedSlaves: true,
        IdleShutdown: 20,
        Logging: deadline_1.SpotEventPluginLoggingLevel.STANDARD,
        PreJobTaskMode: deadline_1.SpotEventPluginPreJobTaskMode.CONSERVATIVE,
        Region: 'us-west-2',
        ResourceTracker: true,
        StaggerInstances: 50,
        State: deadline_1.SpotEventPluginState.GLOBAL_ENABLED,
        StrictHardCap: true,
    };
    const validConvertedPluginConfig = {
        AWSInstanceStatus: 'Disabled',
        DeleteInterruptedSlaves: true,
        DeleteTerminatedSlaves: true,
        IdleShutdown: 20,
        Logging: 'Standard',
        PreJobTaskMode: 'Conservative',
        Region: 'us-west-2',
        ResourceTracker: true,
        StaggerInstances: 50,
        State: 'Global Enabled',
        StrictHardCap: true,
    };
    // Valid configurations
    const noPluginConfigs = {
        connection: validConnection,
        spotFleetRequestConfigurations: validSpotFleetRequestConfig,
    };
    const noFleetRequestConfigs = {
        spotPluginConfigurations: validSpotEventPluginConfig,
        connection: validConnection,
    };
    const allConfigs = {
        spotPluginConfigurations: validSpotEventPluginConfig,
        connection: validConnection,
        spotFleetRequestConfigurations: validSpotFleetRequestConfig,
    };
    const noConfigs = {
        connection: validConnection,
    };
    describe('doCreate', () => {
        let handler;
        let mockSpotEventPluginClient;
        beforeEach(() => {
            mockSpotEventPluginClient = {
                saveServerData: jest.fn(),
                configureSpotEventPlugin: jest.fn(),
            };
            handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            async function returnSpotEventPluginClient(_v1) {
                return mockSpotEventPluginClient;
            }
            // eslint-disable-next-line dot-notation
            handler['spotEventPluginClient'] = jest.fn((a) => returnSpotEventPluginClient(a));
        });
        afterEach(() => {
            jest.clearAllMocks();
        });
        test('with no configs', async () => {
            // GIVEN
            async function returnTrue(_v1) {
                return true;
            }
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const result = await handler.doCreate('physicalId', noConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(0);
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(0);
        });
        test('save spot fleet request configs', async () => {
            // GIVEN
            async function returnTrue(_v1) {
                return true;
            }
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', noPluginConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            const calledWithString = mockSaveServerData.mock.calls[0][0];
            const calledWithObject = JSON.parse(calledWithString);
            expect(calledWithObject).toEqual(validConvertedSpotFleetRequestConfig);
        });
        test('save spot fleet request configs without BlockDeviceMappings', async () => {
            // GIVEN
            async function returnTrue(_v1) {
                return true;
            }
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const noEbs = {
                ...noPluginConfigs,
                spotFleetRequestConfigurations: {
                    ...validSpotFleetRequestConfig,
                    group_name1: {
                        ...validSpotFleetRequestProps,
                        LaunchSpecifications: [
                            {
                                ...validLaunchSpecification,
                                BlockDeviceMappings: undefined,
                            },
                        ],
                    },
                },
            };
            const convertedNoEbs = {
                ...validConvertedSpotFleetRequestConfig,
                group_name1: {
                    ...validConvertedSpotFleetRequestProps,
                    LaunchSpecifications: [
                        {
                            ...validConvertedLaunchSpecifications,
                            BlockDeviceMappings: undefined,
                        },
                    ],
                },
            };
            // WHEN
            await handler.doCreate('physicalId', noEbs);
            const calledWithString = mockSaveServerData.mock.calls[0][0];
            const calledWithObject = JSON.parse(calledWithString);
            // THEN
            expect(calledWithObject).toEqual(convertedNoEbs);
        });
        test('save spot fleet request configs without Ebs', async () => {
            // GIVEN
            async function returnTrue(_v1) {
                return true;
            }
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const blockDevicesNoEbs = [{
                    DeviceName: 'device',
                }];
            const noEbs = {
                ...noPluginConfigs,
                spotFleetRequestConfigurations: {
                    ...validSpotFleetRequestConfig,
                    group_name1: {
                        ...validSpotFleetRequestProps,
                        LaunchSpecifications: [
                            {
                                ...validLaunchSpecification,
                                BlockDeviceMappings: blockDevicesNoEbs,
                            },
                        ],
                    },
                },
            };
            const convertedNoEbs = {
                ...validConvertedSpotFleetRequestConfig,
                group_name1: {
                    ...validConvertedSpotFleetRequestProps,
                    LaunchSpecifications: [
                        {
                            ...validConvertedLaunchSpecifications,
                            BlockDeviceMappings: blockDevicesNoEbs,
                        },
                    ],
                },
            };
            // WHEN
            await handler.doCreate('physicalId', noEbs);
            const calledWithString = mockSaveServerData.mock.calls[0][0];
            const calledWithObject = JSON.parse(calledWithString);
            // THEN
            expect(calledWithObject).toEqual(convertedNoEbs);
        });
        test('save spot event plugin configs', async () => {
            // GIVEN
            async function returnTrue(_v1) {
                return true;
            }
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validConvertedPluginConfig)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            const result = await handler.doCreate('physicalId', noFleetRequestConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('save both configs', async () => {
            // GIVEN
            async function returnTrue(_v1) {
                return true;
            }
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validConvertedPluginConfig)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            const result = await handler.doCreate('physicalId', allConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            expect(mockSaveServerData.mock.calls[0][0]).toEqual(JSON.stringify(validConvertedSpotFleetRequestConfig));
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('throw when cannot save spot fleet request configs', async () => {
            // GIVEN
            async function returnFalse(_v1) {
                return false;
            }
            const mockSaveServerData = jest.fn((a) => returnFalse(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const promise = handler.doCreate('physicalId', noPluginConfigs);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save spot fleet request with configuration/);
        });
        test('throw when cannot save spot event plugin configs', async () => {
            // GIVEN
            async function returnFalse(_v1) {
                return false;
            }
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnFalse(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const promise = handler.doCreate('physicalId', noFleetRequestConfigs);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save Spot Event Plugin Configurations/);
        });
    });
    test('doDelete does not do anything', async () => {
        // GIVEN
        const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
        // WHEN
        const promise = await handler.doDelete('physicalId', noConfigs);
        // THEN
        await expect(promise).toBeUndefined();
    });
    describe('.validateInput()', () => {
        describe('should return true', () => {
            test.each([
                allConfigs,
                noPluginConfigs,
                noFleetRequestConfigs,
                noConfigs,
            ])('with valid input', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
        });
        // Invalid connection
        const noProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            port: '4433',
        };
        const noHostnameConnection = {
            protocol: 'HTTPS',
            port: '4433',
        };
        const noPortConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            caCertificateArn: secretArn,
        };
        const invalidHostnameConnection = {
            hostname: 10,
            protocol: 'HTTPS',
            port: '4433',
        };
        const invalidProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'TCP',
            port: '4433',
        };
        const invalidProtocolTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: ['HTTPS'],
            port: '4433',
        };
        const invalidPortTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: 4433,
        };
        const invalidPortRange1Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '-1',
        };
        const invalidPortRange2Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '65536',
        };
        const invalidPortRange3Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: Number.NaN.toString(),
        };
        const invalidCaCertConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '4433',
            caCertificateArn: 'notArn',
        };
        describe('should return false if', () => {
            test.each([
                noProtocolConnection,
                noHostnameConnection,
                noPortConnection,
                invalidCaCertConnection,
                invalidHostnameConnection,
                invalidProtocolConnection,
                invalidProtocolTypeConnection,
                invalidPortTypeConnection,
                invalidPortRange1Connection,
                invalidPortRange2Connection,
                invalidPortRange3Connection,
                undefined,
                [],
            ])('invalid connection', (invalidConnection) => {
                // GIVEN
                const input = {
                    spotPluginConfigurations: validSpotEventPluginConfig,
                    connection: invalidConnection,
                    spotFleetRequestConfigurations: validSpotFleetRequestConfig,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
            test.each([
                undefined,
                [],
                '',
            ])('{input=%s}', (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.isSecretArnOrUndefined()', () => {
        describe('should return true if', () => {
            test.each([
                secretArn,
                undefined,
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeTruthy();
            });
        });
        describe('should return false if', () => {
            test.each([
                'any string',
                10,
                [],
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.spotEventPluginClient()', () => {
        test('creates a valid object with http', async () => {
            // GIVEN
            const validHTTPConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPConnection);
            expect(result).toBeDefined();
        });
        test('creates a valid object with https', async () => {
            // GIVEN
            const validHTTPSConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
                caCertificateArn: secretArn,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPSConnection);
            expect(result).toBeDefined();
        });
    });
    describe('.toKeyValueArray()', () => {
        test('converts to array of key value pairs', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: 'Disabled',
            };
            const expectedResult = {
                Key: 'AWSInstanceStatus',
                Value: 'Disabled',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const returnValue = handler['toKeyValueArray'](pluginConfig);
            // THEN
            expect(returnValue).toContainEqual(expectedResult);
        });
        test('throws with undefined values', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: undefined,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            function toKeyValueArray() {
                // eslint-disable-next-line dot-notation
                handler['toKeyValueArray'](pluginConfig);
            }
            // THEN
            expect(toKeyValueArray).toThrowError();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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