"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const conversion_1 = require("../conversion");
describe('convertSpotEventPluginSettings()', () => {
    test('does not convert properties with correct types', () => {
        // GIVEN
        const defaultPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: false,
            DeleteTerminatedSlaves: false,
            IdleShutdown: 10,
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: true,
            StaggerInstances: 50,
            State: 'Disabled',
            StrictHardCap: false,
        };
        const defaultConvertedPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: false,
            DeleteTerminatedSlaves: false,
            IdleShutdown: 10,
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: true,
            StaggerInstances: 50,
            State: 'Disabled',
            StrictHardCap: false,
        };
        // WHEN
        const returnValue = conversion_1.convertSpotEventPluginSettings(defaultPluginConfig);
        // THEN
        expect(returnValue).toEqual(defaultConvertedPluginConfig);
    });
    test('converts properties of type string', () => {
        // GIVEN
        const defaultPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: 'false',
            DeleteTerminatedSlaves: 'false',
            IdleShutdown: '10',
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: 'true',
            StaggerInstances: '50',
            State: 'Disabled',
            StrictHardCap: 'false',
        };
        const defaultConvertedPluginConfig = {
            AWSInstanceStatus: 'Disabled',
            DeleteInterruptedSlaves: false,
            DeleteTerminatedSlaves: false,
            IdleShutdown: 10,
            Logging: 'Standard',
            PreJobTaskMode: 'Conservative',
            Region: 'eu-west-1',
            ResourceTracker: true,
            StaggerInstances: 50,
            State: 'Disabled',
            StrictHardCap: false,
        };
        // WHEN
        // Need this trick so TS allows to pass config with string properties.
        const config = defaultPluginConfig;
        const returnValue = conversion_1.convertSpotEventPluginSettings(config);
        // THEN
        expect(returnValue).toEqual(defaultConvertedPluginConfig);
    });
});
describe('convertToInt()', () => {
    test.each([
        ['10', 10],
        [10, 10],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.convertToInt(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10.6,
        [],
        {},
        'string',
        undefined,
    ])('throws an error with %p', input => {
        // WHEN
        const propertyName = 'propertyName';
        function callingConvertToInt() {
            conversion_1.convertToInt(input, propertyName);
        }
        // THEN
        expect(callingConvertToInt).toThrowError(`The value of ${propertyName} should be an integer. Received: ${input}`);
    });
});
describe('convertToIntOptional()', () => {
    test.each([
        ['10', 10],
        [10, 10],
        [undefined, undefined],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.convertToIntOptional(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10.6,
        [],
        {},
        'string',
    ])('throws an error with %p', input => {
        // WHEN
        const propertyName = 'propertyName';
        function callingConvertToIntOptional() {
            conversion_1.convertToIntOptional(input, propertyName);
        }
        // THEN
        expect(callingConvertToIntOptional).toThrowError(`The value of ${propertyName} should be an integer. Received: ${input}`);
    });
});
describe('convertToBoolean()', () => {
    test.each([
        [true, true],
        ['true', true],
        [false, false],
        ['false', false],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.convertToBoolean(input, 'property');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10.6,
        [],
        {},
        'string',
        undefined,
    ])('throws an error with %p', input => {
        // WHEN
        const propertyName = 'propertyName';
        function callingConvertToBoolean() {
            conversion_1.convertToBoolean(input, propertyName);
        }
        // THEN
        expect(callingConvertToBoolean).toThrowError(`The value of ${propertyName} should be a boolean. Received: ${input}`);
    });
});
describe('convertToBooleanOptional()', () => {
    test.each([
        [true, true],
        ['true', true],
        [false, false],
        ['false', false],
        [undefined, undefined],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.convertToBooleanOptional(input, 'property');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10.6,
        [],
        {},
        'string',
    ])('throws an error with %p', input => {
        // WHEN
        const propertyName = 'propertyName';
        function callingConvertToBooleanOptional() {
            conversion_1.convertToBooleanOptional(input, propertyName);
        }
        // THEN
        expect(callingConvertToBooleanOptional).toThrowError(`The value of ${propertyName} should be a boolean. Received: ${input}`);
    });
});
describe('validateString()', () => {
    test.each([
        ['string', 'string'],
        ['10', '10'],
        ['true', 'true'],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.validateString(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10,
        [],
        {},
        undefined,
    ])('throws an error with %p', input => {
        // WHEN
        const propertyName = 'propertyName';
        function callingValidateString() {
            conversion_1.validateString(input, propertyName);
        }
        // THEN
        expect(callingValidateString).toThrowError(`The value of ${propertyName} should be a string. Received: ${input} of type ${typeof (input)}`);
    });
});
describe('validateStringOptional()', () => {
    test.each([
        ['string', 'string'],
        ['10', '10'],
        ['true', 'true'],
        [undefined, undefined],
    ])('correctly converts %p to %p', (input, expected) => {
        // WHEN
        const returnValue = conversion_1.validateStringOptional(input, 'propertyName');
        // THEN
        expect(returnValue).toBe(expected);
    });
    test.each([
        10,
        [],
        {},
    ])('throws an error with %p', input => {
        // WHEN
        const propertyName = 'propertyName';
        function callingValidateStringOptional() {
            conversion_1.validateStringOptional(input, propertyName);
        }
        // THEN
        expect(callingValidateStringOptional).toThrowError(`The value of ${propertyName} should be a string. Received: ${input} of type ${typeof (input)}`);
    });
});
describe('validateArray', () => {
    test.each([
        undefined,
        {},
        [],
    ])('throws with invalid input %p', (invalidInput) => {
        // WHEN
        const propertyName = 'propertyName';
        function callingValidateArray() {
            conversion_1.validateArray(invalidInput, propertyName);
        }
        // THEN
        expect(callingValidateArray).toThrowError(`${propertyName} should be an array with at least one element.`);
    });
    test('passes with not empty array', () => {
        // GIVEN
        const nonEmptyArray = ['value'];
        // WHEN
        function callingValidateArray() {
            conversion_1.validateArray(nonEmptyArray, 'propertyName');
        }
        // THEN
        expect(callingValidateArray).not.toThrowError();
    });
});
describe('isValidSecurityGroup', () => {
    // Valid security groups
    const validSecurityGroup = {
        GroupId: 'groupId',
    };
    // Invalid security groups
    const groupIdNotString = {
        GroupId: 10,
    };
    const noGroupId = {};
    test.each([
        undefined,
        [],
        '',
        groupIdNotString,
        noGroupId,
    ])('returns false with invalid input %p', (invalidInput) => {
        // WHEN
        const result = conversion_1.isValidSecurityGroup(invalidInput);
        // THEN
        expect(result).toBeFalsy();
    });
    test('returns true with a valid input', () => {
        // WHEN
        const result = conversion_1.isValidSecurityGroup(validSecurityGroup);
        // THEN
        expect(result).toBeTruthy();
    });
});
describe('isValidTagSpecification', () => {
    // Valid tag specifications
    const validTagSpecification = {
        ResourceType: 'type',
        Tags: [{
                Key: 'key',
                Value: 'value',
            }],
    };
    // Invalid tag specifications
    const noResourceType = {};
    const resourceTypeNotSting = {
        ResourceType: 10,
    };
    const noTags = {
        ResourceType: 'type',
    };
    const tagsNotArray = {
        ResourceType: 'type',
        Tags: '',
    };
    const tagElementUndefined = {
        ResourceType: 'type',
        Tags: [undefined],
    };
    const tagElementWrongType = {
        ResourceType: 'type',
        Tags: [''],
    };
    const tagElementNoKey = {
        ResourceType: 'type',
        Tags: [{}],
    };
    const tagElementKeyNotString = {
        ResourceType: 'type',
        Tags: [{
                Key: 10,
            }],
    };
    const tagElementNoValue = {
        ResourceType: 'type',
        Tags: [{
                Key: 'key',
            }],
    };
    const tagElementValueNotString = {
        ResourceType: 'type',
        Tags: [{
                Key: 'key',
                Value: 10,
            }],
    };
    test.each([
        undefined,
        [],
        '',
        noResourceType,
        resourceTypeNotSting,
        noTags,
        tagsNotArray,
        tagElementUndefined,
        tagElementWrongType,
        tagElementNoKey,
        tagElementKeyNotString,
        tagElementNoValue,
        tagElementValueNotString,
    ])('returns false with invalid input %p', (invalidInput) => {
        // WHEN
        const result = conversion_1.isValidTagSpecification(invalidInput);
        // THEN
        expect(result).toBeFalsy();
    });
    test('returns true with a valid input', () => {
        // WHEN
        const result = conversion_1.isValidTagSpecification(validTagSpecification);
        // THEN
        expect(result).toBeTruthy();
    });
});
describe('isValidDeviceMapping', () => {
    test.each([
        undefined,
        [],
        '',
    ])('returns false with invalid input %p', (invalidInput) => {
        // WHEN
        const result = conversion_1.isValidDeviceMapping(invalidInput);
        // THEN
        expect(result).toBeFalsy();
    });
    test('returns true with a valid input', () => {
        // GIVEN
        const anyObject = {};
        // WHEN
        const result = conversion_1.isValidDeviceMapping(anyObject);
        // THEN
        expect(result).toBeTruthy();
    });
});
describe('isValidInstanceProfile', () => {
    // Valid instance profiles
    const validInstanceProfile = {
        Arn: 'arn',
    };
    // Invalid instance profiles
    const noArn = {};
    const arnNotString = {
        Arn: 10,
    };
    test.each([
        undefined,
        [],
        '',
        noArn,
        arnNotString,
    ])('returns false with invalid input %p', (invalidInput) => {
        // WHEN
        const result = conversion_1.isValidInstanceProfile(invalidInput);
        // THEN
        expect(result).toBeFalsy();
    });
    test('returns true with a valid input', () => {
        // WHEN
        const result = conversion_1.isValidInstanceProfile(validInstanceProfile);
        // THEN
        expect(result).toBeTruthy();
    });
});
describe('validateProperty', () => {
    test('throws with invalid input', () => {
        // WHEN
        const propertyName = 'propertyName';
        function returnFalse(_input) {
            return false;
        }
        function callingValidateProperty() {
            conversion_1.validateProperty(returnFalse, 'anyValue', propertyName);
        }
        // THEN
        expect(callingValidateProperty).toThrowError(`${propertyName} type is not valid.`);
    });
    test('passes with a valid input', () => {
        // WHEN
        function returnTrue(_input) {
            return true;
        }
        function callingValidateProperty() {
            conversion_1.validateProperty(returnTrue, 'anyValue', 'propertyName');
        }
        // THEN
        expect(callingValidateProperty).not.toThrowError();
    });
});
//# sourceMappingURL=data:application/json;base64,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