"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let versionedInstallers;
let dependencyStack;
let dockerContainer;
let images;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let vpc;
let workerFleet;
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        versionedInstallers = new lib_1.VersionQuery(dependencyStack, 'VersionQuery');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: versionedInstallers,
            }),
            version: versionedInstallers,
        });
        stack = new core_1.Stack(app, 'Stack', { env });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(stack, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        licenses = [lib_1.UsageBasedLicense.forMaya()];
    });
    test('creates an ECS cluster', () => {
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'UBL', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
                certificateSecret,
                desiredCount: 2,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // WHEN
            const logGroup = ubl.node.findChild('UBLLogGroup');
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
        test('uses the supplied security group', () => {
            const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'UblSecurityGroup', {
                vpc,
            });
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
                securityGroup,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroup.securityGroupId)),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // WHEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // WHEN
                new lib_1.UsageBasedLicensing(stack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                // WHEN
                new lib_1.UsageBasedLicensing(stack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                    desiredCount,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // WHEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // WHEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: certificateSecret.secretArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('License Forwarder is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = 'licenseForwarder';
        // WHEN
        new lib_1.UsageBasedLicensing(stack, id, {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
                vpc,
                images,
                certificateSecret,
                renderQueue,
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4151, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5555, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
            workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(workerStack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                }),
            });
            // WHEN
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                assert_1.expect(workerStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(ubl).stackName}:ExportsOutputFnGetAttUBLClusterASGInstanceSecurityGroup*`),
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
                    vpc,
                    images,
                    certificateSecret: certificateSecret,
                    licenses: [],
                    renderQueue,
                });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule for asg', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 8080,
                ToPort: 8080,
                GroupId: {
                    'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLBSecurityGroup*`),
                },
                SourceSecurityGroupId: {
                    'Fn::GetAtt': [
                        'UBLClusterASGInstanceSecurityGroupAA1A7A2D',
                        'GroupId',
                    ],
                },
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'http://',
                                        {
                                            'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLB*`),
                                        },
                                        ':8080',
                                    ],
                                ],
                            },
                        }),
                    },
                ],
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                new lib_1.UsageBasedLicensing(stack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                });
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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