"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const asset_constants_1 = require("../../deadline/test/asset-constants");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
let app;
let stack;
let vpc;
let version;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC', {
        subnetConfiguration: [
            {
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            {
                name: 'Isolated',
                subnetType: aws_ec2_1.SubnetType.ISOLATED,
            },
        ],
    });
    class MockVersion extends lib_1.Version {
        constructor() {
            super(...arguments);
            this.linuxInstallers = {
                patchVersion: 0,
                repository: {
                    objectKey: 'testInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxInstallerBucket'),
                },
            };
        }
        linuxFullVersionString() {
            return this.toString();
        }
    }
    version = new MockVersion([10, 1, 9, 2]);
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: [
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ],
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const isolatedSubnetIds = ['IsolatedSubnet1', 'IsolatedSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        isolatedSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: isolatedSubnetIds,
    }));
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // gpg get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                ],
                Effect: 'Allow',
                Resource: {
                    Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                },
            }),
        },
    }));
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Effect: 'Allow',
                Resource: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            '/*',
                        ],
                    ],
                }),
            }),
        },
    }));
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    }));
});
test('repository installer logs all required files', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/bitrock_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    }));
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct',
        }),
    ]));
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBInstance'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }));
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: warningMsg,
        }),
    ]));
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new core_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' }));
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::DocDB::DBInstance', instanceCount, {
        AutoMinorVersionUpgrade: true,
    }));
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(period),
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    }));
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(20),
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Backup retention for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
        backup: {
            retention: core_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::FileSystem'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::MountTarget'));
    expect(repo.node.tryFindChild('PadEfsStorage')).toBeDefined();
    expect(repo.node.findChild('FileSystem').node.tryFindChild('PaddingAccessPoint')).toBeDefined();
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT30M',
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/10\.1\.9\.2/);
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    }));
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const asgLogicalId = stack.getLogicalId(repo.node.defaultChild.node.defaultChild);
    // THEN
    const regionToken = escapeTokenRegex('${Token[AWS.Region.\\d+]}');
    const expectedString = `function exitTrap\\(\\)\\{\nexitCode=\\$\\?\nsleep 1m\nINSTANCE="\\$\\(curl http:\\/\\/169\\.254\\.169\\.254\\/latest\\/meta-data\\/instance-id\\)"\nASG="\\$\\(aws --region ${regionToken} ec2 describe-tags --filters "Name=resource-id,Values=\\$\\{INSTANCE\\}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags\\[0\\]\\.Value" --output text\\)"\naws --region ${regionToken} autoscaling update-auto-scaling-group --auto-scaling-group-name \\$\\{ASG\\} --min-size 0 --max-size 0 --desired-capacity 0\n\\/opt\\/aws\\/bin\\/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalId} --region ${regionToken} -e \\$exitCode \\|\\| echo 'Failed to send Cloudformation Signal'\n\\}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(new RegExp(expectedString));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: 'autoscaling:UpdateAutoScalingGroup',
                Condition: {
                    StringEquals: {
                        'autoscaling:ResourceTag/resourceLogicalId': core_1.Names.uniqueId(repo),
                    },
                },
                Effect: 'Allow',
                Resource: '*',
            }, {
                Action: 'ec2:DescribeTags',
                Effect: 'Allow',
                Resource: '*',
            }),
        },
    }));
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository false rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    const regex = new RegExp(escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \\"/mnt/repository/DeadlineRepository\\"'));
    expect(userData).toMatch(regex);
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 2, {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Effect: 'Allow',
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Resource: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                            ],
                        ],
                    },
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                                '/*',
                            ],
                        ],
                    },
                ],
            }),
        },
    }));
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
describe('Security Groups', () => {
    let repositorySecurityGroup;
    beforeEach(() => {
        repositorySecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'AdditionalSecurityGroup', { vpc });
    });
    describe('DocDB', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    database: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
                VpcSecurityGroupIds: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('EFS', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    fileSystem: repositorySecurityGroup,
                },
            });
            // THEN
            // The EFS construct adds the security group to each mount target, and one mount target is generated per subnet.
            const numMountTargets = vpc.selectSubnets().subnets.length;
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::EFS::MountTarget', numMountTargets, {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('Installer', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    installer: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
});
test('validates VersionQuery is not in a different stack', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    version = new lib_1.VersionQuery(stack, 'Version');
    new lib_1.Repository(newStack, 'Repository', {
        vpc,
        version,
    });
    // WHEN
    function synth() {
        assert_1.SynthUtils.synthesize(newStack);
    }
    // THEN
    expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
});
test('creates an EFS AccessPoint when no filesystem is supplied', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
    });
    // THEN
    const efsResource = repo.node.findChild('FileSystem').node.defaultChild;
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::AccessPoint', {
        FileSystemId: stack.resolve(efsResource.ref),
        PosixUser: {
            Gid: '0',
            Uid: '0',
        },
        RootDirectory: {},
    }));
});
test('throws an error if supplied a MountableEfs with no Access Point', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const fs = new aws_efs_1.FileSystem(newStack, 'FileSystem', {
        vpc,
    });
    const mountableFs = new core_2.MountableEfs(newStack, {
        filesystem: fs,
    });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version,
            vpc,
            fileSystem: mountableFs,
        });
    }
    // THEN
    expect(when).toThrow('When using EFS with the Repository, you must provide an EFS Access Point');
});
test('imports repository settings', () => {
    // GIVEN
    const repositorySettings = new aws_s3_assets_1.Asset(stack, 'RepositorySettingsAsset', {
        path: __filename,
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        repositorySettings,
    });
    // THEN
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`aws s3 cp '${repositorySettings.s3ObjectUrl}'`);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVwb3NpdG9yeS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmVwb3NpdG9yeS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7O0FBRUgsNENBUXlCO0FBRXpCLGtEQUFtRDtBQUNuRCw4Q0FjMEI7QUFDMUIsOENBSTBCO0FBQzFCLDRDQUF5QztBQUN6QywwREFBK0M7QUFDL0Msd0NBT3VCO0FBRXZCLHFDQUVvQjtBQUNwQiw2REFFcUM7QUFDckMseUVBRTZDO0FBQzdDLGdDQU1nQjtBQUNoQix1REFFMkI7QUFFM0IsSUFBSSxHQUFRLENBQUM7QUFDYixJQUFJLEtBQVksQ0FBQztBQUNqQixJQUFJLEdBQVMsQ0FBQztBQUNkLElBQUksT0FBaUIsQ0FBQztBQUV0QixTQUFTLGdCQUFnQixDQUFDLENBQVM7SUFDakMsNkNBQTZDO0lBQzdDLG1FQUFtRTtJQUNuRSxtQ0FBbUM7SUFDbkMsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxNQUFNLENBQUMsQ0FBQztBQUN6QyxDQUFDO0FBRUQsVUFBVSxDQUFDLEdBQUcsRUFBRTtJQUNkLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0lBQ2hCLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDaEMsR0FBRyxHQUFHLElBQUksYUFBRyxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7UUFDMUIsbUJBQW1CLEVBQUU7WUFDbkI7Z0JBQ0UsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsVUFBVSxFQUFFLG9CQUFVLENBQUMsTUFBTTthQUM5QjtZQUNEO2dCQUNFLElBQUksRUFBRSxTQUFTO2dCQUNmLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE9BQU87YUFDL0I7WUFDRDtnQkFDRSxJQUFJLEVBQUUsVUFBVTtnQkFDaEIsVUFBVSxFQUFFLG9CQUFVLENBQUMsUUFBUTthQUNoQztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxXQUFZLFNBQVEsYUFBTztRQUFqQzs7WUFDVyxvQkFBZSxHQUFHO2dCQUN6QixZQUFZLEVBQUUsQ0FBQztnQkFDZixVQUFVLEVBQUU7b0JBQ1YsU0FBUyxFQUFFLGVBQWU7b0JBQzFCLFFBQVEsRUFBRSxJQUFJLGVBQU0sQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLENBQUM7aUJBQ3BEO2FBQ0YsQ0FBQTtRQUtILENBQUM7UUFIUSxzQkFBc0I7WUFDM0IsT0FBTyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDekIsQ0FBQztLQUNGO0lBRUQsT0FBTyxHQUFHLElBQUksV0FBVyxDQUFDLENBQUMsRUFBRSxFQUFDLENBQUMsRUFBQyxDQUFDLEVBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUN4QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsUUFBUTtJQUNSLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO1FBQzdCLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtZQUM3QixHQUFHO1lBQ0gsT0FBTztTQUNSLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztBQUNuQixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvREFBb0QsRUFBRSxHQUFHLEVBQUU7SUFDOUQsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQ0FBb0MsRUFBRTtRQUN6RSxVQUFVLEVBQUU7WUFDVixPQUFPLEVBQUUsR0FBRztZQUNaLE9BQU8sRUFBRSxHQUFHO1lBQ1osaUJBQWlCLEVBQUU7Z0JBQ2pCO29CQUNFLEdBQUcsRUFBRSxpQ0FBaUM7aUJBQ3ZDO2dCQUNEO29CQUNFLEdBQUcsRUFBRSxpQ0FBaUM7aUJBQ3ZDO2FBQ0Y7U0FDRjtRQUNELGNBQWMsRUFBRTtZQUNkLGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsQ0FBQztnQkFDUixPQUFPLEVBQUUsT0FBTzthQUNqQjtTQUNGO1FBQ0QsWUFBWSxFQUFFO1lBQ1osMEJBQTBCLEVBQUU7Z0JBQzFCLFdBQVcsRUFBRSxJQUFJO2FBQ2xCO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLG1DQUFtQyxFQUFFLElBQUk7YUFDMUM7U0FDRjtRQUNELFNBQVMsRUFBRTtZQUNULHNEQUFzRDtTQUN2RDtLQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7SUFDckMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1Q0FBdUMsRUFBRTtRQUM1RSxZQUFZLEVBQUUsVUFBVTtLQUN6QixDQUFDLENBQUMsQ0FBQztJQUVKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7UUFDckUsVUFBVSxFQUFFLEtBQUs7UUFDakIsUUFBUSxFQUFFLElBQUk7UUFDZCxNQUFNLEVBQUUsSUFBSTtRQUNaLE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRTtnQkFDWix1REFBdUQ7Z0JBQ3ZELFNBQVM7YUFDVjtTQUNGO1FBQ0QscUJBQXFCLEVBQUU7WUFDckIsWUFBWSxFQUFFO2dCQUNaLGtEQUFrRDtnQkFDbEQsU0FBUzthQUNWO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtJQUNqRCxvRkFBb0Y7SUFDcEYsbUJBQW1CO0lBRW5CLE9BQU87SUFDUCxNQUFNLGlCQUFpQixHQUFHLENBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLENBQUUsQ0FBQztJQUNuRSxNQUFNLE9BQU8sR0FBRyxhQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtRQUN0RCxpQkFBaUIsRUFBRSxDQUFDLFlBQVksRUFBRSxZQUFZLENBQUM7UUFDL0MsS0FBSyxFQUFFLE9BQU87UUFDZCxpQkFBaUI7S0FDbEIsQ0FBQyxDQUFDO0lBQ0gsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUMzQyxHQUFHLEVBQUUsT0FBTztRQUNaLE9BQU87UUFDUCxVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxRQUFRLEVBQUU7S0FDaEQsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7UUFDekUsaUJBQWlCLEVBQUUsaUJBQWlCO0tBQ3JDLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7UUFDckUsVUFBVSxFQUFFLEtBQUs7UUFDakIsUUFBUSxFQUFFLElBQUk7UUFDZCxNQUFNLEVBQUUsSUFBSTtRQUNaLE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRTtnQkFDWix1REFBdUQ7Z0JBQ3ZELFNBQVM7YUFDVjtTQUNGO1FBQ0QscUJBQXFCLEVBQUU7WUFDckIsWUFBWSxFQUFFO2dCQUNaLGtEQUFrRDtnQkFDbEQsU0FBUzthQUNWO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztJQUNKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7UUFDckUsVUFBVSxFQUFFLEtBQUs7UUFDakIsUUFBUSxFQUFFO1lBQ1IsWUFBWSxFQUFFO2dCQUNaLDZDQUE2QztnQkFDN0MsTUFBTTthQUNQO1NBQ0Y7UUFDRCxNQUFNLEVBQUU7WUFDTixZQUFZLEVBQUU7Z0JBQ1osNkNBQTZDO2dCQUM3QyxNQUFNO2FBQ1A7U0FDRjtRQUNELE9BQU8sRUFBRTtZQUNQLFlBQVksRUFBRTtnQkFDWiwwREFBMEQ7Z0JBQzFELFNBQVM7YUFDVjtTQUNGO1FBQ0QscUJBQXFCLEVBQUU7WUFDckIsWUFBWSxFQUFFO2dCQUNaLGtEQUFrRDtnQkFDbEQsU0FBUzthQUNWO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUg7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E0Qkc7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0JBQWtCLEVBQUU7UUFDdkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQUM7Z0JBQ25CLE1BQU0sRUFBRTtvQkFDTiwrQkFBK0I7b0JBQy9CLCtCQUErQjtpQkFDaEM7Z0JBQ0QsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsUUFBUSxFQUFFO29CQUNSLEdBQUcsRUFBRSw2REFBNkQ7aUJBQ25FO2FBQ0YsQ0FBQztTQUNIO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxREFBcUQsRUFBRSxHQUFHLEVBQUU7SUFDL0QsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQkFBa0IsRUFBRTtRQUN2RCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUUsa0JBQVMsQ0FDbEI7Z0JBQ0UsTUFBTSxFQUFFO29CQUNOLGVBQWU7b0JBQ2YsZUFBZTtvQkFDZixVQUFVO2lCQUNYO2dCQUNELE1BQU0sRUFBRSxPQUFPO2dCQUNmLFFBQVEsRUFBRSxrQkFBUyxDQUNqQjtvQkFDRSxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxNQUFNOzRCQUNOO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELFFBQVE7NEJBQ1I7Z0NBQ0UsR0FBRyxFQUFFLGlDQUFlLENBQUMsTUFBTTs2QkFDNUI7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsRUFDRDtvQkFDRSxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxNQUFNOzRCQUNOO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELFFBQVE7NEJBQ1I7Z0NBQ0UsR0FBRyxFQUFFLGlDQUFlLENBQUMsTUFBTTs2QkFDNUI7NEJBQ0QsSUFBSTt5QkFDTDtxQkFDRjtpQkFDRixDQUNGO2FBQ0YsQ0FDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7SUFDcEUsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1FBQ3ZELGVBQWUsRUFBRSxDQUFDO1FBQ2xCLFlBQVksRUFBRSxpQ0FBaUM7S0FDaEQsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFHLEVBQUU7SUFDeEQsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsMkdBQTJHO0lBQzNHLHNHQUFzRztJQUN0RyxpQ0FBaUM7SUFDakMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxxQkFBcUIsRUFBRTtRQUMxRCxJQUFJLEVBQUUsUUFBUTtRQUNkLEtBQUssRUFBRTtZQUNMLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLG9GQUFvRjtvQkFDcEYsRUFBRTtvQkFDRiw0SkFBNEo7b0JBQzVKLEVBQUU7b0JBQ0YsME9BQTBPO2lCQUMzTzthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlDQUF5QyxFQUFFLEdBQUcsRUFBRTtJQUNuRCxRQUFRO0lBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLFFBQVEsR0FBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQWlDLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBRWhGLE9BQU87SUFDUCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLGdCQUFnQixDQUFDLHdEQUF3RCxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ25ILENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztJQUNSLENBQUMsb0JBQWEsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDO0lBQ2pDLENBQUMsb0JBQWEsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDO0lBQ2hDLENBQUMsb0JBQWEsQ0FBQyxRQUFRLEVBQUUsVUFBVSxDQUFDO0NBQ3JDLENBQUMsQ0FBQywrQ0FBK0MsRUFBRSxDQUFDLE1BQXFCLEVBQUUsUUFBZ0IsRUFBRSxFQUFFO0lBQzlGLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsUUFBUSxFQUFFLE1BQU07U0FDakI7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtRQUM1RCxjQUFjLEVBQUUsUUFBUTtLQUN6QixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztJQUNSLENBQUMsb0JBQWEsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDO0lBQ2pDLENBQUMsb0JBQWEsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDO0lBQ2hDLENBQUMsb0JBQWEsQ0FBQyxRQUFRLEVBQUUsVUFBVSxDQUFDO0NBQ3JDLENBQUMsQ0FBQyxpREFBaUQsRUFBRSxDQUFDLE1BQXFCLEVBQUUsUUFBZ0IsRUFBRSxFQUFFO0lBQ2hHLE9BQU87SUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQzNDLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsVUFBVSxFQUFFLE1BQU07U0FDbkI7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxzQkFBc0IsRUFBRTtRQUMzRCxjQUFjLEVBQUUsUUFBUTtLQUN6QixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDRFQUE0RSxFQUFFLEdBQUcsRUFBRTtJQUN0RixRQUFRO0lBQ1IsTUFBTSxPQUFPLEdBQUcsSUFBSSxvQkFBYSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtRQUM1RCxHQUFHO0tBQ0osQ0FBQyxDQUFDO0lBQ0gsTUFBTSxNQUFNLEdBQUcsSUFBSSxxQkFBVyxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtRQUN2RCxVQUFVLEVBQUUsT0FBTztLQUNwQixDQUFDLENBQUM7SUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLG1CQUFZLENBQUMsS0FBSyxFQUFFO1FBQ3JDLFVBQVUsRUFBRSxPQUFPO1FBQ25CLFdBQVcsRUFBRSxNQUFNO0tBQ3BCLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxVQUFVLEVBQUUsTUFBTTtRQUNsQixPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsVUFBVSxFQUFFLG9CQUFhLENBQUMsT0FBTztTQUNsQztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQ2hDLE1BQU0sQ0FBQyxlQUFlLENBQUM7UUFDckIsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3RCLElBQUksRUFBRSxpQkFBaUI7WUFDdkIsSUFBSSxFQUFFLDRHQUE0RztTQUNuSCxDQUFDO0tBQ0gsQ0FBQyxDQUNILENBQUM7QUFDSixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx3RUFBd0UsRUFBRSxHQUFHLEVBQUU7SUFDbEYsUUFBUTtJQUNSLE1BQU0sVUFBVSxHQUFHLElBQUksMkJBQWUsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1FBQzdELFVBQVUsRUFBRTtZQUNWLFFBQVEsRUFBRSxRQUFRO1NBQ25CO1FBQ0QsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUMzQix1QkFBYSxDQUFDLEVBQUUsRUFDaEIsc0JBQVksQ0FBQyxLQUFLLENBQ25CO1FBQ0QsR0FBRztRQUNILFVBQVUsRUFBRTtZQUNWLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLG9CQUFVLENBQUMsT0FBTztTQUMvQjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxRQUFRLEVBQUUsd0JBQWtCLENBQUMsUUFBUSxDQUFDLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsVUFBVSxDQUFDLE1BQU8sRUFBRSxDQUFDO1FBQzFGLE9BQU87UUFDUCxhQUFhLEVBQUU7WUFDYixRQUFRLEVBQUUsb0JBQWEsQ0FBQyxPQUFPO1NBQ2hDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FDaEMsTUFBTSxDQUFDLGVBQWUsQ0FBQztRQUNyQixNQUFNLENBQUMsZ0JBQWdCLENBQUM7WUFDdEIsSUFBSSxFQUFFLGlCQUFpQjtZQUN2QixJQUFJLEVBQUUsd0dBQXdHO1NBQy9HLENBQUM7S0FDSCxDQUFDLENBQ0gsQ0FBQztBQUNKLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtJQUNoRSxNQUFNLE9BQU8sR0FBRyxJQUFJLG9CQUFhLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzVELEdBQUc7S0FDSixDQUFDLENBQUM7SUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLHFCQUFXLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBQ3ZELFVBQVUsRUFBRSxPQUFPO0tBQ3BCLENBQUMsQ0FBQztJQUNILE1BQU0sTUFBTSxHQUFHLElBQUksbUJBQVksQ0FBQyxLQUFLLEVBQUU7UUFDckMsVUFBVSxFQUFFLE9BQU87UUFDbkIsV0FBVyxFQUFFLE1BQU07S0FDcEIsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILFVBQVUsRUFBRSxNQUFNO1FBQ2xCLE9BQU87S0FDUixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQztJQUMzRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxDQUFDO0lBQzVELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUU7UUFDNUQsMkJBQTJCLEVBQUUsQ0FBRSxPQUFPLENBQUU7S0FDekMsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7SUFDN0IsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxxQ0FBcUMsRUFBRTtRQUMxRSxVQUFVLEVBQUU7WUFDVixVQUFVLEVBQUUsU0FBUztTQUN0QjtLQUNGLEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO0lBQzdCLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsd0JBQXdCLEVBQUU7UUFDN0QsdUJBQXVCLEVBQUUsSUFBSTtLQUM5QixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtEQUErRCxFQUFFLEdBQUcsRUFBRTtJQUN6RSxNQUFNLE9BQU8sR0FBRyxJQUFJLG9CQUFhLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFO1FBQzVELEdBQUc7S0FDSixDQUFDLENBQUM7SUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLHFCQUFXLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBQ3ZELFVBQVUsRUFBRSxPQUFPO0tBQ3BCLENBQUMsQ0FBQztJQUNILE1BQU0sTUFBTSxHQUFHLElBQUksbUJBQVksQ0FBQyxLQUFLLEVBQUU7UUFDckMsVUFBVSxFQUFFLE9BQU87UUFDbkIsV0FBVyxFQUFFLE1BQU07S0FDcEIsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILFVBQVUsRUFBRSxNQUFNO1FBQ2xCLE9BQU87UUFDUCxvQkFBb0IsRUFBRSxLQUFLO0tBQzVCLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxDQUFDO0lBQzNELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUU7UUFDL0QsMkJBQTJCLEVBQUUsQ0FBRSxPQUFPLENBQUU7S0FDekMsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7SUFDN0IsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyxxQ0FBcUMsRUFBRTtRQUM3RSxVQUFVLEVBQUU7WUFDVixVQUFVLEVBQUUsU0FBUztTQUN0QjtLQUNGLEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO0FBQy9CLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDRFQUE0RSxFQUFFLEdBQUcsRUFBRTtJQUN0RixRQUFRO0lBQ1IsTUFBTSxVQUFVLEdBQUcsSUFBSSwyQkFBZSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDN0QsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLFFBQVE7U0FDbkI7UUFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxFQUFFLENBQzNCLHVCQUFhLENBQUMsRUFBRSxFQUNoQixzQkFBWSxDQUFDLEtBQUssQ0FDbkI7UUFDRCxHQUFHO1FBQ0gsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLElBQUk7WUFDZCxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxPQUFPO1NBQy9CO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILE9BQU87UUFDUCxhQUFhLEVBQUU7WUFDYixVQUFVLEVBQUUsb0JBQWEsQ0FBQyxPQUFPO1NBQ2xDO1FBQ0QsUUFBUSxFQUFFLHdCQUFrQixDQUFDLFFBQVEsQ0FBQyxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLFVBQVUsQ0FBQyxNQUFPLEVBQUUsQ0FBQztRQUMxRixvQkFBb0IsRUFBRSxJQUFJO0tBQzNCLENBQUMsQ0FBQztJQUVILE1BQU0sVUFBVSxHQUFHLDJHQUEyRztRQUM1SCxtRUFBbUUsQ0FBQztJQUV0RSxPQUFPO0lBQ1AsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUNoQyxNQUFNLENBQUMsZUFBZSxDQUFDO1FBQ3JCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztZQUN0QixJQUFJLEVBQUUsaUJBQWlCO1lBQ3ZCLElBQUksRUFBRSxVQUFVO1NBQ2pCLENBQUM7S0FDSCxDQUFDLENBQ0gsQ0FBQztBQUNKLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtJQUN2QyxRQUFRO0lBQ1IsTUFBTSxlQUFlLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQ25ELE1BQU0sYUFBYSxHQUFHLElBQUksYUFBRyxDQUFDLGVBQWUsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUV6RCxNQUFNLE9BQU8sR0FBRztRQUNkLGdCQUFNLENBQUMsb0JBQW9CLENBQUMsZUFBZSxFQUFFLFNBQVMsRUFBRTtZQUN0RCxRQUFRLEVBQUUsV0FBVztZQUNyQixnQkFBZ0IsRUFBRSxZQUFZO1NBQy9CLENBQUM7UUFDRixnQkFBTSxDQUFDLG9CQUFvQixDQUFDLGVBQWUsRUFBRSxTQUFTLEVBQUU7WUFDdEQsUUFBUSxFQUFFLFdBQVc7WUFDckIsZ0JBQWdCLEVBQUUsWUFBWTtTQUMvQixDQUFDO0tBQ0gsQ0FBQztJQUNGLE1BQU0sYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztJQUV0RCxPQUFPO0lBQ1AsSUFBSSxnQkFBVSxDQUFDLGFBQWEsRUFBRSxxQkFBcUIsRUFBRTtRQUNuRCxHQUFHLEVBQUUsYUFBYTtRQUNsQixPQUFPO1FBQ1AsVUFBVSxFQUFFO1lBQ1YsT0FBTztTQUNSO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLGVBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMkJBQTJCLEVBQUU7UUFDeEUsU0FBUyxFQUFFO1lBQ1QsV0FBVztZQUNYLFdBQVc7U0FDWjtLQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ0osZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDbEcsZUFBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDcEcsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMkNBQTJDLEVBQUUsR0FBRyxFQUFFO0lBQ3JELFFBQVE7SUFDUixNQUFNLGFBQWEsR0FBRyxDQUFDLENBQUM7SUFFeEIsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87UUFDUCx1QkFBdUIsRUFBRSxhQUFhO0tBQ3ZDLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLHdCQUF3QixFQUFFLGFBQWEsRUFBRTtRQUM5RSx1QkFBdUIsRUFBRSxJQUFJO0tBQzlCLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO0lBQ3ZELFFBQVE7SUFDUixNQUFNLE1BQU0sR0FBRyxFQUFFLENBQUM7SUFFbEIsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDM0MsR0FBRztRQUNILE9BQU87UUFDUCxhQUFhLEVBQUU7WUFDYixpQkFBaUIsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztTQUN6QztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHVCQUF1QixFQUFFO1FBQzVELHFCQUFxQixFQUFFLE1BQU07S0FDOUIsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzREFBc0QsRUFBRSxHQUFHLEVBQUU7SUFDaEUsUUFBUTtJQUNSLE1BQU0sVUFBVSxHQUFHLElBQUksMkJBQWUsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1FBQzdELFVBQVUsRUFBRTtZQUNWLFFBQVEsRUFBRSxRQUFRO1NBQ25CO1FBQ0QsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUMzQix1QkFBYSxDQUFDLEVBQUUsRUFDaEIsc0JBQVksQ0FBQyxLQUFLLENBQ25CO1FBQ0QsR0FBRztRQUNILFVBQVUsRUFBRTtZQUNWLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLG9CQUFVLENBQUMsT0FBTztTQUMvQjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxRQUFRLEVBQUUsd0JBQWtCLENBQUMsUUFBUSxDQUFDLEVBQUUsUUFBUSxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsVUFBVSxDQUFDLE1BQU8sRUFBRSxDQUFDO1FBQzFGLE9BQU87UUFDUCxhQUFhLEVBQUU7WUFDYixpQkFBaUIsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztTQUNyQztLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQ2hDLE1BQU0sQ0FBQyxlQUFlLENBQUM7UUFDckIsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3RCLElBQUksRUFBRSxpQkFBaUI7WUFDdkIsSUFBSSxFQUFFLDJHQUEyRztTQUNsSCxDQUFDO0tBQ0gsQ0FBQyxDQUNILENBQUM7QUFDSixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7SUFFMUQsTUFBTSxVQUFVLEdBQUcsSUFBSSwyQkFBZSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7UUFDN0QsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLFFBQVE7U0FDbkI7UUFDRCxZQUFZLEVBQUUsc0JBQVksQ0FBQyxFQUFFLENBQzNCLHVCQUFhLENBQUMsRUFBRSxFQUNoQixzQkFBWSxDQUFDLEtBQUssQ0FDbkI7UUFDRCxHQUFHO1FBQ0gsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLElBQUk7WUFDZCxVQUFVLEVBQUUsb0JBQVUsQ0FBQyxPQUFPO1NBQy9CO1FBQ0QsTUFBTSxFQUFFO1lBQ04sU0FBUyxFQUFFLGVBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1NBQzdCO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEVBQUU7UUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0tBQzFDO0lBRUQsT0FBTztJQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILFFBQVEsRUFBRSx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxVQUFVLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDekYsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDO0lBQzFELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLENBQUM7SUFDM0QsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDOUQsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0FBQ2xHLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9GQUFvRixFQUFFLEdBQUcsRUFBRTtJQUM5RixNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCxPQUFPO1FBQ1AsNEJBQTRCLEVBQUUsS0FBSztLQUNwQyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxNQUFNLEdBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFpQyxDQUFDLFFBQVEsQ0FBQztJQUNyRSxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLHNCQUFzQixDQUFDLENBQUM7QUFDMUQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0dBQStHLEVBQUUsR0FBRyxFQUFFO0lBQ3pILE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILE9BQU87UUFDUCw0QkFBNEIsRUFBRSxRQUFRO0tBQ3ZDLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxNQUFNLE1BQU0sR0FBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQWlDLENBQUMsUUFBUSxDQUFDO0lBQ3JFLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsc0JBQXNCLENBQUMsQ0FBQztBQUMxRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7SUFDcEUsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUMzQyxHQUFHO1FBQ0gsNkJBQTZCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDbkQsT0FBTztLQUNSLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsb0NBQW9DLEVBQUU7UUFDckUsY0FBYyxFQUFFO1lBQ2QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxDQUFDO2dCQUNSLE9BQU8sRUFBRSxPQUFPO2FBQ2pCO1NBQ0Y7S0FDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9FQUFvRSxFQUFFLEdBQUcsRUFBRTtJQUM5RSxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCw2QkFBNkIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNuRCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBaUMsQ0FBQyxRQUFRLENBQUM7SUFDckUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztBQUNqRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxJQUFJLENBQUM7SUFDUixjQUFjO0lBQ2QsRUFBRTtDQUNILENBQUMsQ0FBQyxnRUFBZ0UsRUFBRSxDQUFDLFVBQWtCLEVBQUUsRUFBRTtJQUMxRixRQUFRO0lBQ1IsTUFBTSxFQUFFLEdBQUcscUJBQXFCLENBQUM7SUFFakMsT0FBTztJQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1FBQ3hCLEdBQUc7UUFDSCxPQUFPO1FBQ1AsYUFBYSxFQUFFO1lBQ2IsY0FBYyxFQUFFLFVBQVU7U0FDM0I7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1FBQ3ZELFlBQVksRUFBRSxVQUFVLEdBQUcsRUFBRTtLQUM5QixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM5QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsNkJBQTZCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDbkQsT0FBTztLQUNSLENBQUMsQ0FBQztJQUNILE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFhLENBQUMsSUFBSSxDQUFDLFlBQTBCLENBQUMsQ0FBQztJQUVqRyxPQUFPO0lBQ1AsTUFBTSxXQUFXLEdBQUcsZ0JBQWdCLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUNsRSxNQUFNLGNBQWMsR0FBRyxnTEFBZ0wsV0FBVyxzTEFBc0wsV0FBVyx5S0FBeUssS0FBSyxDQUFDLFNBQVMsZUFBZSxZQUFZLGFBQWEsV0FBVyx5RUFBeUUsQ0FBQztJQUN4c0IsTUFBTSxDQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBaUMsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztJQUMzRyxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtCQUFrQixFQUFFO1FBQ3ZELGNBQWMsRUFBRTtZQUNkLFNBQVMsRUFBRSxrQkFBUyxDQUNsQjtnQkFDRSxNQUFNLEVBQUUsb0NBQW9DO2dCQUM1QyxTQUFTLEVBQUU7b0JBQ1QsWUFBWSxFQUFFO3dCQUNaLDJDQUEyQyxFQUFFLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDO3FCQUNsRTtpQkFDRjtnQkFDRCxNQUFNLEVBQUUsT0FBTztnQkFDZixRQUFRLEVBQUUsR0FBRzthQUNkLEVBQ0Q7Z0JBQ0UsTUFBTSxFQUFFLGtCQUFrQjtnQkFDMUIsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsUUFBUSxFQUFFLEdBQUc7YUFDZCxDQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxRQUFRO0lBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxxQkFBcUIsRUFBRTtRQUN4RCxHQUFHO1FBQ0gsNkJBQTZCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDbkQsT0FBTztLQUNSLENBQUMsQ0FBQztJQUNILE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQy9DLEdBQUc7UUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztRQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztLQUNuRyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsSUFBSSxDQUFDLHVCQUF1QixDQUFDO1FBQzNCLElBQUksRUFBRSxRQUFRO1FBQ2QsVUFBVSxFQUFFLGlCQUFpQjtLQUM5QixDQUFDLENBQUM7SUFDSCxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBRTVDLE9BQU87SUFDUCx5RkFBeUY7SUFDekYsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQywyREFBMkQsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUVwSCw0Q0FBNEM7SUFDNUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO0lBRXRFLDJGQUEyRjtJQUMzRixNQUFNLEtBQUssR0FBRyxJQUFJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyw0RkFBNEYsQ0FBQyxDQUFDLENBQUM7SUFDekksTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNsQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrREFBK0QsRUFBRSxHQUFHLEVBQUU7SUFDekUsZ0dBQWdHO0lBRWhHLFFBQVE7SUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3hELEdBQUc7UUFDSCw2QkFBNkIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztRQUNuRCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxTQUFTLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7UUFDakQsR0FBRztRQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1FBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO0tBQ25HLENBQUMsQ0FBQztJQUNILE1BQU0sU0FBUyxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1FBQ2pELEdBQUc7UUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztRQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztLQUNuRyxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsSUFBSSxDQUFDLHVCQUF1QixDQUFDO1FBQzNCLElBQUksRUFBRSxTQUFTO1FBQ2YsVUFBVSxFQUFFLGlCQUFpQjtLQUM5QixDQUFDLENBQUM7SUFDSCxJQUFJLENBQUMsdUJBQXVCLENBQUM7UUFDM0IsSUFBSSxFQUFFLFNBQVM7UUFDZixVQUFVLEVBQUUsaUJBQWlCO0tBQzlCLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCw4R0FBOEc7SUFDOUcsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQywyQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLEVBQUU7UUFDNUQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQ2xCO2dCQUNFLE1BQU0sRUFBRSxPQUFPO2dCQUNmLE1BQU0sRUFBRTtvQkFDTixlQUFlO29CQUNmLGVBQWU7b0JBQ2YsVUFBVTtpQkFDWDtnQkFDRCxRQUFRLEVBQUU7b0JBQ1I7d0JBQ0UsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsTUFBTTtnQ0FDTjtvQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCO2lDQUN0QjtnQ0FDRCxRQUFRO2dDQUNSO29DQUNFLEdBQUcsRUFBRSwrQkFBYSxDQUFDLE1BQU07aUNBQzFCOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNEO3dCQUNFLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ047b0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtpQ0FDdEI7Z0NBQ0QsUUFBUTtnQ0FDUjtvQ0FDRSxHQUFHLEVBQUUsK0JBQWEsQ0FBQyxNQUFNO2lDQUMxQjtnQ0FDRCxJQUFJOzZCQUNMO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvREFBb0QsRUFBRSxHQUFHLEVBQUU7SUFDOUQsUUFBUTtJQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDeEQsR0FBRztRQUNILDZCQUE2QixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBQ25ELE9BQU87S0FDUixDQUFDLENBQUM7SUFDSCxNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtRQUMvQyxHQUFHO1FBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7UUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsYUFBYSxDQUFDLHdCQUFjLENBQUMscUNBQXFDLENBQUM7S0FDL0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLENBQUMsdUJBQXVCLENBQUM7WUFDM0IsSUFBSSxFQUFFLFFBQVE7WUFDZCxVQUFVLEVBQUUsTUFBTTtTQUNuQixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsNEVBQTRFLENBQUMsQ0FBQztBQUNoRyxDQUFDLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFO0lBQ3ZCLCtCQUFpQixDQUFDO1FBQ2hCLGFBQWEsRUFBRSxZQUFZO1FBQzNCLGVBQWUsRUFBRSxHQUFHLEVBQUU7WUFDcEIsUUFBUTtZQUNSLE1BQU0sYUFBYSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUN0RCxJQUFJLGdCQUFVLENBQUMsYUFBYSxFQUFFLFlBQVksRUFBRTtnQkFDMUMsR0FBRztnQkFDSCxPQUFPO2FBQ1IsQ0FBQyxDQUFDO1lBQ0gsT0FBTyxhQUFhLENBQUM7UUFDdkIsQ0FBQztRQUNELGtCQUFrQixFQUFFO1lBQ2xCLHlCQUF5QixFQUFFLENBQUM7WUFDNUIscUNBQXFDLEVBQUUsQ0FBQztZQUN4QywyQkFBMkIsRUFBRSxDQUFDO1lBQzlCLDZCQUE2QixFQUFFLENBQUM7WUFDaEMsdUJBQXVCLEVBQUUsQ0FBQztZQUMxQix3QkFBd0IsRUFBRSxDQUFDO1lBQzNCLGdCQUFnQixFQUFFLENBQUM7WUFDbkIsb0NBQW9DLEVBQUUsQ0FBQztZQUN2QyxxQkFBcUIsRUFBRSxDQUFDO1NBQ3pCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO0lBQy9CLElBQUksdUJBQXVDLENBQUM7SUFFNUMsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLHVCQUF1QixHQUFHLElBQUksdUJBQWEsQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQ3pGLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLE9BQU8sRUFBRSxHQUFHLEVBQUU7UUFFckIsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtZQUNoRCxPQUFPO1lBQ1AsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7Z0JBQ2xDLE9BQU87Z0JBQ1AsR0FBRztnQkFDSCxxQkFBcUIsRUFBRTtvQkFDckIsUUFBUSxFQUFFLHVCQUF1QjtpQkFDbEM7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtnQkFDNUQsbUJBQW1CLEVBQUUsa0JBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxDQUFDO2FBQ3ZGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFO1FBRW5CLElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7WUFDaEQsT0FBTztZQUNQLElBQUksZ0JBQVUsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO2dCQUNsQyxPQUFPO2dCQUNQLEdBQUc7Z0JBQ0gscUJBQXFCLEVBQUU7b0JBQ3JCLFVBQVUsRUFBRSx1QkFBdUI7aUJBQ3BDO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGdIQUFnSDtZQUNoSCxNQUFNLGVBQWUsR0FBRyxHQUFHLENBQUMsYUFBYSxFQUFFLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztZQUMzRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLHVCQUF1QixFQUFFLGVBQWUsRUFBRTtnQkFDL0UsY0FBYyxFQUFFLGtCQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyx1QkFBdUIsQ0FBQyxlQUFlLENBQUMsQ0FBQzthQUNsRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsV0FBVyxFQUFFLEdBQUcsRUFBRTtRQUV6QixJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1lBQ2hELE9BQU87WUFDUCxJQUFJLGdCQUFVLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtnQkFDbEMsT0FBTztnQkFDUCxHQUFHO2dCQUNILHFCQUFxQixFQUFFO29CQUNyQixTQUFTLEVBQUUsdUJBQXVCO2lCQUNuQzthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHVDQUF1QyxFQUFFO2dCQUM1RSxjQUFjLEVBQUUsa0JBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxDQUFDO2FBQ2xGLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFFTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9EQUFvRCxFQUFFLEdBQUcsRUFBRTtJQUM5RCxRQUFRO0lBQ1IsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQzVDLE9BQU8sR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQzdDLElBQUksZ0JBQVUsQ0FBQyxRQUFRLEVBQUUsWUFBWSxFQUFFO1FBQ3JDLEdBQUc7UUFDSCxPQUFPO0tBQ1IsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLFNBQVMsS0FBSztRQUNaLG1CQUFVLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRCxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQywyREFBMkQsQ0FBQyxDQUFDO0FBQ3JGLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJEQUEyRCxFQUFFLEdBQUcsRUFBRTtJQUNyRSxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDL0MsT0FBTztRQUNQLEdBQUc7S0FDSixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxXQUFXLEdBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFnQixDQUFDLElBQUksQ0FBQyxZQUE2QixDQUFDO0lBQ3pHLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtRQUN4RCxZQUFZLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDO1FBQzVDLFNBQVMsRUFBRTtZQUNULEdBQUcsRUFBRSxHQUFHO1lBQ1IsR0FBRyxFQUFFLEdBQUc7U0FDVDtRQUNELGFBQWEsRUFBRSxFQUFFO0tBQ2xCLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUVBQWlFLEVBQUUsR0FBRyxFQUFFO0lBQzNFLFFBQVE7SUFDUixNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDNUMsTUFBTSxFQUFFLEdBQUcsSUFBSSxvQkFBYSxDQUFDLFFBQVEsRUFBRSxZQUFZLEVBQUU7UUFDbkQsR0FBRztLQUNKLENBQUMsQ0FBQztJQUNILE1BQU0sV0FBVyxHQUFHLElBQUksbUJBQVksQ0FBQyxRQUFRLEVBQUU7UUFDN0MsVUFBVSxFQUFFLEVBQUU7S0FDZixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsU0FBUyxJQUFJO1FBQ1gsSUFBSSxnQkFBVSxDQUFDLFFBQVEsRUFBRSxNQUFNLEVBQUU7WUFDL0IsT0FBTztZQUNQLEdBQUc7WUFDSCxVQUFVLEVBQUUsV0FBVztTQUN4QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsT0FBTztJQUNQLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsMEVBQTBFLENBQUMsQ0FBQztBQUNuRyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsUUFBUTtJQUNSLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxxQkFBSyxDQUFDLEtBQUssRUFBRSx5QkFBeUIsRUFBRTtRQUNyRSxJQUFJLEVBQUUsVUFBVTtLQUNqQixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxVQUFVLEdBQUcsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7UUFDckQsR0FBRztRQUNILE9BQU87UUFDUCxrQkFBa0I7S0FDbkIsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sY0FBYyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBcUIsQ0FBQztJQUNyRixNQUFNLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxjQUFjLGtCQUFrQixDQUFDLFdBQVcsR0FBRyxDQUFDLENBQUM7QUFDdEcsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQge1xuICBhcnJheVdpdGgsXG4gIGNvdW50UmVzb3VyY2VzTGlrZSxcbiAgZXhwZWN0IGFzIGV4cGVjdENESyxcbiAgaGF2ZVJlc291cmNlLFxuICBoYXZlUmVzb3VyY2VMaWtlLFxuICBSZXNvdXJjZVBhcnQsXG4gIFN5bnRoVXRpbHMsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge0F1dG9TY2FsaW5nR3JvdXB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1hdXRvc2NhbGluZyc7XG5pbXBvcnQge0RhdGFiYXNlQ2x1c3Rlcn0gZnJvbSAnQGF3cy1jZGsvYXdzLWRvY2RiJztcbmltcG9ydCB7XG4gIEFtYXpvbkxpbnV4R2VuZXJhdGlvbixcbiAgSW5zdGFuY2UsXG4gIEluc3RhbmNlQ2xhc3MsXG4gIEluc3RhbmNlU2l6ZSxcbiAgSW5zdGFuY2VUeXBlLFxuICBJU2VjdXJpdHlHcm91cCxcbiAgSVZwYyxcbiAgTWFjaGluZUltYWdlLFxuICBTZWN1cml0eUdyb3VwLFxuICBTdWJuZXQsXG4gIFN1Ym5ldFR5cGUsXG4gIFZwYyxcbiAgV2luZG93c1ZlcnNpb24sXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHtcbiAgQWNjZXNzUG9pbnQsXG4gIENmbkZpbGVTeXN0ZW0sXG4gIEZpbGVTeXN0ZW0gYXMgRWZzRmlsZVN5c3RlbSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVmcyc7XG5pbXBvcnQgeyBCdWNrZXQgfSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgQXNzZXQgfSBmcm9tICdAYXdzLWNkay9hd3MtczMtYXNzZXRzJztcbmltcG9ydCB7XG4gIEFwcCxcbiAgQ2ZuRWxlbWVudCxcbiAgRHVyYXRpb24sXG4gIE5hbWVzLFxuICBSZW1vdmFsUG9saWN5LFxuICBTdGFjayxcbn0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmltcG9ydCB7XG4gIE1vdW50YWJsZUVmcyxcbn0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQge1xuICB0ZXN0Q29uc3RydWN0VGFncyxcbn0gZnJvbSAnLi4vLi4vY29yZS90ZXN0L3RhZy1oZWxwZXJzJztcbmltcG9ydCB7XG4gIENXQV9BU1NFVF9MSU5VWCxcbn0gZnJvbSAnLi4vLi4vZGVhZGxpbmUvdGVzdC9hc3NldC1jb25zdGFudHMnO1xuaW1wb3J0IHtcbiAgRGF0YWJhc2VDb25uZWN0aW9uLFxuICBJVmVyc2lvbixcbiAgUmVwb3NpdG9yeSxcbiAgVmVyc2lvblF1ZXJ5LFxuICBWZXJzaW9uLFxufSBmcm9tICcuLi9saWInO1xuaW1wb3J0IHtcbiAgUkVQT19EQ19BU1NFVCxcbn0gZnJvbSAnLi9hc3NldC1jb25zdGFudHMnO1xuXG5sZXQgYXBwOiBBcHA7XG5sZXQgc3RhY2s6IFN0YWNrO1xubGV0IHZwYzogSVZwYztcbmxldCB2ZXJzaW9uOiBJVmVyc2lvbjtcblxuZnVuY3Rpb24gZXNjYXBlVG9rZW5SZWdleChzOiBzdHJpbmcpOiBzdHJpbmcge1xuICAvLyBBIENESyBUb2tlbiBsb29rcyBsaWtlOiAke1Rva2VuW1RPS0VOLjEyXX1cbiAgLy8gVGhpcyBjb250YWlucyB0aGUgcmVnZXggc3BlY2lhbCBjaGFyYWN0ZXJzOiAuLCAkLCB7LCB9LCBbLCBhbmQgXVxuICAvLyBFc2NhcGUgdGhvc2UgZm9yIHVzZSBpbiBhIHJlZ2V4LlxuICByZXR1cm4gcy5yZXBsYWNlKC9bLiR7fVtcXF1dL2csICdcXFxcJCYnKTtcbn1cblxuYmVmb3JlRWFjaCgoKSA9PiB7XG4gIGFwcCA9IG5ldyBBcHAoKTtcbiAgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snKTtcbiAgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycsIHtcbiAgICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAgICB7XG4gICAgICAgIG5hbWU6ICdQdWJsaWMnLFxuICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyxcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIG5hbWU6ICdQcml2YXRlJyxcbiAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogJ0lzb2xhdGVkJyxcbiAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5JU09MQVRFRCxcbiAgICAgIH0sXG4gICAgXSxcbiAgfSk7XG5cbiAgY2xhc3MgTW9ja1ZlcnNpb24gZXh0ZW5kcyBWZXJzaW9uIGltcGxlbWVudHMgSVZlcnNpb24ge1xuICAgIHJlYWRvbmx5IGxpbnV4SW5zdGFsbGVycyA9IHtcbiAgICAgIHBhdGNoVmVyc2lvbjogMCxcbiAgICAgIHJlcG9zaXRvcnk6IHtcbiAgICAgICAgb2JqZWN0S2V5OiAndGVzdEluc3RhbGxlcicsXG4gICAgICAgIHMzQnVja2V0OiBuZXcgQnVja2V0KHN0YWNrLCAnTGludXhJbnN0YWxsZXJCdWNrZXQnKSxcbiAgICAgIH0sXG4gICAgfVxuXG4gICAgcHVibGljIGxpbnV4RnVsbFZlcnNpb25TdHJpbmcoKSB7XG4gICAgICByZXR1cm4gdGhpcy50b1N0cmluZygpO1xuICAgIH1cbiAgfVxuXG4gIHZlcnNpb24gPSBuZXcgTW9ja1ZlcnNpb24oWzEwLDEsOSwyXSk7XG59KTtcblxudGVzdCgnY2FuIGNyZWF0ZSB0d28gcmVwb3NpdG9yaWVzJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG8xJywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdCgoKSA9PiB7XG4gICAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvMicsIHtcbiAgICAgIHZwYyxcbiAgICAgIHZlcnNpb24sXG4gICAgfSk7XG4gIH0pLm5vdC50b1Rocm93KCk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBpbnN0YWxsZXIgaW5zdGFuY2UgaXMgY3JlYXRlZCBjb3JyZWN0bHknLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICBNYXhTaXplOiAnMScsXG4gICAgICBNaW5TaXplOiAnMScsXG4gICAgICBWUENab25lSWRlbnRpZmllcjogW1xuICAgICAgICB7XG4gICAgICAgICAgUmVmOiAnVlBDUHJpdmF0ZVN1Ym5ldDFTdWJuZXQ4QkNBMTBFMCcsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBSZWY6ICdWUENQcml2YXRlU3VibmV0MlN1Ym5ldENGQ0RBQTdBJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBDcmVhdGlvblBvbGljeToge1xuICAgICAgUmVzb3VyY2VTaWduYWw6IHtcbiAgICAgICAgQ291bnQ6IDEsXG4gICAgICAgIFRpbWVvdXQ6ICdQVDE1TScsXG4gICAgICB9LFxuICAgIH0sXG4gICAgVXBkYXRlUG9saWN5OiB7XG4gICAgICBBdXRvU2NhbGluZ1JlcGxhY2luZ1VwZGF0ZToge1xuICAgICAgICBXaWxsUmVwbGFjZTogdHJ1ZSxcbiAgICAgIH0sXG4gICAgICBBdXRvU2NhbGluZ1NjaGVkdWxlZEFjdGlvbjoge1xuICAgICAgICBJZ25vcmVVbm1vZGlmaWVkR3JvdXBTaXplUHJvcGVydGllczogdHJ1ZSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBEZXBlbmRzT246IFtcbiAgICAgICdyZXBvc2l0b3J5SW5zdGFsbGVyRG9jdW1lbnREYXRhYmFzZUluc3RhbmNlMTFBNkY4QzhFJyxcbiAgICBdLFxuICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6TGF1bmNoQ29uZmlndXJhdGlvbicsIHtcbiAgICBJbnN0YW5jZVR5cGU6ICd0My5sYXJnZScsXG4gIH0pKTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICBGcm9tUG9ydDogMjA0OSxcbiAgICBUb1BvcnQ6IDIwNDksXG4gICAgR3JvdXBJZDoge1xuICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICdyZXBvc2l0b3J5SW5zdGFsbGVyRmlsZVN5c3RlbUVmc1NlY3VyaXR5R3JvdXAyODlEMDQzQycsXG4gICAgICAgICdHcm91cElkJyxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAncmVwb3NpdG9yeUluc3RhbGxlckluc3RhbmNlU2VjdXJpdHlHcm91cDBCNTcwNUQzJyxcbiAgICAgICAgJ0dyb3VwSWQnLFxuICAgICAgXSxcbiAgICB9LFxuICB9KSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBpbnN0YWxsZXIgaG9ub3JzIHZwY1N1Ym5ldCcsICgpID0+IHtcbiAgLy8gTm90ZTogRGVmYXVsdCBpcyBwcml2YXRlIHN1Ym5ldHMsIHNvIGl0J3Mgc3VmZmljaWVudCB0byB0ZXN0IHNvbWV0aGluZyBvdGhlciB0aGFuXG4gIC8vIHByaXZhdGUgc3VibmV0cy5cblxuICAvLyBXSEVOXG4gIGNvbnN0IGlzb2xhdGVkU3VibmV0SWRzID0gWyAnSXNvbGF0ZWRTdWJuZXQxJywgJ0lzb2xhdGVkU3VibmV0MicgXTtcbiAgY29uc3QgYXR0clZwYyA9IFZwYy5mcm9tVnBjQXR0cmlidXRlcyhzdGFjaywgJ1Rlc3RWcGMnLCB7XG4gICAgYXZhaWxhYmlsaXR5Wm9uZXM6IFsndXMtZWFzdC0xYScsICd1cy1lYXN0LTFiJ10sXG4gICAgdnBjSWQ6ICd2cGNpZCcsXG4gICAgaXNvbGF0ZWRTdWJuZXRJZHMsXG4gIH0pO1xuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjOiBhdHRyVnBjLFxuICAgIHZlcnNpb24sXG4gICAgdnBjU3VibmV0czogeyBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLklTT0xBVEVEIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgIFZQQ1pvbmVJZGVudGlmaWVyOiBpc29sYXRlZFN1Ym5ldElkcyxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIHNlY3VyaXR5IGdyb3VwcyBjcmVhdGVkIGNvcnJlY3RseScsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgRnJvbVBvcnQ6IDIwNDksXG4gICAgVG9Qb3J0OiAyMDQ5LFxuICAgIEdyb3VwSWQ6IHtcbiAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAncmVwb3NpdG9yeUluc3RhbGxlckZpbGVTeXN0ZW1FZnNTZWN1cml0eUdyb3VwMjg5RDA0M0MnLFxuICAgICAgICAnR3JvdXBJZCcsXG4gICAgICBdLFxuICAgIH0sXG4gICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgJ3JlcG9zaXRvcnlJbnN0YWxsZXJJbnN0YW5jZVNlY3VyaXR5R3JvdXAwQjU3MDVEMycsXG4gICAgICAgICdHcm91cElkJyxcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzcycsIHtcbiAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICBGcm9tUG9ydDoge1xuICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICdyZXBvc2l0b3J5SW5zdGFsbGVyRG9jdW1lbnREYXRhYmFzZUEzNkNFN0ZFJyxcbiAgICAgICAgJ1BvcnQnLFxuICAgICAgXSxcbiAgICB9LFxuICAgIFRvUG9ydDoge1xuICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICdyZXBvc2l0b3J5SW5zdGFsbGVyRG9jdW1lbnREYXRhYmFzZUEzNkNFN0ZFJyxcbiAgICAgICAgJ1BvcnQnLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEdyb3VwSWQ6IHtcbiAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAncmVwb3NpdG9yeUluc3RhbGxlckRvY3VtZW50RGF0YWJhc2VTZWN1cml0eUdyb3VwQkVGREM1OEYnLFxuICAgICAgICAnR3JvdXBJZCcsXG4gICAgICBdLFxuICAgIH0sXG4gICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgJ3JlcG9zaXRvcnlJbnN0YWxsZXJJbnN0YW5jZVNlY3VyaXR5R3JvdXAwQjU3MDVEMycsXG4gICAgICAgICdHcm91cElkJyxcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbi8qXG5JQU0gUG9saWN5IGRvY3VtZW50IHRlc3RzLiBUaGUgcG9saWN5IGZvciB0aGUgaW5zdGFsbGVyIGluc3RhbmNlIGlzOlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7fSwgLy8gQ2xvdWRXYXRjaCBsb2cgZ3JvdXAgcHV0XG4gICAgICAgIHt9LCAvLyBjbG91ZHdhdGNoIGFnZW50IGluc3RhbGwgc2NyaXB0XG4gICAgICAgIHt9LCAvLyBjbG91ZHdhdGNoIGFnZW50IHN0cmluZyBwYXJhbWV0ZXJzXG4gICAgICAgIHt9LCAvLyBjbG91ZHdhdGNoIGFnZW50IGdldCBpbnN0YWxsZXIgcGVybWlzc2lvbnNcbiAgICAgICAge30sIC8vIGdwZyBnZXQgaW5zdGFsbGVyIHBlcm1pc3Npb25zXG4gICAgICAgIHt9LCAvLyBEb2NEQiBzZWNyZXQgZ2V0XG4gICAgICAgIHt9LCAvLyBmaWxlc3lzdGVtIG1vdW50IHNjcmlwdCBnZXRcbiAgICAgICAge30sIC8vIGluc3RhbGxlciBnZXRcbiAgICAgICAge30sIC8vIHJlcG9zaXRvcnkgaW5zdGFsbGF0aW9uIHNjcmlwdCBhc3NldCBnZXRcbiAgICAgICAge30sIC8vIHVwZGF0ZSBhdXRvc2NhbGluZyBwb2xpY3lcbiAgICAgICAge30sIC8vIGRlc2NyaWJlIHRhZ3NcbiAgICAgIF0sXG4gICAgfSxcbiAgfSkpO1xuXG4gIFdlIG9ubHkgZXhwbGljaXRseSB0ZXN0IGZvciB0aGUgcGVybWlzc2lvbnMgd2UgZXhwbGljaXRseSBhZGQ6XG4gICAgLSBkb2NEQiBzZWNyZXQgZ2V0XG4gICAgLSBpbnN0YWxsZXIgZ2V0XG4gICAgLSBhdXRvc2NhbGluZyBwb2xpY3kgKGhhbmRsZWQgYnk6ICdhbGlkYXRlIGluc3RhbmNlIHNlbGYtdGVybWluYXRpb24nIHRlc3QpXG4gICAgLSBkZXNjcmliZSB0YWdzIChoYW5kbGVkIGJ5OiAnYWxpZGF0ZSBpbnN0YW5jZSBzZWxmLXRlcm1pbmF0aW9uJyB0ZXN0KVxuICBUaGUgb3RoZXJzIGFyZSBzaWRlLWVmZmVjdHMgb2YgYWRkaW5nIHRoZSBjbG91ZHdhdGNoIGFnZW50LCBhbmQgbW91bnRpbmcgdGhlIGZpbGVzeXN0ZW0uXG4gIFdlIG1ha2Ugc3VyZSB0aGF0IHdlIGRvIHRob3NlIHRoaW5ncyBpbiBvdGhlciB3YXlzLCBhbmQgdHJ1c3QgdG8gdGhlaXIgY29uc3RydWN0cyB0byBiZVxuICB2ZXJpZnlpbmcgcHJvcGVyIHBlcm1pc3Npb25zLlxuICovXG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIGlhbSBwZXJtaXNzaW9uczogZGIgc2VjcmV0IGFjY2VzcycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IGFycmF5V2l0aCh7XG4gICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkRlc2NyaWJlU2VjcmV0JyxcbiAgICAgICAgXSxcbiAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgIFJlZjogJ3JlcG9zaXRvcnlJbnN0YWxsZXJEb2N1bWVudERhdGFiYXNlU2VjcmV0QXR0YWNobWVudDI5NzUzQjdDJyxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdyZXBvc2l0b3J5IGluc3RhbGxlciBpYW0gcGVybWlzc2lvbnM6IGluc3RhbGxlciBnZXQnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBhcnJheVdpdGgoXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICdzMzpMaXN0KicsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUmVzb3VyY2U6IGFycmF5V2l0aChcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6czM6OjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IENXQV9BU1NFVF9MSU5VWC5CdWNrZXQsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpzMzo6OicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogQ1dBX0FTU0VUX0xJTlVYLkJ1Y2tldCxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnLyonLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICksXG4gICAgICAgIH0sXG4gICAgICApLFxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdkZWZhdWx0IHJlcG9zaXRvcnkgaW5zdGFsbGVyIGxvZyBncm91cCBjcmVhdGVkIGNvcnJlY3RseScsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6TG9nUmV0ZW50aW9uJywge1xuICAgIFJldGVudGlvbkluRGF5czogMyxcbiAgICBMb2dHcm91cE5hbWU6ICcvcmVuZGVyZmFybS9yZXBvc2l0b3J5SW5zdGFsbGVyJyxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFsbGVyIGxvZ3MgYWxsIHJlcXVpcmVkIGZpbGVzJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICAvLyBUaGUgQ2xvdWRXYXRjaEFnZW50IHN0b3JlcyB0aGUgYWdlbnQgY29uZmlndXJhdGlvbiBpbiBhbiBTU00gUGFyYW1ldGVyLiBDaGVjayBpdCBmb3IgdGhlIHJlcXVpcmVkIHNldHVwLlxuICAvLyBOb3RlOiBUaGlzIHdvdWxkIGJlIGJldHRlciBpbXBsZW1lbnRlZCB1c2luZyB0aGUgdG9vbHMgaW46IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzg0NDRcbiAgLy8gYnV0IHRoYXQgaXMgbm90IHlldCBhdmFpbGFibGUuXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpTU006OlBhcmFtZXRlcicsIHtcbiAgICBUeXBlOiAnU3RyaW5nJyxcbiAgICBWYWx1ZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICd7XFxcImxvZ3NcXFwiOntcXFwibG9nc19jb2xsZWN0ZWRcXFwiOntcXFwiZmlsZXNcXFwiOntcXFwiY29sbGVjdF9saXN0XFxcIjpbe1xcXCJsb2dfZ3JvdXBfbmFtZVxcXCI6XFxcIicsXG4gICAgICAgICAge30sIC8vIGxvZyBncm91cCBuYW1lLiBjaGVja2VkIGluIGFub3RoZXIgdGVzdC5cbiAgICAgICAgICAnXFxcIixcXFwibG9nX3N0cmVhbV9uYW1lXFxcIjpcXFwiY2xvdWQtaW5pdC1vdXRwdXQte2luc3RhbmNlX2lkfVxcXCIsXFxcImZpbGVfcGF0aFxcXCI6XFxcIi92YXIvbG9nL2Nsb3VkLWluaXQtb3V0cHV0LmxvZ1xcXCIsXFxcInRpbWV6b25lXFxcIjpcXFwiTG9jYWxcXFwifSx7XFxcImxvZ19ncm91cF9uYW1lXFxcIjpcXFwiJyxcbiAgICAgICAgICB7fSwgLy8gbG9nIGdyb3VwIG5hbWUgYWdhaW4uXG4gICAgICAgICAgJ1xcXCIsXFxcImxvZ19zdHJlYW1fbmFtZVxcXCI6XFxcImRlYWRsaW5lUmVwb3NpdG9yeUluc3RhbGxhdGlvbkxvZ3Mte2luc3RhbmNlX2lkfVxcXCIsXFxcImZpbGVfcGF0aFxcXCI6XFxcIi90bXAvYml0cm9ja19pbnN0YWxsZXIubG9nXFxcIixcXFwidGltZXpvbmVcXFwiOlxcXCJMb2NhbFxcXCJ9XX19LFxcXCJsb2dfc3RyZWFtX25hbWVcXFwiOlxcXCJEZWZhdWx0TG9nU3RyZWFtLXtpbnN0YW5jZV9pZH1cXFwiLFxcXCJmb3JjZV9mbHVzaF9pbnRlcnZhbFxcXCI6MTV9fScsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdyZXBvc2l0b3J5IG1vdW50cyByZXBvc2l0b3J5IGZpbGVzeXN0ZW0nLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3QgdXNlckRhdGEgPSAocmVwby5ub2RlLmRlZmF1bHRDaGlsZCBhcyBBdXRvU2NhbGluZ0dyb3VwKS51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoJ21vdW50RWZzLnNoICR7VG9rZW5bVE9LRU4uXFxcXGQrXX0gL21udC9lZnMvZnMxIGZhbHNlIHJ3JykpKTtcbn0pO1xuXG50ZXN0LmVhY2goW1xuICBbUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLCAnRGVsZXRlJ10sXG4gIFtSZW1vdmFsUG9saWN5LlJFVEFJTiwgJ1JldGFpbiddLFxuICBbUmVtb3ZhbFBvbGljeS5TTkFQU0hPVCwgJ1NuYXBzaG90J10sXG5dKSgncmVwb3NpdG9yeSBob25vcnMgZGF0YWJhc2UgcmVtb3ZhbCBwb2xpY3k6ICVwJywgKHBvbGljeTogUmVtb3ZhbFBvbGljeSwgZXhwZWN0ZWQ6IHN0cmluZykgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZW1vdmFsUG9saWN5OiB7XG4gICAgICBkYXRhYmFzZTogcG9saWN5LFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQkNsdXN0ZXInLCB7XG4gICAgRGVsZXRpb25Qb2xpY3k6IGV4cGVjdGVkLFxuICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG59KTtcblxudGVzdC5lYWNoKFtcbiAgW1JlbW92YWxQb2xpY3kuREVTVFJPWSwgJ0RlbGV0ZSddLFxuICBbUmVtb3ZhbFBvbGljeS5SRVRBSU4sICdSZXRhaW4nXSxcbiAgW1JlbW92YWxQb2xpY3kuU05BUFNIT1QsICdTbmFwc2hvdCddLFxuXSkoJ3JlcG9zaXRvcnkgaG9ub3JzIGZpbGVzeXN0ZW0gcmVtb3ZhbCBwb2xpY3k6ICVwJywgKHBvbGljeTogUmVtb3ZhbFBvbGljeSwgZXhwZWN0ZWQ6IHN0cmluZykgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZW1vdmFsUG9saWN5OiB7XG4gICAgICBmaWxlc3lzdGVtOiBwb2xpY3ksXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUZTOjpGaWxlU3lzdGVtJywge1xuICAgIERlbGV0aW9uUG9saWN5OiBleHBlY3RlZCxcbiAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgd2FybnMgaWYgcmVtb3ZhbCBwb2xpY3kgZm9yIGZpbGVzeXN0ZW0gd2hlbiBmaWxlc3lzdGVtIHByb3ZpZGVkJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCB0ZXN0RUZTID0gbmV3IEVmc0ZpbGVTeXN0ZW0oc3RhY2ssICdUZXN0RWZzRmlsZVN5c3RlbScsIHtcbiAgICB2cGMsXG4gIH0pO1xuICBjb25zdCB0ZXN0QVAgPSBuZXcgQWNjZXNzUG9pbnQoc3RhY2ssICdUZXN0QWNjZXNzUG9pbnQnLCB7XG4gICAgZmlsZVN5c3RlbTogdGVzdEVGUyxcbiAgfSk7XG4gIGNvbnN0IHRlc3RGUyA9IG5ldyBNb3VudGFibGVFZnMoc3RhY2ssIHtcbiAgICBmaWxlc3lzdGVtOiB0ZXN0RUZTLFxuICAgIGFjY2Vzc1BvaW50OiB0ZXN0QVAsXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgZmlsZVN5c3RlbTogdGVzdEZTLFxuICAgIHZlcnNpb24sXG4gICAgcmVtb3ZhbFBvbGljeToge1xuICAgICAgZmlsZXN5c3RlbTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHJlcG8ubm9kZS5tZXRhZGF0YSkudG9FcXVhbChcbiAgICBleHBlY3QuYXJyYXlDb250YWluaW5nKFtcbiAgICAgIGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICAgICAgdHlwZTogJ2F3czpjZGs6d2FybmluZycsXG4gICAgICAgIGRhdGE6ICdSZW1vdmFsUG9saWN5IGZvciBmaWxlc3lzdGVtIHdpbGwgbm90IGJlIGFwcGxpZWQgc2luY2UgYSBmaWxlc3lzdGVtIGlzIG5vdCBiZWluZyBjcmVhdGVkIGJ5IHRoaXMgY29uc3RydWN0JyxcbiAgICAgIH0pLFxuICAgIF0pLFxuICApO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgd2FybnMgaWYgcmVtb3ZhbCBwb2xpY3kgZm9yIGRhdGFiYXNlIHdoZW4gZGF0YWJhc2UgcHJvdmlkZWQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGZzRGF0YWJhc2UgPSBuZXcgRGF0YWJhc2VDbHVzdGVyKHN0YWNrLCAnVGVzdERiQ2x1c3RlcicsIHtcbiAgICBtYXN0ZXJVc2VyOiB7XG4gICAgICB1c2VybmFtZTogJ21hc3RlcicsXG4gICAgfSxcbiAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihcbiAgICAgIEluc3RhbmNlQ2xhc3MuUjQsXG4gICAgICBJbnN0YW5jZVNpemUuTEFSR0UsXG4gICAgKSxcbiAgICB2cGMsXG4gICAgdnBjU3VibmV0czoge1xuICAgICAgb25lUGVyQXo6IHRydWUsXG4gICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICBkYXRhYmFzZTogRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHsgZGF0YWJhc2U6IGZzRGF0YWJhc2UsIGxvZ2luOiBmc0RhdGFiYXNlLnNlY3JldCEgfSksXG4gICAgdmVyc2lvbixcbiAgICByZW1vdmFsUG9saWN5OiB7XG4gICAgICBkYXRhYmFzZTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHJlcG8ubm9kZS5tZXRhZGF0YSkudG9FcXVhbChcbiAgICBleHBlY3QuYXJyYXlDb250YWluaW5nKFtcbiAgICAgIGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICAgICAgdHlwZTogJ2F3czpjZGs6d2FybmluZycsXG4gICAgICAgIGRhdGE6ICdSZW1vdmFsUG9saWN5IGZvciBkYXRhYmFzZSB3aWxsIG5vdCBiZSBhcHBsaWVkIHNpbmNlIGEgZGF0YWJhc2UgaXMgbm90IGJlaW5nIGNyZWF0ZWQgYnkgdGhpcyBjb25zdHJ1Y3QnLFxuICAgICAgfSksXG4gICAgXSksXG4gICk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBjcmVhdGVzIGRlYWRsaW5lRGF0YWJhc2UgaWYgbm9uZSBwcm92aWRlZCcsICgpID0+IHtcbiAgY29uc3QgdGVzdEVGUyA9IG5ldyBFZnNGaWxlU3lzdGVtKHN0YWNrLCAnVGVzdEVmc0ZpbGVTeXN0ZW0nLCB7XG4gICAgdnBjLFxuICB9KTtcbiAgY29uc3QgdGVzdEFQID0gbmV3IEFjY2Vzc1BvaW50KHN0YWNrLCAnVGVzdEFjY2Vzc1BvaW50Jywge1xuICAgIGZpbGVTeXN0ZW06IHRlc3RFRlMsXG4gIH0pO1xuICBjb25zdCB0ZXN0RlMgPSBuZXcgTW91bnRhYmxlRWZzKHN0YWNrLCB7XG4gICAgZmlsZXN5c3RlbTogdGVzdEVGUyxcbiAgICBhY2Nlc3NQb2ludDogdGVzdEFQLFxuICB9KTtcblxuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgZmlsZVN5c3RlbTogdGVzdEZTLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RG9jREI6OkRCQ2x1c3RlcicpKTtcbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RG9jREI6OkRCSW5zdGFuY2UnKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpEb2NEQjo6REJDbHVzdGVyJywge1xuICAgIEVuYWJsZUNsb3Vkd2F0Y2hMb2dzRXhwb3J0czogWyAnYXVkaXQnIF0sXG4gIH0sIFJlc291cmNlUGFydC5Qcm9wZXJ0aWVzKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpEb2NEQjo6REJDbHVzdGVyUGFyYW1ldGVyR3JvdXAnLCB7XG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgYXVkaXRfbG9nczogJ2VuYWJsZWQnLFxuICAgIH0sXG4gIH0sIFJlc291cmNlUGFydC5Qcm9wZXJ0aWVzKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpEb2NEQjo6REJJbnN0YW5jZScsIHtcbiAgICBBdXRvTWlub3JWZXJzaW9uVXBncmFkZTogdHJ1ZSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ2Rpc2FibGluZyBBdWRpdCBsb2dnaW5nIGRvZXMgbm90IGVuYWJsZSBDbG91ZHdhdGNoIGF1ZGl0IGxvZ3MnLCAoKSA9PiB7XG4gIGNvbnN0IHRlc3RFRlMgPSBuZXcgRWZzRmlsZVN5c3RlbShzdGFjaywgJ1Rlc3RFZnNGaWxlU3lzdGVtJywge1xuICAgIHZwYyxcbiAgfSk7XG4gIGNvbnN0IHRlc3RBUCA9IG5ldyBBY2Nlc3NQb2ludChzdGFjaywgJ1Rlc3RBY2Nlc3NQb2ludCcsIHtcbiAgICBmaWxlU3lzdGVtOiB0ZXN0RUZTLFxuICB9KTtcbiAgY29uc3QgdGVzdEZTID0gbmV3IE1vdW50YWJsZUVmcyhzdGFjaywge1xuICAgIGZpbGVzeXN0ZW06IHRlc3RFRlMsXG4gICAgYWNjZXNzUG9pbnQ6IHRlc3RBUCxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIGZpbGVTeXN0ZW06IHRlc3RGUyxcbiAgICB2ZXJzaW9uLFxuICAgIGRhdGFiYXNlQXVkaXRMb2dnaW5nOiBmYWxzZSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpEb2NEQjo6REJDbHVzdGVyJykpO1xuICBleHBlY3RDREsoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RG9jREI6OkRCQ2x1c3RlcicsIHtcbiAgICBFbmFibGVDbG91ZHdhdGNoTG9nc0V4cG9ydHM6IFsgJ2F1ZGl0JyBdLFxuICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICBleHBlY3RDREsoc3RhY2spLm5vdFRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RG9jREI6OkRCQ2x1c3RlclBhcmFtZXRlckdyb3VwJywge1xuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIGF1ZGl0X2xvZ3M6ICdlbmFibGVkJyxcbiAgICB9LFxuICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgd2FybnMgaWYgZGF0YWJhc2VBdWRpdExvZ2dpbmcgZGVmaW5lZCBhbmQgZGF0YWJhc2UgaXMgc3BlY2lmaWVkJywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBmc0RhdGFiYXNlID0gbmV3IERhdGFiYXNlQ2x1c3RlcihzdGFjaywgJ1Rlc3REYkNsdXN0ZXInLCB7XG4gICAgbWFzdGVyVXNlcjoge1xuICAgICAgdXNlcm5hbWU6ICdtYXN0ZXInLFxuICAgIH0sXG4gICAgaW5zdGFuY2VUeXBlOiBJbnN0YW5jZVR5cGUub2YoXG4gICAgICBJbnN0YW5jZUNsYXNzLlI0LFxuICAgICAgSW5zdGFuY2VTaXplLkxBUkdFLFxuICAgICksXG4gICAgdnBjLFxuICAgIHZwY1N1Ym5ldHM6IHtcbiAgICAgIG9uZVBlckF6OiB0cnVlLFxuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFLFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgICByZW1vdmFsUG9saWN5OiB7XG4gICAgICBmaWxlc3lzdGVtOiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSxcbiAgICBkYXRhYmFzZTogRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHsgZGF0YWJhc2U6IGZzRGF0YWJhc2UsIGxvZ2luOiBmc0RhdGFiYXNlLnNlY3JldCEgfSksXG4gICAgZGF0YWJhc2VBdWRpdExvZ2dpbmc6IHRydWUsXG4gIH0pO1xuXG4gIGNvbnN0IHdhcm5pbmdNc2cgPSAnVGhlIHBhcmFtZXRlciBkYXRhYmFzZUF1ZGl0TG9nZ2luZyBvbmx5IGhhcyBhbiBlZmZlY3Qgd2hlbiB0aGUgUmVwb3NpdG9yeSBpcyBjcmVhdGluZyBpdHMgb3duIGRhdGFiYXNlLlxcbicgK1xuICAgICdQbGVhc2UgZW5zdXJlIHRoYXQgdGhlIERhdGFiYXNlIHByb3ZpZGVkIGlzIGNvbmZpZ3VyZWQgY29ycmVjdGx5Lic7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVwby5ub2RlLm1ldGFkYXRhKS50b0VxdWFsKFxuICAgIGV4cGVjdC5hcnJheUNvbnRhaW5pbmcoW1xuICAgICAgZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgICB0eXBlOiAnYXdzOmNkazp3YXJuaW5nJyxcbiAgICAgICAgZGF0YTogd2FybmluZ01zZyxcbiAgICAgIH0pLFxuICAgIF0pLFxuICApO1xufSk7XG5cbnRlc3QoJ2hvbm9ycyBzdWJuZXQgc3BlY2lmaWNhdGlvbicsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgZGVwZW5kZW5jeVN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ0RlcFN0YWNrJyk7XG4gIGNvbnN0IGRlcGVuZGVuY3lWcGMgPSBuZXcgVnBjKGRlcGVuZGVuY3lTdGFjaywgJ0RlcFZwYycpO1xuXG4gIGNvbnN0IHN1Ym5ldHMgPSBbXG4gICAgU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKGRlcGVuZGVuY3lTdGFjaywgJ1N1Ym5ldDEnLCB7XG4gICAgICBzdWJuZXRJZDogJ1N1Ym5ldElEMScsXG4gICAgICBhdmFpbGFiaWxpdHlab25lOiAndXMtd2VzdC0yYScsXG4gICAgfSksXG4gICAgU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKGRlcGVuZGVuY3lTdGFjaywgJ1N1Ym5ldDInLCB7XG4gICAgICBzdWJuZXRJZDogJ1N1Ym5ldElEMicsXG4gICAgICBhdmFpbGFiaWxpdHlab25lOiAndXMtd2VzdC0yYicsXG4gICAgfSksXG4gIF07XG4gIGNvbnN0IGlzb2xhdGVkU3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnSXNvbGF0ZWRTdGFjaycpO1xuXG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoaXNvbGF0ZWRTdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjOiBkZXBlbmRlbmN5VnBjLFxuICAgIHZlcnNpb24sXG4gICAgdnBjU3VibmV0czoge1xuICAgICAgc3VibmV0cyxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkRvY0RCOjpEQlN1Ym5ldEdyb3VwJywge1xuICAgIFN1Ym5ldElkczogW1xuICAgICAgJ1N1Ym5ldElEMScsXG4gICAgICAnU3VibmV0SUQyJyxcbiAgICBdLFxuICB9KSk7XG4gIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVGUzo6TW91bnRUYXJnZXQnLCB7IFN1Ym5ldElkOiAnU3VibmV0SUQxJyB9KSk7XG4gIGV4cGVjdENESyhpc29sYXRlZFN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVGUzo6TW91bnRUYXJnZXQnLCB7IFN1Ym5ldElkOiAnU3VibmV0SUQyJyB9KSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBob25vcnMgZGF0YWJhc2UgaW5zdGFuY2UgY291bnQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGluc3RhbmNlQ291bnQgPSAyO1xuXG4gIC8vIFdIRU5cbiAgbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICAgIGRvY3VtZW50RGJJbnN0YW5jZUNvdW50OiBpbnN0YW5jZUNvdW50LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oY291bnRSZXNvdXJjZXNMaWtlKCdBV1M6OkRvY0RCOjpEQkluc3RhbmNlJywgaW5zdGFuY2VDb3VudCwge1xuICAgIEF1dG9NaW5vclZlcnNpb25VcGdyYWRlOiB0cnVlLFxuICB9KSk7XG59KTtcblxudGVzdCgncmVwb3NpdG9yeSBob25vcnMgZGF0YWJhc2UgcmV0ZW50aW9uIHBlcmlvZCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgcGVyaW9kID0gMjA7XG5cbiAgLy8gV0hFTlxuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgYmFja3VwT3B0aW9uczoge1xuICAgICAgZGF0YWJhc2VSZXRlbnRpb246IER1cmF0aW9uLmRheXMocGVyaW9kKSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpEb2NEQjo6REJDbHVzdGVyJywge1xuICAgIEJhY2t1cFJldGVudGlvblBlcmlvZDogcGVyaW9kLFxuICB9KSk7XG59KTtcblxudGVzdCgnd2FybnMgaWYgYm90aCByZXRlbnRpb24gcGVyaW9kIGFuZCBkYXRhYmFzZSBwcm92aWRlZCcsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgZnNEYXRhYmFzZSA9IG5ldyBEYXRhYmFzZUNsdXN0ZXIoc3RhY2ssICdUZXN0RGJDbHVzdGVyJywge1xuICAgIG1hc3RlclVzZXI6IHtcbiAgICAgIHVzZXJuYW1lOiAnbWFzdGVyJyxcbiAgICB9LFxuICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICApLFxuICAgIHZwYyxcbiAgICB2cGNTdWJuZXRzOiB7XG4gICAgICBvbmVQZXJBejogdHJ1ZSxcbiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBXSEVOXG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIGRhdGFiYXNlOiBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yRG9jREIoeyBkYXRhYmFzZTogZnNEYXRhYmFzZSwgbG9naW46IGZzRGF0YWJhc2Uuc2VjcmV0ISB9KSxcbiAgICB2ZXJzaW9uLFxuICAgIGJhY2t1cE9wdGlvbnM6IHtcbiAgICAgIGRhdGFiYXNlUmV0ZW50aW9uOiBEdXJhdGlvbi5kYXlzKDIwKSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChyZXBvLm5vZGUubWV0YWRhdGEpLnRvRXF1YWwoXG4gICAgZXhwZWN0LmFycmF5Q29udGFpbmluZyhbXG4gICAgICBleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgICAgIHR5cGU6ICdhd3M6Y2RrOndhcm5pbmcnLFxuICAgICAgICBkYXRhOiAnQmFja3VwIHJldGVudGlvbiBmb3IgZGF0YWJhc2Ugd2lsbCBub3QgYmUgYXBwbGllZCBzaW5jZSBhIGRhdGFiYXNlIGlzIG5vdCBiZWluZyBjcmVhdGVkIGJ5IHRoaXMgY29uc3RydWN0JyxcbiAgICAgIH0pLFxuICAgIF0pLFxuICApO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgY3JlYXRlcyBmaWxlc3lzdGVtIGlmIG5vbmUgcHJvdmlkZWQnLCAoKSA9PiB7XG5cbiAgY29uc3QgZnNEYXRhYmFzZSA9IG5ldyBEYXRhYmFzZUNsdXN0ZXIoc3RhY2ssICdUZXN0RGJDbHVzdGVyJywge1xuICAgIG1hc3RlclVzZXI6IHtcbiAgICAgIHVzZXJuYW1lOiAnbWFzdGVyJyxcbiAgICB9LFxuICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKFxuICAgICAgSW5zdGFuY2VDbGFzcy5SNCxcbiAgICAgIEluc3RhbmNlU2l6ZS5MQVJHRSxcbiAgICApLFxuICAgIHZwYyxcbiAgICB2cGNTdWJuZXRzOiB7XG4gICAgICBvbmVQZXJBejogdHJ1ZSxcbiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICB9LFxuICAgIGJhY2t1cDoge1xuICAgICAgcmV0ZW50aW9uOiBEdXJhdGlvbi5kYXlzKDE1KSxcbiAgICB9LFxuICB9KTtcblxuICBpZiAoIWZzRGF0YWJhc2Uuc2VjcmV0KSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdzZWNyZXQgY2Fubm90IGJlIG51bGwnKTtcbiAgfVxuXG4gIC8vIFdIRU5cbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgZGF0YWJhc2U6IERhdGFiYXNlQ29ubmVjdGlvbi5mb3JEb2NEQih7IGRhdGFiYXNlOiBmc0RhdGFiYXNlLCBsb2dpbjogZnNEYXRhYmFzZS5zZWNyZXQgfSksXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFRlM6OkZpbGVTeXN0ZW0nKSk7XG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVGUzo6TW91bnRUYXJnZXQnKSk7XG4gIGV4cGVjdChyZXBvLm5vZGUudHJ5RmluZENoaWxkKCdQYWRFZnNTdG9yYWdlJykpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChyZXBvLm5vZGUuZmluZENoaWxkKCdGaWxlU3lzdGVtJykubm9kZS50cnlGaW5kQ2hpbGQoJ1BhZGRpbmdBY2Nlc3NQb2ludCcpKS50b0JlRGVmaW5lZCgpO1xufSk7XG5cbnRlc3QoJ2RlZmF1bHQgcmVwb3NpdG9yeSBpbnN0YW5jZSBpcyBjcmVhdGVkIHVzaW5nIHVzZXIgZGVmaW5lZCBpbnN0YWxsYXRpb24gcGF0aCBwcmVmaXgnLCAoKSA9PiB7XG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgcmVwb3NpdG9yeUluc3RhbGxhdGlvblByZWZpeDogJ3h5eicsXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgY29uc3Qgc2NyaXB0ID0gKHJlcG8ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQXV0b1NjYWxpbmdHcm91cCkudXNlckRhdGE7XG4gIGV4cGVjdChzY3JpcHQucmVuZGVyKCkpLnRvTWF0Y2goL1xcL21udFxcL2Vmc1xcL2ZzMVxcL3h5ei8pO1xufSk7XG5cbnRlc3QoJ2RlZmF1bHQgcmVwb3NpdG9yeSBpbnN0YW5jZSBpcyBjcmVhdGVkIHVzaW5nIHVzZXIgZGVmaW5lZCBpbnN0YWxsYXRpb24gcGF0aCBwcmVmaXggd2l0aCBleHRyYSBzbGFzaGVzIGluIHBhdGgnLCAoKSA9PiB7XG4gIGNvbnN0IHJlcG8gPSBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHZlcnNpb24sXG4gICAgcmVwb3NpdG9yeUluc3RhbGxhdGlvblByZWZpeDogJy94eXovLycsXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgY29uc3Qgc2NyaXB0ID0gKHJlcG8ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQXV0b1NjYWxpbmdHcm91cCkudXNlckRhdGE7XG4gIGV4cGVjdChzY3JpcHQucmVuZGVyKCkpLnRvTWF0Y2goL1xcL21udFxcL2Vmc1xcL2ZzMVxcL3h5ei8pO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFuY2UgaXMgY3JlYXRlZCB3aXRoIHVzZXIgZGVmaW5lZCB0aW1lb3V0JywgKCkgPT4ge1xuICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ3JlcG9zaXRvcnlJbnN0YWxsZXInLCB7XG4gICAgdnBjLFxuICAgIHJlcG9zaXRvcnlJbnN0YWxsYXRpb25UaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDMwKSxcbiAgICB2ZXJzaW9uLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJywge1xuICAgIENyZWF0aW9uUG9saWN5OiB7XG4gICAgICBSZXNvdXJjZVNpZ25hbDoge1xuICAgICAgICBDb3VudDogMSxcbiAgICAgICAgVGltZW91dDogJ1BUMzBNJyxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgaW5zdGFuY2UgaXMgY3JlYXRlZCB3aXRoIGNvcnJlY3QgaW5zdGFsbGVyIHBhdGggdmVyc2lvbicsICgpID0+IHtcbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgcmVwb3NpdG9yeUluc3RhbGxhdGlvblRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMzApLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgY29uc3Qgc2NyaXB0ID0gKHJlcG8ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQXV0b1NjYWxpbmdHcm91cCkudXNlckRhdGE7XG4gIGV4cGVjdChzY3JpcHQucmVuZGVyKCkpLnRvTWF0Y2goLzEwXFwuMVxcLjlcXC4yLyk7XG59KTtcblxudGVzdC5lYWNoKFtcbiAgJ3Rlc3QtcHJlZml4LycsXG4gICcnLFxuXSkoJ3JlcG9zaXRvcnkgaW5zdGFuY2UgaXMgY3JlYXRlZCB3aXRoIGNvcnJlY3QgTG9nR3JvdXAgcHJlZml4ICVzJywgKHRlc3RQcmVmaXg6IHN0cmluZykgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBpZCA9ICdyZXBvc2l0b3J5SW5zdGFsbGVyJztcblxuICAvLyBXSEVOXG4gIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCBpZCwge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICAgIGxvZ0dyb3VwUHJvcHM6IHtcbiAgICAgIGxvZ0dyb3VwUHJlZml4OiB0ZXN0UHJlZml4LFxuICAgIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6TG9nUmV0ZW50aW9uJywge1xuICAgIExvZ0dyb3VwTmFtZTogdGVzdFByZWZpeCArIGlkLFxuICB9KSk7XG59KTtcblxudGVzdCgndmFsaWRhdGUgaW5zdGFuY2Ugc2VsZi10ZXJtaW5hdGlvbicsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uVGltZW91dDogRHVyYXRpb24ubWludXRlcygzMCksXG4gICAgdmVyc2lvbixcbiAgfSk7XG4gIGNvbnN0IGFzZ0xvZ2ljYWxJZCA9IHN0YWNrLmdldExvZ2ljYWxJZChyZXBvLm5vZGUuZGVmYXVsdENoaWxkIS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5FbGVtZW50KTtcblxuICAvLyBUSEVOXG4gIGNvbnN0IHJlZ2lvblRva2VuID0gZXNjYXBlVG9rZW5SZWdleCgnJHtUb2tlbltBV1MuUmVnaW9uLlxcXFxkK119Jyk7XG4gIGNvbnN0IGV4cGVjdGVkU3RyaW5nID0gYGZ1bmN0aW9uIGV4aXRUcmFwXFxcXChcXFxcKVxcXFx7XFxuZXhpdENvZGU9XFxcXCRcXFxcP1xcbnNsZWVwIDFtXFxuSU5TVEFOQ0U9XCJcXFxcJFxcXFwoY3VybCBodHRwOlxcXFwvXFxcXC8xNjlcXFxcLjI1NFxcXFwuMTY5XFxcXC4yNTRcXFxcL2xhdGVzdFxcXFwvbWV0YS1kYXRhXFxcXC9pbnN0YW5jZS1pZFxcXFwpXCJcXG5BU0c9XCJcXFxcJFxcXFwoYXdzIC0tcmVnaW9uICR7cmVnaW9uVG9rZW59IGVjMiBkZXNjcmliZS10YWdzIC0tZmlsdGVycyBcIk5hbWU9cmVzb3VyY2UtaWQsVmFsdWVzPVxcXFwkXFxcXHtJTlNUQU5DRVxcXFx9XCIgXCJOYW1lPWtleSxWYWx1ZXM9YXdzOmF1dG9zY2FsaW5nOmdyb3VwTmFtZVwiIC0tcXVlcnkgXCJUYWdzXFxcXFswXFxcXF1cXFxcLlZhbHVlXCIgLS1vdXRwdXQgdGV4dFxcXFwpXCJcXG5hd3MgLS1yZWdpb24gJHtyZWdpb25Ub2tlbn0gYXV0b3NjYWxpbmcgdXBkYXRlLWF1dG8tc2NhbGluZy1ncm91cCAtLWF1dG8tc2NhbGluZy1ncm91cC1uYW1lIFxcXFwkXFxcXHtBU0dcXFxcfSAtLW1pbi1zaXplIDAgLS1tYXgtc2l6ZSAwIC0tZGVzaXJlZC1jYXBhY2l0eSAwXFxuXFxcXC9vcHRcXFxcL2F3c1xcXFwvYmluXFxcXC9jZm4tc2lnbmFsIC0tc3RhY2sgJHtzdGFjay5zdGFja05hbWV9IC0tcmVzb3VyY2UgJHthc2dMb2dpY2FsSWR9IC0tcmVnaW9uICR7cmVnaW9uVG9rZW59IC1lIFxcXFwkZXhpdENvZGUgXFxcXHxcXFxcfCBlY2hvICdGYWlsZWQgdG8gc2VuZCBDbG91ZGZvcm1hdGlvbiBTaWduYWwnXFxuXFxcXH1gO1xuICBleHBlY3QoKHJlcG8ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQXV0b1NjYWxpbmdHcm91cCkudXNlckRhdGEucmVuZGVyKCkpLnRvTWF0Y2gobmV3IFJlZ0V4cChleHBlY3RlZFN0cmluZykpO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogYXJyYXlXaXRoKFxuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiAnYXV0b3NjYWxpbmc6VXBkYXRlQXV0b1NjYWxpbmdHcm91cCcsXG4gICAgICAgICAgQ29uZGl0aW9uOiB7XG4gICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICAgJ2F1dG9zY2FsaW5nOlJlc291cmNlVGFnL3Jlc291cmNlTG9naWNhbElkJzogTmFtZXMudW5pcXVlSWQocmVwbyksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFJlc291cmNlOiAnKicsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246ICdlYzI6RGVzY3JpYmVUYWdzJyxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3JlcG9zaXRvcnkgY29uZmlndXJlIGNsaWVudCBpbnN0YW5jZScsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgcmVwb3NpdG9yeUluc3RhbGxhdGlvblRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMzApLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuICBjb25zdCBpbnN0YW5jZSA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlJywge1xuICAgIHZwYyxcbiAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gIH0pO1xuXG4gIC8vIFdIRU5cbiAgcmVwby5jb25maWd1cmVDbGllbnRJbnN0YW5jZSh7XG4gICAgaG9zdDogaW5zdGFuY2UsXG4gICAgbW91bnRQb2ludDogJy9tbnQvcmVwb3NpdG9yeScsXG4gIH0pO1xuICBjb25zdCB1c2VyRGF0YSA9IGluc3RhbmNlLnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gIC8vIFRIRU5cbiAgLy8gd2hpdGUtYm94IHRlc3RpbmcuIElmIHdlIG1vdW50IHRoZSBmaWxlc3lzdGVtLCB0aGVuIHdlJ3ZlIGNhbGxlZDogc2V0dXBEaXJlY3RDb25uZWN0KClcbiAgZXhwZWN0KHVzZXJEYXRhKS50b01hdGNoKG5ldyBSZWdFeHAoZXNjYXBlVG9rZW5SZWdleCgnbW91bnRFZnMuc2ggJHtUb2tlbltUT0tFTi5cXFxcZCtdfSAvbW50L3JlcG9zaXRvcnkgZmFsc2UgcncnKSkpO1xuXG4gIC8vIE1ha2Ugc3VyZSB3ZSBhZGRlZCB0aGUgREIgY29ubmVjdGlvbiBhcmdzXG4gIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaCgvLipleHBvcnQgLWYgY29uZmlndXJlX2RlYWRsaW5lX2RhdGFiYXNlLiovKTtcblxuICAvLyBNYWtlIHN1cmUgd2UgY2FsbCB0aGUgY29uZmlndXJlUmVwb3NpdG9yeURpcmVjdENvbm5lY3Qgc2NyaXB0IHdpdGggYXBwcm9wcmlhdGUgYXJndW1lbnQuXG4gIGNvbnN0IHJlZ2V4ID0gbmV3IFJlZ0V4cChlc2NhcGVUb2tlblJlZ2V4KCdcXCcvdG1wLyR7VG9rZW5bVE9LRU4uXFxcXGQrXX0ke1Rva2VuW1RPS0VOLlxcXFxkK119XFwnIFxcXFxcIi9tbnQvcmVwb3NpdG9yeS9EZWFkbGluZVJlcG9zaXRvcnlcXFxcXCInKSk7XG4gIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChyZWdleCk7XG59KTtcblxudGVzdCgnY29uZmlndXJlQ2xpZW50SW5zdGFuY2UgdXNlcyBzaW5nbGV0b24gZm9yIHJlcG8gY29uZmlnIHNjcmlwdCcsICgpID0+IHtcbiAgLy8gTm90ZTogSWYgdGhpcyB0ZXN0IGZhaWxzLCB0aGVuIGNoZWNrIHRoZSBhc3NldCBoYXNoIGZvciBSRVBPX0RDX0FTU0VUIC0tIGl0IG1heSBoYXZlIGNoYW5nZWQuXG5cbiAgLy8gR0lWRU5cbiAgY29uc3QgcmVwbyA9IG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAncmVwb3NpdG9yeUluc3RhbGxlcicsIHtcbiAgICB2cGMsXG4gICAgcmVwb3NpdG9yeUluc3RhbGxhdGlvblRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMzApLFxuICAgIHZlcnNpb24sXG4gIH0pO1xuICBjb25zdCBpbnN0YW5jZTEgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZTEnLCB7XG4gICAgdnBjLFxuICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgfSk7XG4gIGNvbnN0IGluc3RhbmNlMiA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlMicsIHtcbiAgICB2cGMsXG4gICAgaW5zdGFuY2VUeXBlOiBuZXcgSW5zdGFuY2VUeXBlKCd0My5zbWFsbCcpLFxuICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICB9KTtcblxuICAvLyBXSEVOXG4gIHJlcG8uY29uZmlndXJlQ2xpZW50SW5zdGFuY2Uoe1xuICAgIGhvc3Q6IGluc3RhbmNlMSxcbiAgICBtb3VudFBvaW50OiAnL21udC9yZXBvc2l0b3J5JyxcbiAgfSk7XG4gIHJlcG8uY29uZmlndXJlQ2xpZW50SW5zdGFuY2Uoe1xuICAgIGhvc3Q6IGluc3RhbmNlMixcbiAgICBtb3VudFBvaW50OiAnL21udC9yZXBvc2l0b3J5JyxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICAvLyBNYWtlIHN1cmUgdGhhdCBib3RoIGluc3RhbmNlcyBoYXZlIGFjY2VzcyB0byB0aGUgc2FtZSBBc3NldCBmb3IgdGhlIGNvbmZpZ3VyZVJlcG9zaXRvcnlEaXJlY3RDb25uZWN0IHNjcmlwdFxuICBleHBlY3RDREsoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIDIsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBhcnJheVdpdGgoXG4gICAgICAgIHtcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAnczM6R2V0QnVja2V0KicsXG4gICAgICAgICAgICAnczM6TGlzdConLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgUmVzb3VyY2U6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6czM6OjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFJFUE9fRENfQVNTRVQuQnVja2V0LFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6czM6OjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFJFUE9fRENfQVNTRVQuQnVja2V0LFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICcvKicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3dpbmRvd3MgY2xpZW50IGNhbm5vdCBkaXJlY3QgY29ubmVjdCB0byByZXBvc2l0b3J5JywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdyZXBvc2l0b3J5SW5zdGFsbGVyJywge1xuICAgIHZwYyxcbiAgICByZXBvc2l0b3J5SW5zdGFsbGF0aW9uVGltZW91dDogRHVyYXRpb24ubWludXRlcygzMCksXG4gICAgdmVyc2lvbixcbiAgfSk7XG4gIGNvbnN0IGluc3RhbmNlID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgdnBjLFxuICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RXaW5kb3dzKFdpbmRvd3NWZXJzaW9uLldJTkRPV1NfU0VSVkVSXzIwMTlfRU5HTElTSF9GVUxMX0JBU0UpLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdCgoKSA9PiB7XG4gICAgcmVwby5jb25maWd1cmVDbGllbnRJbnN0YW5jZSh7XG4gICAgICBob3N0OiBpbnN0YW5jZSxcbiAgICAgIG1vdW50UG9pbnQ6ICdkOlxcXFwnLFxuICAgIH0pO1xuICB9KS50b1Rocm93RXJyb3IoJ0RlYWRsaW5lIGRpcmVjdCBjb25uZWN0IG9uIFdpbmRvd3MgaG9zdHMgaXMgbm90IHlldCBzdXBwb3J0ZWQgYnkgdGhlIFJGREsuJyk7XG59KTtcblxuZGVzY3JpYmUoJ3RhZ2dpbmcnLCAoKSA9PiB7XG4gIHRlc3RDb25zdHJ1Y3RUYWdzKHtcbiAgICBjb25zdHJ1Y3ROYW1lOiAnUmVwb3NpdG9yeScsXG4gICAgY3JlYXRlQ29uc3RydWN0OiAoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgaXNvbGF0ZWRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdJc29sYXRlZFN0YWNrJyk7XG4gICAgICBuZXcgUmVwb3NpdG9yeShpc29sYXRlZFN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICAgICAgdnBjLFxuICAgICAgICB2ZXJzaW9uLFxuICAgICAgfSk7XG4gICAgICByZXR1cm4gaXNvbGF0ZWRTdGFjaztcbiAgICB9LFxuICAgIHJlc291cmNlVHlwZUNvdW50czoge1xuICAgICAgJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJzogMyxcbiAgICAgICdBV1M6OkRvY0RCOjpEQkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCc6IDEsXG4gICAgICAnQVdTOjpEb2NEQjo6REJTdWJuZXRHcm91cCc6IDEsXG4gICAgICAnQVdTOjpTZWNyZXRzTWFuYWdlcjo6U2VjcmV0JzogMSxcbiAgICAgICdBV1M6OkRvY0RCOjpEQkNsdXN0ZXInOiAxLFxuICAgICAgJ0FXUzo6RG9jREI6OkRCSW5zdGFuY2UnOiAxLFxuICAgICAgJ0FXUzo6SUFNOjpSb2xlJzogMSxcbiAgICAgICdBV1M6OkF1dG9TY2FsaW5nOjpBdXRvU2NhbGluZ0dyb3VwJzogMSxcbiAgICAgICdBV1M6OlNTTTo6UGFyYW1ldGVyJzogMSxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG5kZXNjcmliZSgnU2VjdXJpdHkgR3JvdXBzJywgKCkgPT4ge1xuICBsZXQgcmVwb3NpdG9yeVNlY3VyaXR5R3JvdXA6IElTZWN1cml0eUdyb3VwO1xuXG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIHJlcG9zaXRvcnlTZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdBZGRpdGlvbmFsU2VjdXJpdHlHcm91cCcsIHsgdnBjIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnRG9jREInLCAoKSA9PiB7XG5cbiAgICB0ZXN0KCdhZGRzIHNlY3VyaXR5IGdyb3VwcyBvbiBjb25zdHJ1Y3Rpb24nLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgICAgIHZlcnNpb24sXG4gICAgICAgIHZwYyxcbiAgICAgICAgc2VjdXJpdHlHcm91cHNPcHRpb25zOiB7XG4gICAgICAgICAgZGF0YWJhc2U6IHJlcG9zaXRvcnlTZWN1cml0eUdyb3VwLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpEb2NEQjo6REJDbHVzdGVyJywge1xuICAgICAgICBWcGNTZWN1cml0eUdyb3VwSWRzOiBhcnJheVdpdGgoc3RhY2sucmVzb2x2ZShyZXBvc2l0b3J5U2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQpKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ0VGUycsICgpID0+IHtcblxuICAgIHRlc3QoJ2FkZHMgc2VjdXJpdHkgZ3JvdXBzIG9uIGNvbnN0cnVjdGlvbicsICgpID0+IHtcbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBSZXBvc2l0b3J5KHN0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICAgICAgdmVyc2lvbixcbiAgICAgICAgdnBjLFxuICAgICAgICBzZWN1cml0eUdyb3Vwc09wdGlvbnM6IHtcbiAgICAgICAgICBmaWxlU3lzdGVtOiByZXBvc2l0b3J5U2VjdXJpdHlHcm91cCxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICAvLyBUaGUgRUZTIGNvbnN0cnVjdCBhZGRzIHRoZSBzZWN1cml0eSBncm91cCB0byBlYWNoIG1vdW50IHRhcmdldCwgYW5kIG9uZSBtb3VudCB0YXJnZXQgaXMgZ2VuZXJhdGVkIHBlciBzdWJuZXQuXG4gICAgICBjb25zdCBudW1Nb3VudFRhcmdldHMgPSB2cGMuc2VsZWN0U3VibmV0cygpLnN1Ym5ldHMubGVuZ3RoO1xuICAgICAgZXhwZWN0Q0RLKHN0YWNrKS50byhjb3VudFJlc291cmNlc0xpa2UoJ0FXUzo6RUZTOjpNb3VudFRhcmdldCcsIG51bU1vdW50VGFyZ2V0cywge1xuICAgICAgICBTZWN1cml0eUdyb3VwczogYXJyYXlXaXRoKHN0YWNrLnJlc29sdmUocmVwb3NpdG9yeVNlY3VyaXR5R3JvdXAuc2VjdXJpdHlHcm91cElkKSksXG4gICAgICB9KSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdJbnN0YWxsZXInLCAoKSA9PiB7XG5cbiAgICB0ZXN0KCdhZGRzIHNlY3VyaXR5IGdyb3VwcyBvbiBjb25zdHJ1Y3Rpb24nLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgUmVwb3NpdG9yeShzdGFjaywgJ1JlcG9zaXRvcnknLCB7XG4gICAgICAgIHZlcnNpb24sXG4gICAgICAgIHZwYyxcbiAgICAgICAgc2VjdXJpdHlHcm91cHNPcHRpb25zOiB7XG4gICAgICAgICAgaW5zdGFsbGVyOiByZXBvc2l0b3J5U2VjdXJpdHlHcm91cCxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkxhdW5jaENvbmZpZ3VyYXRpb24nLCB7XG4gICAgICAgIFNlY3VyaXR5R3JvdXBzOiBhcnJheVdpdGgoc3RhY2sucmVzb2x2ZShyZXBvc2l0b3J5U2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQpKSxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICB9KTtcbn0pO1xuXG50ZXN0KCd2YWxpZGF0ZXMgVmVyc2lvblF1ZXJ5IGlzIG5vdCBpbiBhIGRpZmZlcmVudCBzdGFjaycsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgbmV3U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnTmV3U3RhY2snKTtcbiAgdmVyc2lvbiA9IG5ldyBWZXJzaW9uUXVlcnkoc3RhY2ssICdWZXJzaW9uJyk7XG4gIG5ldyBSZXBvc2l0b3J5KG5ld1N0YWNrLCAnUmVwb3NpdG9yeScsIHtcbiAgICB2cGMsXG4gICAgdmVyc2lvbixcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBmdW5jdGlvbiBzeW50aCgpIHtcbiAgICBTeW50aFV0aWxzLnN5bnRoZXNpemUobmV3U3RhY2spO1xuICB9XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3ludGgpLnRvVGhyb3coJ0EgVmVyc2lvblF1ZXJ5IGNhbiBub3QgYmUgc3VwcGxpZWQgZnJvbSBhIGRpZmZlcmVudCBzdGFjaycpO1xufSk7XG5cbnRlc3QoJ2NyZWF0ZXMgYW4gRUZTIEFjY2Vzc1BvaW50IHdoZW4gbm8gZmlsZXN5c3RlbSBpcyBzdXBwbGllZCcsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCByZXBvID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgIHZlcnNpb24sXG4gICAgdnBjLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGNvbnN0IGVmc1Jlc291cmNlID0gKHJlcG8ubm9kZS5maW5kQ2hpbGQoJ0ZpbGVTeXN0ZW0nKSBhcyBDZm5FbGVtZW50KS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5GaWxlU3lzdGVtO1xuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFRlM6OkFjY2Vzc1BvaW50Jywge1xuICAgIEZpbGVTeXN0ZW1JZDogc3RhY2sucmVzb2x2ZShlZnNSZXNvdXJjZS5yZWYpLFxuICAgIFBvc2l4VXNlcjoge1xuICAgICAgR2lkOiAnMCcsXG4gICAgICBVaWQ6ICcwJyxcbiAgICB9LFxuICAgIFJvb3REaXJlY3Rvcnk6IHt9LFxuICB9KSk7XG59KTtcblxudGVzdCgndGhyb3dzIGFuIGVycm9yIGlmIHN1cHBsaWVkIGEgTW91bnRhYmxlRWZzIHdpdGggbm8gQWNjZXNzIFBvaW50JywgKCkgPT4ge1xuICAvLyBHSVZFTlxuICBjb25zdCBuZXdTdGFjayA9IG5ldyBTdGFjayhhcHAsICdOZXdTdGFjaycpO1xuICBjb25zdCBmcyA9IG5ldyBFZnNGaWxlU3lzdGVtKG5ld1N0YWNrLCAnRmlsZVN5c3RlbScsIHtcbiAgICB2cGMsXG4gIH0pO1xuICBjb25zdCBtb3VudGFibGVGcyA9IG5ldyBNb3VudGFibGVFZnMobmV3U3RhY2ssIHtcbiAgICBmaWxlc3lzdGVtOiBmcyxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBmdW5jdGlvbiB3aGVuKCkge1xuICAgIG5ldyBSZXBvc2l0b3J5KG5ld1N0YWNrLCAnUmVwbycsIHtcbiAgICAgIHZlcnNpb24sXG4gICAgICB2cGMsXG4gICAgICBmaWxlU3lzdGVtOiBtb3VudGFibGVGcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHdoZW4pLnRvVGhyb3coJ1doZW4gdXNpbmcgRUZTIHdpdGggdGhlIFJlcG9zaXRvcnksIHlvdSBtdXN0IHByb3ZpZGUgYW4gRUZTIEFjY2VzcyBQb2ludCcpO1xufSk7XG5cbnRlc3QoJ2ltcG9ydHMgcmVwb3NpdG9yeSBzZXR0aW5ncycsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3QgcmVwb3NpdG9yeVNldHRpbmdzID0gbmV3IEFzc2V0KHN0YWNrLCAnUmVwb3NpdG9yeVNldHRpbmdzQXNzZXQnLCB7XG4gICAgcGF0aDogX19maWxlbmFtZSxcbiAgfSk7XG5cbiAgLy8gV0hFTlxuICBjb25zdCByZXBvc2l0b3J5ID0gbmV3IFJlcG9zaXRvcnkoc3RhY2ssICdSZXBvc2l0b3J5Jywge1xuICAgIHZwYyxcbiAgICB2ZXJzaW9uLFxuICAgIHJlcG9zaXRvcnlTZXR0aW5ncyxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBjb25zdCBpbnN0YWxsZXJHcm91cCA9IHJlcG9zaXRvcnkubm9kZS50cnlGaW5kQ2hpbGQoJ0luc3RhbGxlcicpIGFzIEF1dG9TY2FsaW5nR3JvdXA7XG4gIGV4cGVjdChpbnN0YWxsZXJHcm91cC51c2VyRGF0YS5yZW5kZXIoKSkudG9Db250YWluKGBhd3MgczMgY3AgJyR7cmVwb3NpdG9yeVNldHRpbmdzLnMzT2JqZWN0VXJsfSdgKTtcbn0pO1xuIl19