"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 *
 * @stability stable
 */
class VersionQuery extends VersionQueryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.expression = props === null || props === void 0 ? void 0 : props.version;
        const match = ((_b = props === null || props === void 0 ? void 0 : props.version) !== null && _b !== void 0 ? _b : '').match(VersionQuery.EXPRESSION_REGEX);
        if (match === null) {
            throw new Error(`Invalid version expression "${props.version}`);
        }
        this.pinnedVersionComponents = (match
            // First capture group is the entire matched string, so slice it off
            .slice(1)
            // Capture groups that are not matched return as undefined, so we filter them out
            .filter(component => component)
            // Parse strings to numbers
            .map(component => Number(component)));
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props === null || props === void 0 ? void 0 : props.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props === null || props === void 0 ? void 0 : props.version),
        };
        this.deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = this.versionComponent({
            expressionIndex: 0,
            customResourceAttribute: 'MajorVersion',
        });
        this.minorVersion = this.versionComponent({
            expressionIndex: 1,
            customResourceAttribute: 'MinorVersion',
        });
        this.releaseVersion = this.versionComponent({
            expressionIndex: 2,
            customResourceAttribute: 'ReleaseVersion',
        });
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(this.deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: this.deadlineResource.getAttString('LinuxRepositoryInstaller'),
                s3Bucket: aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', this.deadlineResource.getAttString('S3Bucket')),
            },
        };
    }
    versionComponent(args) {
        const { expressionIndex, customResourceAttribute } = args;
        return this.pinnedVersionComponents.length > expressionIndex
            ? this.pinnedVersionComponents[expressionIndex]
            : core_1.Token.asNumber(this.deadlineResource.getAtt(customResourceAttribute));
    }
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     *
     * @stability stable
     */
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    /**
     * Returns whether this version is less than another version.
     *
     * @stability stable
     */
    isLessThan(other) {
        if (other.patchVersion !== 0) {
            throw new Error('Cannot compare a VersionQuery to a fully-qualified version with a non-zero patch number');
        }
        // We compare each component from highest order to lowest
        const componentGetters = [
            v => v.majorVersion,
            v => v.minorVersion,
            v => v.releaseVersion,
        ];
        for (const componentGetter of componentGetters) {
            const thisComponent = componentGetter(this);
            const otherComponent = componentGetter(other);
            if (core_1.Token.isUnresolved(thisComponent)) {
                // Unresolved components are unpinned. These will resolve to the latest and are not less than any provided
                // version
                return false;
            }
            else {
                const componentDiff = thisComponent - otherComponent;
                if (componentDiff !== 0) {
                    // If the components are different, return whether this component is smaller than the other component
                    return componentDiff < 0;
                }
            }
        }
        // If we've exited the loop naturally, it means all version components are pinned and equal. This means the version
        // is not less than the other, they are the same
        return false;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version === null || version === void 0 ? void 0 : version.split('.').map(x => parseInt(x));
        if (!components || (components === null || components === void 0 ? void 0 : components.length) != 4) {
            return false;
        }
        return true;
    }
    /**
     * A string representation of the version using the best available information at synthesis-time.
     *
     * This value is not guaranteed to be resolved, and is intended for output to CDK users.
     *
     * @stability stable
     */
    get versionString() {
        var _b;
        return (_b = this.expression) !== null && _b !== void 0 ? _b : '(latest)';
    }
}
exports.VersionQuery = VersionQuery;
_a = JSII_RTTI_SYMBOL_1;
VersionQuery[_a] = { fqn: "aws-rfdk.deadline.VersionQuery", version: "0.30.0" };
/**
 * Regular expression for valid version query expressions
 */
VersionQuery.EXPRESSION_REGEX = /^(?:(\d+)(?:\.(\d+)(?:\.(\d+)(?:\.(\d+))?)?)?)?$/;
//# sourceMappingURL=data:application/json;base64,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