"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerImages = exports.AwsThinkboxEulaAcceptance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path = require("path");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
/**
 * Choices for signifying the user's stance on the terms of the AWS Thinkbox End-User License Agreement (EULA).
 *
 * See: https://www.awsthinkbox.com/end-user-license-agreement
 *
 * @stability stable
 */
var AwsThinkboxEulaAcceptance;
(function (AwsThinkboxEulaAcceptance) {
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_REJECTS_AWS_THINKBOX_EULA"] = 0] = "USER_REJECTS_AWS_THINKBOX_EULA";
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_ACCEPTS_AWS_THINKBOX_EULA"] = 1] = "USER_ACCEPTS_AWS_THINKBOX_EULA";
})(AwsThinkboxEulaAcceptance = exports.AwsThinkboxEulaAcceptance || (exports.AwsThinkboxEulaAcceptance = {}));
/**
 * An API for interacting with publicly available Deadline container images published by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * Successful usage of the published Deadline container images with this class requires:
 *
 * 1) Explicit acceptance of the terms of the AWS Thinkbox End User License Agreement, under which Deadline is
 *     distributed; and
 * 2) The lambda on which the custom resource looks up the Thinkbox container images is able to make HTTPS
 *     requests to the official AWS Thinbox download site: https://downloads.thinkboxsoftware.com
 *
 * Resources Deployed
 * ------------------------
 * - A Lambda function containing a script to look up the AWS Thinkbox container image registry
 *
 * Security Considerations
 * ------------------------
 * - CDK deploys the code for this lambda as an S3 object in the CDK bootstrap bucket. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly
 *    recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS
 *    CloudTrail on your account to assist in post-incident analysis of compromised production environments.
 *
 * For example, to construct a RenderQueue using the images:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { AwsThinkboxEulaAcceptance, RenderQueue, Repository, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    images: images.forRenderQueue(),
 *    // ...
 * });
 * ```
 *
 * @stability stable
 */
class ThinkboxDockerImages extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.userAwsThinkboxEulaAcceptance = props.userAwsThinkboxEulaAcceptance;
        this.version = props === null || props === void 0 ? void 0 : props.version;
        const lambdaCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '08553416-1fc9-4be9-a818-609a31ae1b5b',
            description: 'Used by the ThinkboxDockerImages construct to look up the ECR repositories where AWS Thinkbox publishes Deadline container images.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'ecr-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const ecrProvider = new core_1.CustomResource(this, 'ThinkboxEcrProvider', {
            serviceToken: lambdaFunc.functionArn,
            properties: {
                // create a random string that will force the Lambda to "update" on each deployment. Changes to its output will
                // be propagated to any CloudFormation resource providers that reference the output ARN
                ForceRun: this.forceRun(),
            },
            resourceType: 'Custom::RFDK_EcrProvider',
        });
        this.node.defaultChild = ecrProvider;
        this.ecrBaseURI = ecrProvider.getAtt('EcrURIPrefix').toString();
        this.remoteConnectionServer = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER);
        this.licenseForwarder = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const errors = [];
        // Users must accept the AWS Thinkbox EULA to use the container images
        if (this.userAwsThinkboxEulaAcceptance !== AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA) {
            errors.push(ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE);
        }
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                errors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return errors;
    }
    ecrImageForRecipe(recipe) {
        let registryName = `${this.ecrBaseURI}${recipe}`;
        if (this.versionString) {
            registryName += `:${this.versionString}`;
        }
        return aws_ecs_1.ContainerImage.fromRegistry(registryName);
    }
    /**
     * Returns container images for use with the {@link RenderQueue} construct.
     *
     * @stability stable
     */
    forRenderQueue() {
        return this;
    }
    /**
     * Returns container images for use with the {@link UsageBasedLicensing} construct.
     *
     * @stability stable
     */
    forUsageBasedLicensing() {
        return this;
    }
    /**
     * A string representation of the Deadline version to retrieve images for.
     *
     * This can be undefined - in which case the latest available version of Deadline is used.
     */
    get versionString() {
        function numAsString(num) {
            return core_1.Token.isUnresolved(num) ? core_1.Token.asString(num) : num.toString();
        }
        const version = this.version;
        if (version) {
            const major = numAsString(version.majorVersion);
            const minor = numAsString(version.minorVersion);
            const release = numAsString(version.releaseVersion);
            return `${major}.${minor}.${release}`;
        }
        return undefined;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.ThinkboxDockerImages = ThinkboxDockerImages;
_a = JSII_RTTI_SYMBOL_1;
ThinkboxDockerImages[_a] = { fqn: "aws-rfdk.deadline.ThinkboxDockerImages", version: "0.30.0" };
/**
 * The AWS Thinkbox licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the AWS Thinkbox EULA.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

Deadline is provided by AWS Thinkbox under the AWS Thinkbox End User License
Agreement (EULA). By installing Deadline, you are agreeing to the terms of this
license. Follow the link below to read the terms of the AWS Thinkbox EULA.

https://www.awsthinkbox.com/end-user-license-agreement

By using the ThinkboxDockerImages to install Deadline you agree to the terms of
the AWS Thinkbox EULA.

Please set the userAwsThinkboxEulaAcceptance property to
USER_ACCEPTS_AWS_THINKBOX_EULA to signify your acceptance of the terms of the
AWS Thinkbox EULA.
`;
//# sourceMappingURL=data:application/json;base64,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