"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const fs = require("fs");
const path = require("path");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const _1 = require("./");
/**
 * Class for interacting with the Deadline stage directory.
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 *
 * Note: Current version of RFDK supports Deadline v10.1.9.2 and later.
 *
 * @stability stable
 */
class Stage {
    /**
     * Constructs a Stage instance.
     *
     * @stability stable
     */
    constructor(props) {
        Stage.validateManifest(props.manifest);
        this.dirPath = props.path;
        this.manifest = props.manifest;
    }
    /**
     * Loads and parses the manifest file from a given path.
     *
     * @param manifestPath The path to the manifest JSON file.
     * @stability stable
     */
    static loadManifest(manifestPath) {
        const content = fs.readFileSync(manifestPath, { encoding: 'utf8' });
        return JSON.parse(content);
    }
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory.
     * @stability stable
     */
    static fromDirectory(stagePath) {
        const manifest = Stage.loadManifest(path.join(stagePath, Stage.MANIFEST_REL_PATH));
        return new Stage({
            manifest,
            path: stagePath,
        });
    }
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    static validateManifest(rawManifest) {
        if (rawManifest !== undefined && typeof rawManifest !== 'object') {
            throw new TypeError(`Expected object but got ${typeof rawManifest}`);
        }
        const schema = rawManifest.schema;
        if (schema === undefined) {
            throw new Error('Manifest contains no "schema" key');
        }
        else if (typeof schema !== 'number') {
            throw new TypeError(`Expected a numeric "schema" but got: ${typeof schema}`);
        }
        const version = rawManifest.version;
        if (version === undefined) {
            throw new Error('Manifest contains no "version" key');
        }
        else if (typeof version !== 'string') {
            throw new TypeError(`Expected a string "version" but got: ${typeof version}`);
        }
        // Do minimum supported deadline version check
        const stagedVersion = _1.Version.parse(version);
        if (stagedVersion.isLessThan(_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION)) {
            throw new TypeError(`Staged Deadline Version (${version}) is less than the minimum supported version (${_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()})`);
        }
        return true;
    }
    /**
     * Creates a {@link Version} based on the manifest version.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @stability stable
     */
    getVersion(scope, id) {
        const releaseVersion = this.getReleaseVersion(this.manifest.version);
        return new _1.VersionQuery(scope, id, { version: releaseVersion });
    }
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage.
     *
     * @param scope The scope for the {@link DockerImageAsset}.
     * @param id The construct ID of the {@link DockerImageAsset}.
     * @param recipeName The name of the recipe.
     * @stability stable
     */
    imageFromRecipe(scope, id, recipeName) {
        const recipe = this.manifest.recipes[recipeName];
        if (!recipe) {
            throw new Error(`No such recipe: ${recipeName}`);
        }
        return new aws_ecr_assets_1.DockerImageAsset(scope, id, {
            directory: this.dirPath,
            ...recipe,
        });
    }
    /**
     * This removes the patch version from a full version string. No validation is done as that is handled
     * in the constructor with the version check.
     */
    getReleaseVersion(fullVersion) {
        const versionComponents = fullVersion.split('.');
        return `${versionComponents[0]}.${versionComponents[1]}.${versionComponents[2]}`;
    }
}
exports.Stage = Stage;
_a = JSII_RTTI_SYMBOL_1;
Stage[_a] = { fqn: "aws-rfdk.deadline.Stage", version: "0.30.0" };
/**
 * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
 */
Stage.MANIFEST_REL_PATH = 'manifest.json';
//# sourceMappingURL=data:application/json;base64,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