"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path_1 = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments.
 *
 * @stability stable
 */
class RenderQueue extends RenderQueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
        super(scope, id);
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        // The RCS does not currently support horizontal scaling behind a load-balancer, so we limit to at most one instance
        if (props.renderQueueSize && props.renderQueueSize.min !== undefined && props.renderQueueSize.min > 1) {
            throw new Error(`renderQueueSize.min cannot be greater than 1 - got ${props.renderQueueSize.min}`);
        }
        if (props.renderQueueSize && props.renderQueueSize.desired !== undefined && props.renderQueueSize.desired > 1) {
            throw new Error(`renderQueueSize.desired cannot be greater than 1 - got ${props.renderQueueSize.desired}`);
        }
        this.version = props === null || props === void 0 ? void 0 : props.version;
        let externalProtocol;
        if ((_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.externalTLS) {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
            if ((props.trafficEncryption.externalTLS.acmCertificate === undefined) ===
                (props.trafficEncryption.externalTLS.rfdkCertificate === undefined)) {
                throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
            }
            else if (props.trafficEncryption.externalTLS.rfdkCertificate) {
                if (props.trafficEncryption.externalTLS.rfdkCertificate.certChain === undefined) {
                    throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
                }
                this.clientCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', props.trafficEncryption.externalTLS.rfdkCertificate);
                this.certChain = props.trafficEncryption.externalTLS.rfdkCertificate.certChain;
            }
            else {
                if (props.trafficEncryption.externalTLS.acmCertificateChain === undefined) {
                    throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
                }
                this.clientCert = props.trafficEncryption.externalTLS.acmCertificate;
                this.certChain = props.trafficEncryption.externalTLS.acmCertificateChain;
            }
        }
        else {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP;
        }
        this.version = props.version;
        const internalProtocol = (_d = (_c = props.trafficEncryption) === null || _c === void 0 ? void 0 : _c.internalProtocol) !== null && _d !== void 0 ? _d : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && !props.hostname) {
            throw new Error('A hostname must be provided when the external protocol is HTTPS');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_f = (_e = props.renderQueueSize) === null || _e === void 0 ? void 0 : _e.min) !== null && _f !== void 0 ? _f : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_g = props.vpcSubnets) !== null && _g !== void 0 ? _g : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: (_h = props.instanceType) !== null && _h !== void 0 ? _h : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_j = props.renderQueueSize) === null || _j === void 0 ? void 0 : _j.desired,
            maxCapacity: 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: undefined,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.rollingUpdate(),
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: (_k = props.securityGroups) === null || _k === void 0 ? void 0 : _k.backend,
        });
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        if ((_l = props.enableLocalFileCaching) !== null && _l !== void 0 ? _l : false) {
            // Has to be done before any filesystems mount.
            this.enableFilecaching(this.asg);
        }
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        let loadBalancerFQDN;
        if (props.hostname) {
            const label = (_m = props.hostname.hostname) !== null && _m !== void 0 ? _m : 'renderqueue';
            if (props.hostname.hostname && !RenderQueue.RE_VALID_HOSTNAME.test(label)) {
                throw new Error(`Invalid RenderQueue hostname: ${label}`);
            }
            loadBalancerFQDN = `${label}.${props.hostname.zone.zoneName}`;
        }
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: (_o = props.vpcSubnetsAlb) !== null && _o !== void 0 ? _o : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
            internetFacing: false,
            deletionProtection: (_p = props.deletionProtection) !== null && _p !== void 0 ? _p : true,
            securityGroup: (_q = props.securityGroups) === null || _q === void 0 ? void 0 : _q.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_r = props.renderQueueSize) === null || _r === void 0 ? void 0 : _r.desired,
            domainZone: (_s = props.hostname) === null || _s === void 0 ? void 0 : _s.zone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN !== null && loadBalancerFQDN !== void 0 ? loadBalancerFQDN : this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    enableFilecaching(asg) {
        const script = core_2.ScriptAsset.fromPathConvention(this, 'FilecachingScript', {
            osType: asg.osType,
            baseName: 'enableCacheFilesd',
            rootDir: path_1.join(__dirname, '..', 'scripts'),
        });
        // A comment in userData to make this easier to test.
        asg.userData.addCommands('# RenderQueue file caching enabled');
        script.executeOn({
            host: asg,
        });
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "0.30.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
//# sourceMappingURL=data:application/json;base64,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