"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 *
 * @stability stable
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     *
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @stability stable
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     *
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @stability stable
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
_a = JSII_RTTI_SYMBOL_1;
DatabaseConnection[_a] = { fqn: "aws-rfdk.deadline.DatabaseConnection", version: "0.30.0" };
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        if (!this.isCompatibleDocDBVersion()) {
            core_1.Annotations.of(props.database).addError('engineVersion must be 3.6.0 to be compatible with Deadline');
        }
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        let added = false;
        const errorReasons = [];
        if (this.props.database instanceof aws_docdb_1.DatabaseCluster) {
            const resource = this.props.database.node.tryFindChild('Resource');
            // TODO: Replace this code with the addSecurityGroup method of DatabaseCluster once this PR is merged:
            // https://github.com/aws/aws-cdk/pull/13290
            if (resource instanceof aws_docdb_1.CfnDBCluster) {
                const cfnCluster = resource;
                const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
                if (cfnCluster.vpcSecurityGroupIds === undefined) {
                    cfnCluster.vpcSecurityGroupIds = securityGroupIds;
                }
                else {
                    cfnCluster.vpcSecurityGroupIds.push(...securityGroupIds);
                }
                added = true;
            }
            else {
                errorReasons.push('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.');
            }
        }
        else {
            errorReasons.push('The "database" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.');
        }
        if (!added) {
            core_1.Annotations.of(this.props.database).addWarning(`Failed to add the following security groups to ${this.props.database.node.id}: ${securityGroups.map(sg => sg.node.id).join(', ')}. ` +
                errorReasons.join(' '));
        }
    }
    /**
     * Deadline is only compatible with MongoDB 3.6. This function attempts to determine whether
     * the given DocDB version is compatible.
     */
    isCompatibleDocDBVersion() {
        var _b, _c;
        // The defaultChild of a DocDB DatabaseCluster is a CfnDBCluster, but we only have this
        // child if the customer didn't import from attributes. We can check the DB version by
        // checking the value of the engineVersion property of that object.
        if (this.props.database.node.defaultChild) {
            const cluster = this.props.database.node.defaultChild;
            return (_c = (_b = cluster.engineVersion) === null || _b === void 0 ? void 0 : _b.startsWith('3.6')) !== null && _c !== void 0 ? _c : false;
        }
        return true; // No information, assume it's compatible.
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        this.props.database.addSecurityGroup(...securityGroups);
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        var _b;
        const stack = core_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: host.osType,
            baseName: 'getSecretToFile',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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