"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const render_queue_1 = require("./render-queue");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 *
 * @stability stable
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 *
 * @stability stable
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 *
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 *
 * @stability stable
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 *
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 *
 * @stability stable
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo0";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from '@aws-cdk/aws-ec2';
 * import { AwsThinkboxEulaAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    vpc,
 *    images: images.forRenderQueue(),
 *    repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *    vpc,
 *    renderQueue,
 *    deadlineGroups: ['group_name'],
 *    instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *    workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *    naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *    vpc,
 *    renderQueue: renderQueue,
 *    spotFleets: [fleet],
 *    configuration: {
 *      enableResourceTracker: true,
 *    },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Create the Deadline Group associated with the Spot Fleet Request Configuration. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/pools-and-groups.html).
 * - Create the Deadline Pools to which the fleet Workers are added. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/pools-and-groups.html).
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *   - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *   - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *   - A policy to pass a fleet and instance role
 *   - A policy to create tags for spot fleet requests
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *    and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *    execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class ConfigureSpotEventPlugin extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2;
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: ['arn:aws:ec2:*:*:spot-fleet-request/*'],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = core_1.Construct.isConstruct(props.renderQueue) ? core_1.Stack.of(props.renderQueue).region : core_1.Stack.of(this).region;
        const timeoutMins = 15;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
                LAMBDA_TIMEOUT_MINS: timeoutMins.toString(),
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: core_1.Duration.minutes(timeoutMins),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        (_c = props.renderQueue.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: (_e = (_d = props.configuration) === null || _d === void 0 ? void 0 : _d.awsInstanceStatus) !== null && _e !== void 0 ? _e : SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: (_g = (_f = props.configuration) === null || _f === void 0 ? void 0 : _f.deleteEC2SpotInterruptedWorkers) !== null && _g !== void 0 ? _g : false,
            DeleteTerminatedSlaves: (_j = (_h = props.configuration) === null || _h === void 0 ? void 0 : _h.deleteSEPTerminatedWorkers) !== null && _j !== void 0 ? _j : false,
            IdleShutdown: (_m = (_l = (_k = props.configuration) === null || _k === void 0 ? void 0 : _k.idleShutdown) === null || _l === void 0 ? void 0 : _l.toMinutes({ integral: true })) !== null && _m !== void 0 ? _m : 10,
            Logging: (_p = (_o = props.configuration) === null || _o === void 0 ? void 0 : _o.loggingLevel) !== null && _p !== void 0 ? _p : SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: (_r = (_q = props.configuration) === null || _q === void 0 ? void 0 : _q.preJobTaskMode) !== null && _r !== void 0 ? _r : SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: (_t = (_s = props.configuration) === null || _s === void 0 ? void 0 : _s.region) !== null && _t !== void 0 ? _t : region,
            ResourceTracker: (_v = (_u = props.configuration) === null || _u === void 0 ? void 0 : _u.enableResourceTracker) !== null && _v !== void 0 ? _v : true,
            StaggerInstances: (_x = (_w = props.configuration) === null || _w === void 0 ? void 0 : _w.maximumInstancesStartedPerCycle) !== null && _x !== void 0 ? _x : 50,
            State: (_z = (_y = props.configuration) === null || _y === void 0 ? void 0 : _y.state) !== null && _z !== void 0 ? _z : SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: (_1 = (_0 = props.configuration) === null || _0 === void 0 ? void 0 : _0.strictHardCap) !== null && _1 !== void 0 ? _1 : false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: (_2 = props.renderQueue.certChain) === null || _2 === void 0 ? void 0 : _2.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return core_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each congiguration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        var _b;
        const securityGroupsToken = core_1.Lazy.any({ produce: () => {
                return fleet.securityGroups.map(sg => {
                    const securityGroupId = {
                        GroupId: sg.securityGroupId,
                    };
                    return securityGroupId;
                });
            } });
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(fleet.userData.render()) });
        const blockDeviceMappings = (fleet.blockDevices !== undefined ?
            this.synthesizeBlockDeviceMappings(fleet.blockDevices) : undefined);
        const { subnetIds } = fleet.subnets;
        const subnetId = subnetIds.join(',');
        const instanceTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.INSTANCE);
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const launchSpecifications = [];
        fleet.instanceTypes.map(instanceType => {
            const launchSpecification = {
                BlockDeviceMappings: blockDeviceMappings,
                IamInstanceProfile: {
                    Arn: fleet.instanceProfile.attrArn,
                },
                ImageId: fleet.imageId,
                KeyName: fleet.keyName,
                // Need to convert from IResolvable to bypass TypeScript
                SecurityGroups: securityGroupsToken,
                SubnetId: subnetId,
                // Need to convert from IResolvable to bypass TypeScript
                TagSpecifications: instanceTagsToken,
                UserData: userDataToken,
                InstanceType: instanceType.toString(),
            };
            launchSpecifications.push(launchSpecification);
        });
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchSpecifications: launchSpecifications,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: (_b = fleet.validUntil) === null || _b === void 0 ? void 0 : _b.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    /**
     * Synthesize an array of block device mappings from a list of block devices
     *
     * @param blockDevices list of block devices
     */
    synthesizeBlockDeviceMappings(blockDevices) {
        return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
            const { virtualName, ebsDevice: ebs } = volume;
            if (volume === aws_autoscaling_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
                return {
                    DeviceName: deviceName,
                    // To omit the device from the block device mapping, specify an empty string.
                    // See https://docs.aws.amazon.com/cli/latest/reference/ec2/request-spot-fleet.html
                    NoDevice: '',
                };
            }
            let Ebs;
            if (ebs) {
                const { iops, volumeType, volumeSize, snapshotId, deleteOnTermination } = ebs;
                Ebs = {
                    DeleteOnTermination: deleteOnTermination,
                    Iops: iops,
                    SnapshotId: snapshotId,
                    VolumeSize: volumeSize,
                    VolumeType: volumeType,
                    // encrypted is not exposed as part of ebsDeviceProps so we need to access it via [].
                    // eslint-disable-next-line dot-notation
                    Encrypted: 'encrypted' in ebs ? ebs['encrypted'] : undefined,
                };
            }
            return {
                DeviceName: deviceName,
                Ebs,
                VirtualName: virtualName,
            };
        });
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "0.30.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,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