"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const mount_permissions_helper_1 = require("../lib/mount-permissions-helper");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableEFS', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let instance;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new efs.FileSystem(stack, 'EFS', { vpc });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the EFS's security group
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 2049,
            ToPort: 2049,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'InstanceInstanceSecurityGroupF0E2D5BE',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'EFSEfsSecurityGroup56F189CE',
                    'GroupId',
                ],
            },
        }));
        // Make sure we download the mountEfs script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/efs/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false r')));
    });
    describe.each([
        [undefined],
        [lib_1.MountPermissions.READONLY],
        [lib_1.MountPermissions.READWRITE],
    ])('access point with %s access permissions', (mountPermission) => {
        describe.each([
            [
                'unspecified POSIX user',
                {
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved non-root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '1000' },
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '0' },
                    expectedClientRootAccess: true,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '0', gid: '1000' },
                    expectedClientRootAccess: true,
                },
            ],
        ])('%s', (_name, testCase) => {
            // GIVEN
            const { posixUser, expectedClientRootAccess } = testCase;
            const expectedActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mountPermission);
            if (expectedClientRootAccess) {
                expectedActions.push('elasticfilesystem:ClientRootAccess');
            }
            const mountPath = '/mnt/efs/fs1';
            let userData;
            let accessPoint;
            let expectedMountMode;
            beforeEach(() => {
                // GIVEN
                accessPoint = new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser,
                });
                const mount = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                });
                expectedMountMode = (mountPermission === lib_1.MountPermissions.READONLY) ? 'ro' : 'rw';
                // WHEN
                mount.mountToLinuxInstance(instance, {
                    location: mountPath,
                    permissions: mountPermission,
                });
                userData = stack.resolve(instance.userData.render());
            });
            test('userdata specifies access point when mounting', () => {
                // THEN
                expect(userData).toEqual({
                    'Fn::Join': [
                        '',
                        expect.arrayContaining([
                            expect.stringMatching(new RegExp('(\\n|^)bash \\./mountEfs.sh $')),
                            stack.resolve(efsFS.fileSystemId),
                            ` ${mountPath} false ${expectedMountMode},iam,accesspoint=`,
                            stack.resolve(accessPoint.accessPointId),
                            expect.stringMatching(/^\n/),
                        ]),
                    ],
                });
            });
            test('grants IAM access point permissions', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: expectedActions.length === 1 ? expectedActions[0] : expectedActions,
                            Condition: {
                                StringEquals: {
                                    'elasticfilesystem:AccessPointArn': stack.resolve(accessPoint.accessPointArn),
                                },
                            },
                            Effect: 'Allow',
                            Resource: stack.resolve(efsFS.node.defaultChild.attrArn),
                        }),
                        Version: '2012-10-17',
                    }),
                    Roles: assert_1.arrayWith(
                    // The Policy construct micro-optimizes the reference to a role in the same stack using its logical ID
                    stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw,option1,option2')));
    });
    test('asset is singleton', () => {
        var _a;
        // GIVEN
        const mount1 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        const mount2 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = (_a = userData.match(regex)) !== null && _a !== void 0 ? _a : [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    describe('resolves mount target using API', () => {
        describe.each([
            ['with access point', () => {
                    return new efs.AccessPoint(stack, 'AccessPoint', {
                        fileSystem: efsFS,
                        posixUser: {
                            gid: '1',
                            uid: '1',
                        },
                    });
                }],
            ['without access point', () => undefined],
        ])('%s', (_, getAccessPoint) => {
            let accessPoint;
            beforeEach(() => {
                // GIVEN
                accessPoint = getAccessPoint();
                const mountable = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                    resolveMountTargetDnsWithApi: true,
                });
                // WHEN
                mountable.mountToLinuxInstance(instance, {
                    location: '/mnt/efs',
                });
            });
            test('grants DescribeMountTargets permission', () => {
                const expectedResources = [
                    stack.resolve(efsFS.node.defaultChild.attrArn),
                ];
                if (accessPoint) {
                    expectedResources.push(stack.resolve(accessPoint === null || accessPoint === void 0 ? void 0 : accessPoint.accessPointArn));
                }
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: 'elasticfilesystem:DescribeMountTargets',
                            Effect: 'Allow',
                            Resource: expectedResources.length == 1 ? expectedResources[0] : expectedResources,
                        }),
                    }),
                    Roles: assert_1.arrayWith(stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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