"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const tag_helpers_1 = require("./tag-helpers");
let app;
let infraStack;
let hmStack;
let wfStack;
let vpc;
let healthMonitor;
class TestMonitorableFleet extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            minCapacity: props.minCapacity,
        });
        this.connections = new aws_ec2_1.Connections();
        this.targetCapacity = parseInt(fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [fleet.autoScalingGroupArn],
                })],
        });
    }
}
describe('HealthMonitor', () => {
    beforeEach(() => {
        app = new core_1.App();
        infraStack = new core_1.Stack(app, 'infraStack');
        hmStack = new core_1.Stack(app, 'hmStack');
        wfStack = new core_1.Stack(app, 'wfStack');
        vpc = new aws_ec2_1.Vpc(infraStack, 'VPC');
    });
    test('validating default health monitor properties', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: 'kms:*',
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:GenerateDataKey',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: '*',
                    },
                ],
            },
            Description: `This key is used to encrypt SNS messages for ${core_1.Names.uniqueId(healthMonitor)}.`,
            EnableKeyRotation: true,
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: {
                            Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                        },
                        Sid: '0',
                    },
                ],
            },
            Topics: [
                {
                    Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                },
            ],
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: {
                'Fn::GetAtt': [
                    `${hmStack.getLogicalId(healthMonitor.node.findChild('SNSEncryptionKey').node.defaultChild)}`,
                    'Arn',
                ],
            },
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating health monitor properties while passing a key', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
            encryptionKey: aws_kms_1.Key.fromKeyArn(hmStack, 'importedKey', 'arn:aws:kms:us-west-2:123456789012:key/testarn'),
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::KMS::Key'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: 'arn:aws:kms:us-west-2:123456789012:key/testarn',
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating the target with default health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(hmStack).notTo((assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: assert_1.arrayWith(assert_1.deepObjectLike({
                CidrIp: '0.0.0.0/0',
                FromPort: 8081,
                IpProtocol: 'tcp',
                ToPort: 8081,
            })),
        })));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '8081',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('validating the target with custom health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {
            port: 7171,
        });
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('2 ASG gets registered to same LB', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, { port: 7171 });
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, { port: 7171 });
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 8,
            ActionsEnabled: true,
            DatapointsToAlarm: 8,
            Threshold: 0,
            TreatMissingData: 'notBreaching',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            ActionsEnabled: true,
            DatapointsToAlarm: 1,
            Threshold: 35,
            TreatMissingData: 'notBreaching',
        }));
    });
    test('validating LB target limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 50,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating LB listener limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'listeners-per-application-load-balancer',
                    max: 1,
                }, {
                    name: 'target-groups-per-action-on-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target group limit per lb', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'target-groups-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target limit exhaustion', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 2,
        });
        expect(() => {
            healthMonitor.registerFleet(fleet, {});
        }).toThrowError(/AWS service limit \"targets-per-application-load-balancer\" reached. Limit: 1/);
    });
    test('validating deletion protection', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            deletionProtection: false,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    test('specifying a subnet', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        // Make sure it has the public subnets
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assert_1.stringLike('*PublicSubnet*') },
                { 'Fn::ImportValue': assert_1.stringLike('*PublicSubnet*') },
            ],
        }));
        // Make sure the private subnets aren't present
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assert_1.notMatching(assert_1.stringLike('*PrivateSubnet*')) },
                { 'Fn::ImportValue': assert_1.notMatching(assert_1.stringLike('*PrivateSubnet*')) },
            ],
        }));
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'HealthMonitor',
            createConstruct: () => {
                // GIVEN
                const fleetStack = new core_1.Stack(app, 'FleetStack');
                const fleet = new TestMonitorableFleet(fleetStack, 'workerFleet', {
                    vpc,
                });
                // WHEN
                healthMonitor = new lib_1.HealthMonitor(hmStack, 'HealthMonitor', {
                    vpc,
                });
                healthMonitor.registerFleet(fleet, {});
                return hmStack;
            },
            resourceTypeCounts: {
                'AWS::KMS::Key': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::EC2::SecurityGroup': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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