"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 *
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Elastic Network Interface (ENI).
 * - Security Group for the ASG.
 * - Instance Role and corresponding IAM Policy.
 * - SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack.
 * - Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is deployed through this construct has broad IAM permissions to attach any Elastic
 *    Network Interface (ENI) to any instance. You should not grant any additional actors/principals the ability
 *    to modify or execute this Lambda.
 * - The SNS Topic that is deployed through this construct controls the execution of the Lambda discussed above.
 *    Principals that can publish messages to this SNS Topic will be able to trigger the Lambda to run. You should
 *    not allow any additional principals to publish messages to this SNS Topic.
 *
 * @stability stable
 */
class StaticPrivateIpServer extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            resourceSignalCount: props.resourceSignalTimeout ? 1 : undefined,
            resourceSignalTimeout: props.resourceSignalTimeout,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: core_1.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance in this construct's AutoScalingGroup when a new instance is launched.
     *
     * @stability stable
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     *
     * @stability stable
     */
    setupLifecycleEventHandlerFunction() {
        const stack = core_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                description: `Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = core_1.Names.uniqueId(eventHandler);
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        core_1.Tags.of(this.autoscalingGroup).add(tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPrincipalPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPrincipalPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events.
     *
     * Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     *
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns : Topic, role: Role }
     * @stability stable
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = core_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For RFDK instance-launch notifications for stack '${stack.stackName}'`,
            });
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
_a = JSII_RTTI_SYMBOL_1;
StaticPrivateIpServer[_a] = { fqn: "aws-rfdk.StaticPrivateIpServer", version: "0.30.0" };
//# sourceMappingURL=data:application/json;base64,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