"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableBlockVolume = exports.BlockVolumeFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * Block format options for formatting a blank/new BlockVolume.
 *
 * @stability stable
 */
var BlockVolumeFormat;
(function (BlockVolumeFormat) {
    BlockVolumeFormat["EXT3"] = "ext3";
    BlockVolumeFormat["EXT4"] = "ext4";
    BlockVolumeFormat["XFS"] = "xfs";
})(BlockVolumeFormat = exports.BlockVolumeFormat || (exports.BlockVolumeFormat = {}));
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an Amazon Elastic Block Store (EBS) Volume to itself when it is launched.
 *
 * The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *    **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
class MountableBlockVolume {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        var _b;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        this.grantRequiredPermissions(target);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScriptZip = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const volumeFormat = (_b = this.props.volumeFormat) !== null && _b !== void 0 ? _b : BlockVolumeFormat.XFS;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScriptZip}`, `bash ./mountEbsBlockVolume.sh ${this.props.blockVolume.volumeId} ${volumeFormat} ${mountDir} ${mountOptionsStr} ""`, 'popd', `rm -f ${mountScriptZip}`);
    }
    /**
     * Grant required permissions to the target.
     *
     * The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     *
     * @stability stable
     */
    grantRequiredPermissions(target) {
        // Volume.grantAttachVolumeByResourceTag() requires that the target be a construct; it adds a tag to the construct.
        // So, we fail if we're given something that is not compatible.
        if (!core_1.Construct.isConstruct(target)) {
            throw new Error('Target instance must be a construct. It cannot be constructed from attributes.');
        }
        // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
        // Accessed July 2020
        // ec2:DescribeVolumes does not support resource or condition constraints.
        target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ec2:DescribeVolumes',
            ],
            resources: ['*'],
        }));
        this.props.blockVolume.grantAttachVolumeByResourceTag(target.grantPrincipal, [target]);
    }
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton(scope) {
        var _b;
        const stack = core_1.Stack.of(scope);
        const uuid = '01ca4aa6-d440-4f83-84d8-80a5a21fd0e3';
        const uniqueId = 'MountableBlockVolumeAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEbsBlockVolume.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableBlockVolume = MountableBlockVolume;
_a = JSII_RTTI_SYMBOL_1;
MountableBlockVolume[_a] = { fqn: "aws-rfdk.MountableBlockVolume", version: "0.30.0" };
//# sourceMappingURL=data:application/json;base64,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