#!/usr/bin/env python
# -*- coding: utf-8 -*-
# (c) Copyright IBM Corp. 2010, 2020. All Rights Reserved.

""" Implemention of 'resilient-sdk docgen' """

import logging
import os
import re
import shutil

from resilient_sdk.cmds.base_cmd import BaseCmd
from resilient_sdk.util import constants
from resilient_sdk.util import package_file_helpers as package_helpers
from resilient_sdk.util import sdk_helpers
from resilient_sdk.util.resilient_objects import (IGNORED_INCIDENT_FIELDS,
                                                  ResilientObjMap)
from resilient_sdk.util.sdk_exception import SDKException

from resilient import ensure_unicode

# Get the same logger object that is used in app.py
LOG = logging.getLogger(constants.LOGGER_NAME)

# JINJA Constants
README_TEMPLATE_NAME = "README.md.jinja2"


class CmdDocgen(BaseCmd):
    """
    Create a README.md for the specified app. Reads all details from
    the ImportDefinition in the customize.py file. Creates a backup of
    of the README.md if one exists already. The README.md is
    really an 'inventory' of what the app contains and details for
    the app configs
    """

    CMD_NAME = "docgen"
    CMD_HELP = "Generates boilerplate documentation for an app."
    CMD_USAGE = """
    $ resilient-sdk docgen -p <path_to_package>
    $ resilient-sdk docgen -p <name_of_package> --settings <path_to_custom_sdk_settings_file>"""
    CMD_DESCRIPTION = CMD_HELP
    CMD_ADD_PARSERS = [constants.SDK_SETTINGS_PARSER_NAME]

    def setup(self):
        # Define docgen usage and description
        self.parser.usage = self.CMD_USAGE
        self.parser.description = self.CMD_DESCRIPTION

        # Add any positional or optional arguments here
        self.parser.add_argument("-p",
                                 type=ensure_unicode,
                                 help="Path to the package containing the setup.py file",
                                 nargs="?",
                                 default=os.getcwd())

        self.parser.add_argument("-pr", "--poller",
                                 action="store_true",
                                 help="Include poller section in README generated by docgen")

    @staticmethod
    def _get_fn_input_details(function):
        """Return a List of all Function Inputs which are Dictionaries with
        the attributes: api_name, name, type, required, placeholder and tooltip"""

        fn_inputs = []

        for i in function.get("inputs", []):
            the_input = {}

            the_input["api_name"] = i.get("name")
            the_input["name"] = i.get("text")
            the_input["type"] = i.get("input_type")
            the_input["required"] = "Yes" if "always" in i.get("required", "") else "No"
            the_input["placeholder"] = i.get("placeholder") if i.get("placeholder") else "-"
            the_input["tooltip"] = i.get("tooltip") if i.get("tooltip") else "-"

            fn_inputs.append(the_input)

        fn_inputs = sorted(fn_inputs, key=lambda i: i["api_name"])

        return fn_inputs

    @classmethod
    def _get_function_details(cls, export):
        """
        Return a List of Functions which are Dictionaries with
        the attributes: name, simple_name, anchor, description, uuid, inputs,
        workflows, pre_processing_script, post_processing_script.

        The scripts are looked for first in playbooks as since v50
        that is the preferred way for apps to use functions now.
        If playbook scripts associated with the function are not found,
        they are searched for in workflows.
        """

        return_list = []

        functions = export.get("functions")
        workflows = export.get("workflows")
        playbooks = export.get("playbooks")
        scripts = export.get("scripts")

        for fn in functions:
            the_function = {}

            the_function["name"] = fn.get("display_name")
            the_function["simple_name"] = sdk_helpers.simplify_string(the_function.get("name"))
            the_function["anchor"] = sdk_helpers.generate_anchor(the_function.get("name"))
            the_function["description"] = fn.get("description")["content"]
            the_function["uuid"] = fn.get("uuid")
            the_function["inputs"] = cls._get_fn_input_details(fn)
            the_function["message_destination"] = fn.get("destination_handle", "")
            the_function["workflows"] = fn.get("workflows", [])
            the_function["x_api_name"] = fn.get("x_api_name", "")

            # look for pre/post scripts in playbooks first
            pre_script, post_script = cls._get_pre_and_post_processing_scripts_from_playbooks(the_function, playbooks, scripts, export)
            # if not found in playbooks, then look in workflows
            if not pre_script and not post_script:
                pre_script, post_script = cls._get_pre_and_post_processing_scripts_from_workflows(the_function, workflows)

            # save the scripts for jinja to use later
            the_function["pre_processing_script"] = pre_script
            the_function["post_processing_script"] = post_script

            return_list.append(the_function)

        return return_list

    @staticmethod
    def _get_pre_and_post_processing_scripts_from_playbooks(the_function, playbooks, scripts, export):
        """
        Look through all playbooks in the export, searching for any that
        use the function in question. If any do, if the function instance
        uses a pre-processing script, use that as the pre-processing script.
        If any scripts in the playbook reference the output of the function
        in their script (we do a very specific search which looks for Python
        code (not comments) that reference the specific string 
        ``playbook.functions.results.<output_name>``), then use that script
        as an example post-processing script. If none are found in local
        scripts, search the global scripts of the export

        :param the_function: the function in question for which we want example scripts
        :type the_function: dict
        :param playbooks: list of playbooks in the export
        :type playbooks: list[dict]
        :param playbooks: list of global scripts in the export
        :type playbooks: list[dict]
        :return: pre and post processing scripts as text
        :rtype: tuple(str, str)
        """
        pre_script, post_script = None, None

        for playbook in playbooks:
            # This gets all the functions and scripts in the Playbooks's XML
            pb_objects = sdk_helpers.get_playbook_objects(playbook)

            # loop through the playbook to find its functions
            for pb_fn in pb_objects.get("functions", []):
                # the best proxy we have for "post processing" script is
                # if the output name of the function is used in the script's
                # code. we check for that there and if so, we've found a
                # "post processing" script

                # This regex searches for Python code instances of the 
                # output. It is relatively robust and will not match commented code
                # nor non-exact matches
                # For examples on this regex, see: https://regex101.com/r/Fv5AQm
                regex_str = r"^(?!#).*playbook\.functions\.results\.{0}\b.*".format(pb_fn.get("result_name"))
                regex_compiled = re.compile(regex_str, re.MULTILINE)


                # if the pb_fn matches our goal fn:
                if pb_fn.get("uuid", "uuid_not_found_pb") == the_function.get("uuid", "uuid_not_found_fn"):
                    if not pre_script: # use the first pre-script found
                        pre_script = pb_fn["pre_processing_script"]

                    # loop through local scripts looking for a match
                    for pb_sc in pb_objects.get("scripts", []):
                        script_is_found = sdk_helpers.get_script_info(pb_sc, playbook.get("local_scripts"), sdk_helpers.SCRIPT_TYPE_MAP.get("local"))

                        if not post_script and script_is_found and regex_compiled.search(pb_sc.get("script_text")) is not None:
                            post_script = pb_sc.get("script_text")

                        # if we didn't find a local script, we can run the same search on
                        # global scripts and we might get a hit
                        if not script_is_found:
                            for g_sc in sdk_helpers.get_res_obj("scripts", "uuid", "Script", [pb_sc.get("uuid")], export):
                                if not post_script and regex_compiled.search(g_sc.get("script_text")) is not None:
                                    post_script = g_sc.get("script_text")

                    # short the searching if we've already found both
                    if pre_script and post_script:
                        return pre_script, post_script

        return pre_script, post_script

    @staticmethod
    def _get_pre_and_post_processing_scripts_from_workflows(the_function, workflows):
        """
        Get pre and post processing scripts from workflows.
        Search the list of workflows for a match with the function in question.
        If found, return the pre and post processing scripts. Note that it will
        only "short-circuit" return if both are found. So that could mean that
        if a function is used twice in an export, but only has a post processing
        script in the second workflow, docgen would take the pre processing
        script of the first instance and the post script of the second.

        :param the_function: the function in question for which we want example scripts
        :type the_function: dict
        :param workflows: list of workflows in the export
        :type workflows: list[dict]
        :return: pre and post processing scripts as text
        :rtype: tuple(str, str)
        """
        pre_script, post_script = None, None

        # Loop the Function's associated Workflows
        for fn_wf in the_function.get("workflows"):

            fn_wf_name = fn_wf.get(ResilientObjMap.WORKFLOWS)

            # Loop all Workflow Objects
            for wf in workflows:

                # Find a match
                if fn_wf_name == wf.get(ResilientObjMap.WORKFLOWS):

                    # Get List of Function details from Workflow XML
                    workflow_functions = sdk_helpers.get_workflow_functions(wf, the_function.get("uuid"))

                    # Get a valid pre and post process script, then break
                    for a_fn in workflow_functions:

                        if not pre_script:
                            pre_script = a_fn.get("pre_processing_script")

                        if not post_script:
                            post_script = a_fn.get("post_processing_script")

                        if pre_script and post_script:
                            return pre_script, post_script

        # return one or the other (or both None) if not both were found
        # here to maintain same functionality that was there before this was moved out. see PR #643
        return pre_script, post_script

    @staticmethod
    def _get_script_details(scripts):
        """Return a List of all Scripts which are Dictionaries with
        the attributes: name, simple_name, description, object_type, script_text"""

        return_list = []

        for s in scripts:
            the_script = {}

            the_script["name"] = s.get("name")
            the_script["simple_name"] = sdk_helpers.simplify_string(the_script.get("name"))
            the_script["anchor"] = sdk_helpers.generate_anchor(the_script.get("name"))
            the_script["description"] = s.get("description")
            the_script["object_type"] = s.get("object_type")
            the_script["script_text"] = s.get("script_text")
            return_list.append(the_script)

        return return_list

    @staticmethod
    def _get_rule_details(rules):
        """Return a List of all Rules which are Dictionaries with
        the attributes: name, simple_name, object_type and workflow_triggered"""

        return_list = []

        for rule in rules:
            the_rule = {}

            the_rule["name"] = rule.get("name")
            the_rule["simple_name"] = sdk_helpers.simplify_string(the_rule.get("name"))
            the_rule["object_type"] = rule.get("object_type", "")

            rule_workflows = rule.get("workflows", [])
            the_rule["workflow_triggered"] = rule_workflows[0] if rule_workflows else "-"

            the_rule["conditions"] = sdk_helpers.str_repr_activation_conditions(rule) or "-"

            return_list.append(the_rule)

        return return_list

    @staticmethod
    def _get_playbook_details(playbooks):
        """Return a List of all Playbooks which are Dictionaries with
        the attributes: api_name, name, object_type, status and description"""

        return_list = []

        for playbook in playbooks:
            the_playbook = {}

            the_playbook["api_name"] = playbook.get("x_api_name")
            the_playbook["name"] = playbook.get("display_name")
            the_playbook["object_type"] = playbook.get("object_type", "")
            the_playbook["status"] = playbook.get("status", "")
            the_playbook["description"] = playbook.get("description", {}).get("content", "")
            the_playbook["activation_type"] = playbook.get("activation_type", "")
            the_playbook["conditions"] = playbook.get("conditions", "")

            return_list.append(the_playbook)

        return return_list

    @staticmethod
    def _get_datatable_details(datatables):
        """Return a List of all Data Tables which are Dictionaries with
        the attributes: name, anchor, api_name, and columns. Columns is
        also a List of Dictionaries that has the attributes: name,
        api_name, type and tooltip"""

        return_list = []

        for datatable in datatables:
            the_dt = {}

            the_dt["name"] = datatable.get("display_name")
            the_dt["simple_name"] = sdk_helpers.simplify_string(the_dt.get("name"))
            the_dt["anchor"] = sdk_helpers.generate_anchor(the_dt.get("name"))
            the_dt["api_name"] = datatable.get("type_name")

            the_dt_columns = []

            # datatable.fields is a Dict where its values (the columns) also Dicts
            for col in datatable.get("fields", {}).values():
                the_col = {}

                the_col["name"] = col.get("text")
                the_col["api_name"] = col.get("name")
                the_col["type"] = col.get("input_type")
                the_col["tooltip"] = col.get("tooltip") if col.get("tooltip") else "-"

                the_dt_columns.append(the_col)

            the_dt["columns"] = sorted(the_dt_columns, key=lambda c: c["api_name"])

            return_list.append(the_dt)

        return return_list

    @staticmethod
    def _get_custom_fields_details(fields):
        """Return a List of all Custom Incident Fields which are Dictionaries with
        the attributes: api_name, label, type, prefix, placeholder and tooltip"""
        return_list = []

        for field in fields:
            the_field = {}

            the_field["api_name"] = field.get("name")
            the_field["label"] = field.get("text")
            the_field["type"] = field.get("input_type")
            the_field["prefix"] = field.get("prefix")
            the_field["placeholder"] = field.get("placeholder") if field.get("placeholder") else "-"
            the_field["tooltip"] = field.get("tooltip") if field.get("tooltip") else "-"

            return_list.append(the_field)

        return return_list

    @staticmethod
    def _get_custom_artifact_details(custom_artifact_types):
        """Return a List of all Custom Incident Artifact Types which are Dictionaries with
        the attributes: api_name, display_name and description"""
        return_list = []

        for artifact_type in custom_artifact_types:
            the_artifact_type = {}

            the_artifact_type["api_name"] = artifact_type.get("programmatic_name")
            the_artifact_type["display_name"] = artifact_type.get("name")
            the_artifact_type["description"] = artifact_type.get("desc")

            return_list.append(the_artifact_type)

        return return_list

    @staticmethod
    def _get_poller_details(path_to_src, package_name):
        """Return the contents of each template file for a poller"""
        file_names = [
            package_helpers.BASE_NAME_POLLER_CREATE_CASE_TEMPLATE, 
            package_helpers.BASE_NAME_POLLER_UPDATE_CASE_TEMPLATE, 
            package_helpers.BASE_NAME_POLLER_CLOSE_CASE_TEMPLATE
        ]

        contents = {}

        for f in file_names:
            file_path = os.path.join(path_to_src, package_name, "poller", "data", f)
            try:
                sdk_helpers.validate_file_paths(os.R_OK, file_path)
                content = "".join(sdk_helpers.read_file(file_path))
                contents[f] = content
            except SDKException as err:
                sdk_helpers.handle_file_not_found_error(err,
                    u"Error getting poller template contents. No '{0}' file found.".format(f))

        return contents

    def execute_command(self, args):
        LOG.debug("docgen called with %s", args)

        # Set docgen name for SDKException
        SDKException.command_ran = self.CMD_NAME

        # Get absolute path_to_src
        path_to_src = os.path.abspath(args.p)

        LOG.debug("Path to project: %s", path_to_src)

        # Instansiate Jinja2 Environment with path to Jinja2 templates
        jinja_env = sdk_helpers.setup_jinja_env("data/docgen/templates")

        # Load the Jinja2 Templates
        readme_template = jinja_env.get_template(README_TEMPLATE_NAME)

        # Generate path to setup.py file
        path_setup_py_file = os.path.join(path_to_src, package_helpers.BASE_NAME_SETUP_PY)

        try:
            # Ensure we have read permissions for setup.py
            sdk_helpers.validate_file_paths(os.R_OK, path_setup_py_file)
        except SDKException as err:
            err.message += "\nEnsure you are in the directory of the package you want to run docgen for"
            raise err
        
        # Validate that the given path to the sdk settings is valid
        try:
            sdk_helpers.validate_file_paths(os.R_OK, args.settings)
            # Parse the sdk_settings.json file
            settings_file_contents = sdk_helpers.read_json_file(args.settings, "docgen")
        except SDKException as err:
            args.settings = None
            settings_file_contents = {}
            LOG.debug("Given path to SDK Settings is either not valid or not readable. Ignoring and using built-in values for docgen")

        # Parse the setup.py file
        setup_py_attributes = package_helpers.parse_setup_py(path_setup_py_file, package_helpers.SUPPORTED_SETUP_PY_ATTRIBUTE_NAMES)

        package_name = setup_py_attributes.get("name", "")

        # Generate paths to other required directories + files
        path_customize_py_file = os.path.join(path_to_src, package_name, package_helpers.PATH_CUSTOMIZE_PY)
        path_config_py_file = os.path.join(path_to_src, package_name, package_helpers.PATH_CONFIG_PY)
        path_readme = os.path.join(path_to_src, package_helpers.BASE_NAME_README)
        path_screenshots_dir = os.path.join(path_to_src, package_helpers.PATH_SCREENSHOTS)
        path_payload_samples_dir = os.path.join(path_to_src, package_helpers.BASE_NAME_PAYLOAD_SAMPLES_DIR)

        # Ensure we have read permissions for each required file and the file exists
        sdk_helpers.validate_file_paths(os.R_OK, path_setup_py_file, path_customize_py_file, path_config_py_file)

        # Check doc/screenshots directory exists, if not, create it + copy default screenshot
        if not os.path.isdir(path_screenshots_dir):
            os.makedirs(path_screenshots_dir)
            shutil.copy(package_helpers.PATH_DEFAULT_SCREENSHOT, path_screenshots_dir)

        # Get the resilient_circuits dependency string from setup.py file
        res_circuits_dep_str = package_helpers.get_dependency_from_install_requires(
                setup_py_attributes.get(constants.SETUP_PY_INSTALL_REQ_NAME), constants.CIRCUITS_PACKAGE_NAME)

        # Get ImportDefinition from customize.py
        customize_py_import_def = package_helpers.get_import_definition_from_customize_py(path_customize_py_file)

        # Parse the app.configs from the config.py file
        jinja_app_configs = package_helpers.get_configs_from_config_py(path_config_py_file)

        # Get field names from ImportDefinition
        field_names = []
        for f in customize_py_import_def.get("fields", []):
            f_export_key = f.get("export_key")

            if "incident/" in f_export_key and f_export_key not in IGNORED_INCIDENT_FIELDS:
                field_names.append(f.get(ResilientObjMap.FIELDS, ""))

        # Get data from ImportDefinition
        import_def_data = sdk_helpers.get_from_export(customize_py_import_def,
                                                      message_destinations=sdk_helpers.get_object_api_names(ResilientObjMap.MESSAGE_DESTINATIONS, customize_py_import_def.get("message_destinations")),
                                                      functions=sdk_helpers.get_object_api_names(ResilientObjMap.FUNCTIONS, customize_py_import_def.get("functions")),
                                                      workflows=sdk_helpers.get_object_api_names(ResilientObjMap.WORKFLOWS, customize_py_import_def.get("workflows")),
                                                      rules=sdk_helpers.get_object_api_names(ResilientObjMap.RULES, customize_py_import_def.get("actions")),
                                                      fields=field_names,
                                                      artifact_types=sdk_helpers.get_object_api_names(ResilientObjMap.INCIDENT_ARTIFACT_TYPES, customize_py_import_def.get("incident_artifact_types")),
                                                      datatables=sdk_helpers.get_object_api_names(ResilientObjMap.DATATABLES, customize_py_import_def.get("types")),
                                                      tasks=sdk_helpers.get_object_api_names(ResilientObjMap.TASKS, customize_py_import_def.get("automatic_tasks")),
                                                      scripts=sdk_helpers.get_object_api_names(ResilientObjMap.SCRIPTS, customize_py_import_def.get("scripts")),
                                                      playbooks=sdk_helpers.get_object_api_names(ResilientObjMap.PLAYBOOKS, customize_py_import_def.get("playbooks", [])))

        # Lists we use in Jinja Templates
        jinja_functions = self._get_function_details(import_def_data)
        jinja_scripts = self._get_script_details(import_def_data.get("scripts", []))
        jinja_rules = self._get_rule_details(import_def_data.get("rules", []))
        jinja_datatables = self._get_datatable_details(import_def_data.get("datatables", []))
        jinja_custom_fields = self._get_custom_fields_details(import_def_data.get("fields", []))
        jinja_custom_artifact_types = self._get_custom_artifact_details(import_def_data.get("artifact_types", []))
        jinja_playbooks = self._get_playbook_details(import_def_data.get("playbooks", []))

        # Other variables for Jinja Templates
        package_name_dash = package_name.replace("_", "-")
        server_version = customize_py_import_def.get("server_version", {})
        supported_app = settings_file_contents.get("supported_app", 
                                    sdk_helpers.does_url_contain(setup_py_attributes.get("url", ""), "ibm.com/mysupport"))

        # See if a payload_samples dir exists and use the contents for function results
        try:
            sdk_helpers.validate_dir_paths(os.R_OK, path_payload_samples_dir)

            for f in jinja_functions:
                fn_name = f.get("x_api_name")
                path_payload_samples_fn_name = os.path.join(path_payload_samples_dir, fn_name)
                path_output_json_example = os.path.join(path_payload_samples_fn_name, package_helpers.BASE_NAME_PAYLOAD_SAMPLES_EXAMPLE)

                try:
                    sdk_helpers.validate_file_paths(os.R_OK, path_output_json_example)
                    f["results"] = sdk_helpers.read_json_file(path_output_json_example)
                except SDKException as e:
                    sdk_helpers.handle_file_not_found_error(e, u"Error getting results. No '{0}' file found for '{1}'.".format(
                        package_helpers.BASE_NAME_PAYLOAD_SAMPLES_EXAMPLE, fn_name))

        except SDKException as e:
            sdk_helpers.handle_file_not_found_error(e, u"Error getting results. No '{0}' directory found.".format(
                package_helpers.BASE_NAME_PAYLOAD_SAMPLES_EXAMPLE))

        # If poller flag was given try to find the template details
        poller_templates = {}
        if args.poller:
            poller_templates = self._get_poller_details(path_to_src, package_name)

        LOG.info("Rendering README for %s", package_name_dash)

        # Render the README Jinja2 Templeate with parameters
        rendered_readme = readme_template.render({
            "name_underscore": package_name,
            "name_dash": package_name_dash,
            "display_name": setup_py_attributes.get("display_name", package_name),
            "short_description": setup_py_attributes.get("description"),
            "long_description": setup_py_attributes.get("long_description"),
            "version": setup_py_attributes.get("version"),
            "server_version": server_version.get("version"),
            "all_dependencies": setup_py_attributes.get("install_requires", []),
            "res_circuits_dependency_str": res_circuits_dep_str,
            "author": setup_py_attributes.get("author"),
            "support_url": setup_py_attributes.get("url"),
            "supported_app": supported_app,
            "app_configs": jinja_app_configs[1],
            "functions": jinja_functions,
            "scripts": jinja_scripts,
            "rules": jinja_rules,
            "datatables": jinja_datatables,
            "custom_fields": jinja_custom_fields,
            "custom_artifact_types": jinja_custom_artifact_types,
            "playbooks": jinja_playbooks,
            "placeholder_string": constants.DOCGEN_PLACEHOLDER_STRING,
            "poller_flag": args.poller,
            "poller_templates": poller_templates,
            "sdk_version": sdk_helpers.get_resilient_sdk_version()
        })

        # Create a backup if needed of README
        sdk_helpers.rename_to_bak_file(path_readme, package_helpers.PATH_DEFAULT_README)

        LOG.info("Writing README to: %s", path_readme)

        # Write the new README
        sdk_helpers.write_file(path_readme, rendered_readme)
