"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateGraph = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A collection of connected states.
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *    be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *    machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 *
 * @stability stable
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched.
     * @param graphDescription description of the state machine.
     * @stability stable
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements.
         *
         * @stability stable
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(startState);
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph.
     *
     * Called by State.bindToGraph().
     *
     * @stability stable
     */
    registerState(state) {
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(state);
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph.
     *
     * @stability stable
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph.
     *
     * Resource changes will be bubbled up to the given graph.
     *
     * @stability stable
     */
    registerSuperGraph(graph) {
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(graph);
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph.
     *
     * @stability stable
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * Return a string description of this graph.
     *
     * @stability stable
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
_a = JSII_RTTI_SYMBOL_1;
StateGraph[_a] = { fqn: "@aws-cdk/aws-stepfunctions.StateGraph", version: "1.134.0" };
//# sourceMappingURL=data:application/json;base64,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