#!/usr/bin/env python3

# Copyright (c) Facebook, Inc. and its affiliates.
#
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

"""Learning rate policies."""

import numpy as np

# from pycls.core.config import cfg


def lr_fun_steps(cfg, cur_epoch):
    """Steps schedule (cfg.OPTIM.LR_POLICY = 'steps')."""
    ind = [i for i, s in enumerate(cfg.OPTIM.STEPS) if cur_epoch >= s][-1]
    return cfg.OPTIM.BASE_LR * (cfg.OPTIM.LR_MULT ** ind)


def lr_fun_exp(cfg, cur_epoch):
    """Exponential schedule (cfg.OPTIM.LR_POLICY = 'exp')."""
    return cfg.OPTIM.BASE_LR * (cfg.OPTIM.GAMMA ** cur_epoch)


def lr_fun_cos(cfg, cur_epoch):
    """Cosine schedule (cfg.OPTIM.LR_POLICY = 'cos')."""
    base_lr, max_epoch = cfg.OPTIM.BASE_LR, cfg.OPTIM.MAX_EPOCH
    return 0.5 * base_lr * (1.0 + np.cos(np.pi * cur_epoch / max_epoch))


def get_lr_fun(cfg):
    """Retrieves the specified lr policy function"""
    lr_fun = 'lr_fun_' + cfg.OPTIM.LR_POLICY
    if lr_fun not in globals():
        raise NotImplementedError('Unknown LR policy:' + cfg.OPTIM.LR_POLICY)
    return globals()[lr_fun]


def get_epoch_lr(cfg, cur_epoch):
    """Retrieves the lr for the given epoch according to the policy."""
    ## Ramping up the lr when we are not doing active learning
    if cfg.TRAIN.DATASET == "IMAGENET":
        if not cfg.ACTIVE_LEARNING.ACTIVATE and  cur_epoch < 5:
            # 5 because cur_epoch starts from 0
            return cfg.OPTIM.BASE_LR * ((cur_epoch+1)/5.)

    lr = get_lr_fun(cfg)(cfg, cur_epoch)
    # Linear warmup
    if cur_epoch < cfg.OPTIM.WARMUP_EPOCHS:
        alpha = cur_epoch / cfg.OPTIM.WARMUP_EPOCHS
        warmup_factor = cfg.OPTIM.WARMUP_FACTOR * (1.0 - alpha) + alpha
        lr *= warmup_factor
    return lr
