"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RetryWithJitterTask = void 0;
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const ResilientLambdaTask_1 = require("./ResilientLambdaTask");
class RetryWithJitterTask extends sfn.Parallel {
    constructor(scope, id, props) {
        const parallelProps = {
            comment: props.comment,
            inputPath: props.inputPath,
            outputPath: "$[0]",
            resultPath: props.resultPath,
            parameters: {
                "RetryCount.$": "$$.State.RetryCount",
                "Input.$": "$"
            }
        };
        super(scope, id, parallelProps);
        const calculateJitterLambda = new lambda.Function(this, 'CalculateJitterLambda', {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset('lambda/jitter'),
            handler: "main.lambda_handler",
        });
        const calculateJitterTask = new ResilientLambdaTask_1.ResilientLambdaTask(this, this.createStateName("CalculateJitter"), {
            lambdaFunction: calculateJitterLambda,
            payload: sfn.TaskInput.fromObject({
                "RetryCount.$": "$.RetryCount",
                "Backoff": 2
            }),
            resultPath: "$.WaitSeconds",
        });
        const waitTask = new sfn.Wait(this, this.createStateName("WaitBetweenRetries"), {
            time: sfn.WaitTime.secondsPath("$.WaitSeconds"),
        });
        // Need this state to "unwrap" original input to the state.
        // Also, CDK does not support outputPath for Wait state, which I would use for retry path
        const unwrapArguments = new sfn.Pass(this, this.createStateName('Unwrap Input'), {
            outputPath: "$.Input"
        });
        const retryPath = calculateJitterTask.next(waitTask).next(unwrapArguments);
        const choiceState = new sfn.Choice(this, this.createStateName("CheckRetryCount"))
            .when(sfn.Condition.numberGreaterThan("$.RetryCount", 0), retryPath)
            .otherwise(unwrapArguments)
            .afterwards()
            .next(props.tryProcess);
        this.branch(choiceState);
        this.addRetry({
            interval: cdk.Duration.seconds(0),
            maxAttempts: props.retryProps.maxAttempts,
            errors: props.retryProps.errors
        });
    }
    createStateName(name) {
        return `${name}_${this.node.uniqueId}`;
    }
}
exports.RetryWithJitterTask = RetryWithJitterTask;
//# sourceMappingURL=data:application/json;base64,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