"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack, options = {}) {
    const sourceOutput = new codepipeline.Artifact();
    const sourceAction = new cpactions.GitHubSourceAction({
        actionName: 'Source',
        owner: 'aws',
        repo: 'aws-cdk',
        output: sourceOutput,
        oauthToken: core_1.SecretValue.plainText('secret'),
    });
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [sourceAction],
            },
        ],
    });
    const deploymentStage = pipeline.addStage({
        stageName: 'Deploy',
        actions: [
            new cpactions.S3DeployAction({
                accessControl: options.accessControl,
                actionName: 'CopyFiles',
                bucket: options.bucket || new s3.Bucket(stack, 'MyBucket'),
                cacheControl: options.cacheControl,
                extract: options.extract,
                input: sourceOutput,
                objectKey: options.objectKey,
            }),
        ],
    });
    return deploymentStage;
}
module.exports = {
    'S3 Deploy Action': {
        'by default extract artifacts'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                        'Actions': [
                            {
                                'Name': 'Source',
                                'ActionTypeId': {
                                    'Category': 'Source',
                                    'Owner': 'ThirdParty',
                                },
                            },
                        ],
                    },
                    {
                        'Name': 'Deploy',
                        'Actions': [
                            {
                                'ActionTypeId': {
                                    'Category': 'Deploy',
                                    'Provider': 'S3',
                                },
                                'Configuration': {
                                    'Extract': 'true',
                                },
                                'Name': 'CopyFiles',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'grant the pipeline correct access to the target bucket'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Effect': 'Allow',
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                        },
                        {},
                        {
                            'Effect': 'Allow',
                            'Action': 'sts:AssumeRole',
                        },
                    ],
                },
            }));
            test.done();
        },
        'kebab-case CannedACL value'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                accessControl: s3.BucketAccessControl.PUBLIC_READ_WRITE,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'CannedACL': 'public-read-write',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'allow customizing cache-control'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                cacheControl: [
                    cpactions.CacheControl.setPublic(),
                    cpactions.CacheControl.maxAge(core_1.Duration.hours(12)),
                    cpactions.CacheControl.sMaxAge(core_1.Duration.hours(12)),
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'CacheControl': 'public, max-age=43200, s-maxage=43200',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'allow customizing objectKey (deployment path on S3)'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                objectKey: '/a/b/c',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'ObjectKey': '/a/b/c',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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