"use strict";
const assert_1 = require("@aws-cdk/assert");
const codedeploy = require("@aws-cdk/aws-codedeploy");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
function addEcsDeploymentGroup(stack) {
    return codedeploy.EcsDeploymentGroup.fromEcsDeploymentGroupAttributes(stack, 'EDG', {
        application: codedeploy.EcsApplication.fromEcsApplicationName(stack, 'EA', 'MyApplication'),
        deploymentGroupName: 'MyDeploymentGroup',
    });
}
function addCodeDeployECSCodePipeline(stack, props) {
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub',
                        output: props.taskDefinitionTemplateInput || props.taskDefinitionTemplateFile.artifact,
                        oauthToken: cdk.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk',
                    }),
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub2',
                        output: props.appSpecTemplateInput || props.appSpecTemplateFile.artifact,
                        oauthToken: cdk.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk-2',
                    }),
                ],
            },
            {
                stageName: 'Invoke',
                actions: [
                    new cpactions.CodeDeployEcsDeployAction(props),
                ],
            },
        ],
    });
}
module.exports = {
    'CodeDeploy ECS Deploy Action': {
        'throws an exception if more than 4 container image inputs are provided'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const containerImageInputs = [];
            for (let i = 0; i < 5; i++) {
                containerImageInputs.push({
                    input: artifact,
                });
            }
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    appSpecTemplateInput: artifact,
                    containerImageInputs,
                });
            }, /Action cannot have more than 4 container image inputs, got: 5/);
            test.done();
        },
        'throws an exception if both appspec artifact input and file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const artifactPath = new codepipeline.ArtifactPath(artifact, 'hello');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    appSpecTemplateInput: artifact,
                    appSpecTemplateFile: artifactPath,
                });
            }, /Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if neither appspec artifact input nor file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                });
            }, /Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if both task definition artifact input and file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const artifactPath = new codepipeline.ArtifactPath(artifact, 'hello');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    taskDefinitionTemplateFile: artifactPath,
                    appSpecTemplateInput: artifact,
                });
            }, /Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if neither task definition artifact input nor file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    appSpecTemplateInput: artifact,
                });
            }, /Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action/);
            test.done();
        },
        'defaults task definition and appspec template paths'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            addCodeDeployECSCodePipeline(stack, {
                actionName: 'DeployToECS',
                deploymentGroup,
                taskDefinitionTemplateInput: new codepipeline.Artifact('TaskDefArtifact'),
                appSpecTemplateInput: new codepipeline.Artifact('AppSpecArtifact'),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Configuration: {
                                    ApplicationName: 'MyApplication',
                                    DeploymentGroupName: 'MyDeploymentGroup',
                                    TaskDefinitionTemplateArtifact: 'TaskDefArtifact',
                                    AppSpecTemplateArtifact: 'AppSpecArtifact',
                                    TaskDefinitionTemplatePath: 'taskdef.json',
                                    AppSpecTemplatePath: 'appspec.yaml',
                                },
                                InputArtifacts: [
                                    {
                                        Name: 'TaskDefArtifact',
                                    },
                                    {
                                        Name: 'AppSpecArtifact',
                                    },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'defaults task definition placeholder string'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact1 = new codepipeline.Artifact();
            const artifact2 = new codepipeline.Artifact();
            addCodeDeployECSCodePipeline(stack, {
                actionName: 'DeployToECS',
                deploymentGroup,
                taskDefinitionTemplateFile: artifact1.atPath('task-definition.json'),
                appSpecTemplateFile: artifact2.atPath('appspec-test.yaml'),
                containerImageInputs: [
                    {
                        input: artifact1,
                    },
                    {
                        input: artifact2,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Configuration: {
                                    ApplicationName: 'MyApplication',
                                    DeploymentGroupName: 'MyDeploymentGroup',
                                    TaskDefinitionTemplateArtifact: 'Artifact_Source_GitHub',
                                    AppSpecTemplateArtifact: 'Artifact_Source_GitHub2',
                                    TaskDefinitionTemplatePath: 'task-definition.json',
                                    AppSpecTemplatePath: 'appspec-test.yaml',
                                    Image1ArtifactName: 'Artifact_Source_GitHub',
                                    Image1ContainerName: 'IMAGE',
                                    Image2ArtifactName: 'Artifact_Source_GitHub2',
                                    Image2ContainerName: 'IMAGE',
                                },
                                InputArtifacts: [
                                    {
                                        Name: 'Artifact_Source_GitHub',
                                    },
                                    {
                                        Name: 'Artifact_Source_GitHub2',
                                    },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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