"use strict";
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const cpactions = require("../../lib");
function _assertActionMatches(test, stack, actions, provider, category, configuration) {
    const configurationStr = configuration
        ? `, configuration including ${JSON.stringify(stack.resolve(configuration), null, 2)}`
        : '';
    const actionsStr = JSON.stringify(actions.map(a => ({ owner: a.actionProperties.owner, provider: a.actionProperties.provider,
        category: a.actionProperties.category, configuration: stack.resolve(a.actionConfig.configuration),
    })), null, 2);
    test.ok(_hasAction(stack, actions, provider, category, configuration), `Expected to find an action with provider ${provider}, category ${category}${configurationStr}, but found ${actionsStr}`);
}
function _hasAction(stack, actions, provider, category, configuration) {
    for (const action of actions) {
        if (action.actionProperties.provider !== provider) {
            continue;
        }
        if (action.actionProperties.category !== category) {
            continue;
        }
        if (configuration && !action.actionConfig.configuration) {
            continue;
        }
        if (configuration) {
            for (const key of Object.keys(configuration)) {
                if (!_.isEqual(stack.resolve(action.actionConfig.configuration[key]), stack.resolve(configuration[key]))) {
                    continue;
                }
            }
        }
        return true;
    }
    return false;
}
function _assertPermissionGranted(test, stack, statements, action, resource, conditions) {
    const conditionStr = conditions
        ? ` with condition(s) ${JSON.stringify(stack.resolve(conditions))}`
        : '';
    const resolvedStatements = stack.resolve(statements.map(s => s.toStatementJson()));
    const statementsStr = JSON.stringify(resolvedStatements, null, 2);
    test.ok(_grantsPermission(stack, resolvedStatements, action, resource, conditions), `Expected to find a statement granting ${action} on ${JSON.stringify(stack.resolve(resource))}${conditionStr}, found:\n${statementsStr}`);
}
function _grantsPermission(stack, statements, action, resource, conditions) {
    for (const statement of statements.filter(s => s.Effect === 'Allow')) {
        if (!_isOrContains(stack, statement.Action, action)) {
            continue;
        }
        if (!_isOrContains(stack, statement.Resource, resource)) {
            continue;
        }
        if (conditions && !_isOrContains(stack, statement.Condition, conditions)) {
            continue;
        }
        return true;
    }
    return false;
}
function _isOrContains(stack, entity, value) {
    const resolvedValue = stack.resolve(value);
    const resolvedEntity = stack.resolve(entity);
    if (_.isEqual(resolvedEntity, resolvedValue)) {
        return true;
    }
    if (!Array.isArray(resolvedEntity)) {
        return false;
    }
    for (const tested of entity) {
        if (_.isEqual(tested, resolvedValue)) {
            return true;
        }
    }
    return false;
}
function _stackArn(stackName, scope) {
    return cdk.Stack.of(scope).formatArn({
        service: 'cloudformation',
        resource: 'stack',
        resourceName: `${stackName}/*`,
    });
}
class PipelineDouble extends cdk.Resource {
    constructor(scope, id, { pipelineName, role }) {
        super(scope, id);
        this.pipelineName = pipelineName || 'TestPipeline';
        this.pipelineArn = cdk.Stack.of(this).formatArn({ service: 'codepipeline', resource: 'pipeline', resourceName: this.pipelineName });
        this.role = role;
        this.artifactBucket = new BucketDouble(scope, 'BucketDouble');
    }
    onEvent(_id, _options) {
        throw new Error('Method not implemented.');
    }
    onStateChange(_id, _options) {
        throw new Error('Method not implemented.');
    }
}
class FullAction {
    constructor(actionProperties, actionConfig) {
        this.actionProperties = actionProperties;
        this.actionConfig = actionConfig;
        // empty
    }
}
class StageDouble {
    constructor({ name, pipeline, actions }) {
        this.actions = [];
        this.stageName = name || 'TestStage';
        this.pipeline = pipeline;
        const stageParent = new cdk.Construct(pipeline, this.stageName);
        const fullActions = new Array();
        for (const action of actions) {
            const actionParent = new cdk.Construct(stageParent, action.actionProperties.actionName);
            fullActions.push(new FullAction(action.actionProperties, action.bind(actionParent, this, {
                role: pipeline.role,
                bucket: pipeline.artifactBucket,
            })));
        }
        this.fullActions = fullActions;
    }
    get node() {
        throw new Error('StageDouble is not a real construct');
    }
    addAction(_action) {
        throw new Error('addAction() is not supported on StageDouble');
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not supported on StageDouble');
    }
}
class RoleDouble extends iam.Role {
    constructor(scope, id, props = { assumedBy: new iam.ServicePrincipal('test') }) {
        super(scope, id, props);
        this.statements = new Array();
    }
    addToPolicy(statement) {
        super.addToPolicy(statement);
        this.statements.push(statement);
        return true;
    }
}
class BucketDouble extends s3.Bucket {
    grantRead(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
    grantReadWrite(identity, _objectsKeyPattern = '*') {
        return iam.Grant.drop(identity, '');
    }
}
module.exports = nodeunit.testCase({
    CreateReplaceChangeSet: {
        'works'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Stack');
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            const action = new cpactions.CloudFormationCreateReplaceChangeSetAction({
                actionName: 'Action',
                changeSetName: 'MyChangeSet',
                stackName: 'MyStack',
                templatePath: artifact.atPath('path/to/file'),
                adminPermissions: false,
            });
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [action],
            });
            app.synth();
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
            const stackArn = _stackArn('MyStack', stack);
            const changeSetCondition = { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } };
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStacks', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateChangeSet', stackArn, changeSetCondition);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteChangeSet', stackArn, changeSetCondition);
            // TODO: revert "as any" once we move all actions into a single package.
            test.deepEqual(stage.fullActions[0].actionProperties.inputs, [artifact], 'The input was correctly registered');
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_CREATE_REPLACE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet',
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const artifact = new codepipeline.Artifact('TestArtifact');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/file'),
                    }),
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                        adminPermissions: false,
                        templatePath: artifact.atPath('path/to/other/file'),
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::GetAtt': ['PipelineTestStageActionARole9283FBE3', 'Arn'] },
                        { 'Fn::GetAtt': ['PipelineTestStageActionBRoleCABC8FA5', 'Arn'] },
                    ],
                },
                {
                    Action: [
                        'cloudformation:CreateChangeSet',
                        'cloudformation:DeleteChangeSet',
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] },
                    ],
                },
            ]);
            test.done();
        },
    },
    ExecuteChangeSet: {
        'works'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            const stage = new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action',
                        changeSetName: 'MyChangeSet',
                        stackName: 'MyStack',
                    }),
                ],
            });
            const stackArn = _stackArn('MyStack', stack);
            _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:ExecuteChangeSet', stackArn, { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } });
            _assertActionMatches(test, stack, stage.fullActions, 'CloudFormation', 'Deploy', {
                ActionMode: 'CHANGE_SET_EXECUTE',
                StackName: 'MyStack',
                ChangeSetName: 'MyChangeSet',
            });
            test.done();
        },
        'uses a single permission statement if the same ChangeSet name is used'(test) {
            const stack = new cdk.Stack();
            const pipelineRole = new RoleDouble(stack, 'PipelineRole');
            new StageDouble({
                pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
                actions: [
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionA',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackA',
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'ActionB',
                        changeSetName: 'MyChangeSet',
                        stackName: 'StackB',
                    }),
                ],
            });
            test.deepEqual(stack.resolve(pipelineRole.statements.map(s => s.toStatementJson())), [
                {
                    Action: [
                        'cloudformation:DescribeChangeSet',
                        'cloudformation:DescribeStacks',
                        'cloudformation:ExecuteChangeSet',
                    ],
                    Condition: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': 'MyChangeSet' } },
                    Effect: 'Allow',
                    Resource: [
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackA/*']] },
                        // eslint-disable-next-line max-len
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':cloudformation:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':stack/StackB/*']] },
                    ],
                },
            ]);
            test.done();
        },
    },
    'the CreateUpdateStack Action sets the DescribeStack*, Create/Update/DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationCreateUpdateStackAction({
            actionName: 'Action',
            templatePath: new codepipeline.Artifact('TestArtifact').atPath('some/file'),
            stackName: 'MyStack',
            adminPermissions: false,
            replaceOnFailure: true,
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:CreateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:UpdateStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
    'the DeleteStack Action sets the DescribeStack*, DeleteStack & PassRole permissions'(test) {
        const stack = new cdk.Stack();
        const pipelineRole = new RoleDouble(stack, 'PipelineRole');
        const action = new cpactions.CloudFormationDeleteStackAction({
            actionName: 'Action',
            adminPermissions: false,
            stackName: 'MyStack',
        });
        new StageDouble({
            pipeline: new PipelineDouble(stack, 'Pipeline', { role: pipelineRole }),
            actions: [action],
        });
        const stackArn = _stackArn('MyStack', stack);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DescribeStack*', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'cloudformation:DeleteStack', stackArn);
        _assertPermissionGranted(test, stack, pipelineRole.statements, 'iam:PassRole', action.deploymentRole.roleArn);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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