"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Snapshot matching
// --------------------------------------------------------------
test('Test for default Params snapshot match', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-snapshot', props);
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Check for ApiGateway params
// --------------------------------------------------------------
test('Test for default Params construct props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    const construct = new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params', props);
    // Assertion
    expect(construct.apiGateway).not.toBeNull();
    expect(construct.apiGatewayCloudWatchRole).not.toBeNull();
    expect(construct.apiGatewayLogGroup).not.toBeNull();
    expect(construct.apiGatewayRole).not.toBeNull();
});
// --------------------------------------------------------------
// Check for Default IAM Role
// --------------------------------------------------------------
test('Test for default IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "awsapigatewayiotpolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Request Validator
// --------------------------------------------------------------
test('Test for default Params request validator', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-request-validator', props);
    // Assertion
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RequestValidator", {
        ValidateRequestBody: false,
        ValidateRequestParameters: true,
    });
});
// --------------------------------------------------------------
// Check for Integ Props and Method Props
// --------------------------------------------------------------
test('Test for default Params Integ Props and Method Props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-integpros-methodprops', props);
    // Assertion for {topic-level-7} to ensure all Integration Request Params, Integration Responses,
    // Method Request Params and Method Reponses are intact
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "5\\d{2}",
                    StatusCode: "500"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    StatusCode: "403"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.path.topic-level-1": "method.request.path.topic-level-1",
                "integration.request.path.topic-level-2": "method.request.path.topic-level-2",
                "integration.request.path.topic-level-3": "method.request.path.topic-level-3",
                "integration.request.path.topic-level-4": "method.request.path.topic-level-4",
                "integration.request.path.topic-level-5": "method.request.path.topic-level-5",
                "integration.request.path.topic-level-6": "method.request.path.topic-level-6",
                "integration.request.path.topic-level-7": "method.request.path.topic-level-7"
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:${props.iotEndpoint}.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}/{topic-level-4}/{topic-level-5}/{topic-level-6}/{topic-level-7}`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200"
            },
            {
                StatusCode: "500"
            },
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true,
            "method.request.path.topic-level-2": true,
            "method.request.path.topic-level-3": true,
            "method.request.path.topic-level-4": true,
            "method.request.path.topic-level-5": true,
            "method.request.path.topic-level-6": true,
            "method.request.path.topic-level-7": true
        }
    });
});
// --------------------------------------------------------------
// Check for valid IoT Endpoint
// --------------------------------------------------------------
test('Test for valid iot enpoint', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: ' '
    };
    const app = () => {
        new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-no-endpoint', props);
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Check for binaryMediaTypes
// --------------------------------------------------------------
test('Test for Binary Media types', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-binaryMediaTypes', {
        iotEndpoint: 'a1234567890123-ats'
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        BinaryMediaTypes: [
            "application/octet-stream",
        ],
    });
});
// --------------------------------------------------------------
// Check for multiple constructs
// --------------------------------------------------------------
test('Test for multiple constructs usage', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params', props);
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params-1', props);
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Check for ApiGateway Overriden Props Snapshot match
// --------------------------------------------------------------
test('Test for overriden props snapshot', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        restApiName: 'RestApi-Regional',
        description: 'Description for the Regional Rest Api',
        endpointConfiguration: { types: [api.EndpointType.REGIONAL] },
        apiKeySourceType: api.ApiKeySourceType.HEADER,
        defaultMethodOptions: {
            authorizationType: api.AuthorizationType.NONE,
        }
    };
    const policyJSON = {
        Version: "2012-10-17",
        Statement: [
            {
                Action: [
                    "iot:UpdateThingShadow"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                Effect: "Allow"
            },
            {
                Action: [
                    "iot:Publish"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                Effect: "Allow"
            }
        ]
    };
    const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
    const iamRoleProps = {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        path: '/',
        inlinePolicies: { testPolicy: policyDocument }
    };
    // Create a policy that overrides the default policy that gets created with the construct
    const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
    // Api gateway setup
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayCreateApiKey: true,
        apiGatewayExecutionRole,
        apiGatewayProps
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-overriden-params', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Check for Api Name and Desc
// --------------------------------------------------------------
test('Test for Api Name and Desc', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        restApiName: 'RestApi-Regional',
        description: 'Description for the Regional Rest Api'
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-name-desc', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        Name: 'RestApi-Regional',
        Description: 'Description for the Regional Rest Api'
    });
});
// --------------------------------------------------------------
// Check for Overriden IAM Role
// --------------------------------------------------------------
test('Test for overriden IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const policyJSON = {
        Version: "2012-10-17",
        Statement: [
            {
                Action: [
                    "iot:UpdateThingShadow"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/mything1`,
                Effect: "Allow"
            },
            {
                Action: [
                    "iot:Publish"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/topic-abc`,
                Effect: "Allow"
            }
        ]
    };
    const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
    const iamRoleProps = {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        path: '/',
        inlinePolicies: { testPolicy: policyDocument }
    };
    // Create a policy that overrides the default policy that gets created with the construct
    const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayExecutionRole,
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-overriden-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/mything1"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/topic-abc"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "testPolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Api Key Source
// --------------------------------------------------------------
test('Test for APi Key Source', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        apiKeySourceType: api.ApiKeySourceType.AUTHORIZER,
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key-source', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        ApiKeySourceType: "AUTHORIZER"
    });
});
// --------------------------------------------------------------
// Check for Api Key Creation
// --------------------------------------------------------------
test('Test for Api Key Creation', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayCreateApiKey: true
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key', props);
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Properties: {
            ApiKeyRequired: true
        },
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W59"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------
// Test deployment for ApiGateway endPointCongiurationOverride
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-auth-none', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.REGIONAL]
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        EndpointConfiguration: {
            Types: ["REGIONAL"]
        }
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway AuthorizationType to NONE
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
});
// -----------------------------------------------------------------
// Test deployment for fully qualified iotEndpoint name
// -----------------------------------------------------------------
test('Test for handling fully qualified iotEndpoint', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats.iot.ap-south-1.amazonaws.com',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Integration: {
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        ":a1234567890123-ats.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}"
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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