"""
This module contains functions for interacting with the PPRL broker service
provided by the Medical Data Science Group Leipzig.
"""

from typing import Optional, List, NamedTuple, Dict, Union

import requests

from .model import MatchConfiguration, SessionCancellation, BitVectorMetadataSpecification, \
    BitVector, BitVectorMatch, ApiError
from .schema import _serialize_match_configuration, _serialize_session_cancellation, \
    _serialize_metadata_specification, _deserialize_secret_response, _serialize_bit_vector, \
    _deserialize_progress_response, _deserialize_bit_vector_match
from .util import _append_path, _check_response_errors, GenericRestClient


class SessionCreationResponse(NamedTuple):
    """
    Typing for the values returned by a session creation request.
    """
    session_secret: str
    """Generated session secret"""
    cancellation_headers: Dict[str, str]
    """Extra headers generated by chosen cancellation method"""


def _append_auth(headers: Dict[str, str], bearer: str) -> Dict[str, str]:
    headers["Authorization"] = f"Bearer {bearer}"
    return headers


def create_session(
        base_url: str,
        config: Optional[MatchConfiguration] = None,
        session_cancellation: Optional[Union[str, SessionCancellation]] = None,
        metadata_specifications: Optional[List[BitVectorMetadataSpecification]] = None
) -> SessionCreationResponse:
    """
    Creates a new session.

    :param base_url: Broker service base URL
    :param config: Match configuration for the session
    :param session_cancellation: Session cancellation strategy name or instance
    :param metadata_specifications: List of metadata specifications to assign
    :return: Generated session secret
    """
    json_data = {}

    if config is not None:
        json_data["matchConfiguration"] = _serialize_match_configuration(config)

    if session_cancellation is not None:
        if type(session_cancellation) == str:
            session_cancellation = SessionCancellation(session_cancellation)

        json_data["sessionCancellation"] = _serialize_session_cancellation(session_cancellation)

    if metadata_specifications is not None:
        json_data["metadataSpecification"] = [
            _serialize_metadata_specification(s) for s in metadata_specifications
        ]

    url = _append_path(base_url, "session")
    r = requests.post(url, json=json_data)

    if r.status_code != requests.codes.created:
        raise ApiError("Session couldn't be created", r.status_code)

    session_secret = _deserialize_secret_response(r.json())
    session_cancellation_headers: Dict[str, str] = {}

    # iterate over headers
    for header_name, header_value in r.headers.items():
        header_name = header_name.lower()
        header_value = header_value.lower()

        header_start = header_name[0:2].lower()

        # check if there are any custom ones
        if header_start == "x-":
            # if so, remember them
            session_cancellation_headers[header_name] = header_value

    return SessionCreationResponse(session_secret, session_cancellation_headers)


def cancel_session(
        base_url: str,
        session_secret: str,
        cancellation_options: Dict[str, str] = None
):
    """
    Cancels a session.

    :param base_url: Broker service base URL
    :param session_secret: Secret of the session to cancel
    :param cancellation_options: Additional cancellation options
    """
    if cancellation_options is None:
        cancellation_options = {}

    url = _append_path(base_url, "session")
    r = requests.delete(url, headers=_append_auth(cancellation_options, session_secret))

    if r.status_code != requests.codes.accepted:
        raise ApiError("Session couldn't be cancelled", r.status_code)


def create_client(
        base_url: str,
        session_secret: str
) -> str:
    """
    Creates a new client inside a session.

    :param base_url: Broker service base URL
    :param session_secret: Secret of the session to create a new client in
    :return: Generated client secret
    """
    url = _append_path(base_url, "submit")
    r = requests.get(url, headers=_append_auth({}, session_secret))

    if r.status_code != requests.codes.created:
        raise ApiError("Client couldn't be created", r.status_code)

    return _deserialize_secret_response(r.json())


def submit_bit_vectors(
        base_url: str,
        client_secret: str,
        vectors: List[BitVector] = None
):
    """
    Submits bit vectors to a session.

    :param base_url: Broker service base URL
    :param client_secret: Secret of the client to submit bit vectors
    :param vectors: Bit vectors to submit
    """
    if len(vectors) == 0:
        return

    json_data = {
        "vectors": [
            _serialize_bit_vector(v) for v in vectors
        ]
    }

    url = _append_path(base_url, "submit")
    r = requests.post(url, json=json_data, headers=_append_auth({}, client_secret))

    _check_response_errors(r, requests.codes.accepted, {
        requests.codes.unauthorized: "Client has not been registered at the broker"
    })


def get_session_progress(
        base_url: str,
        session_secret: str
) -> float:
    """
    Returns the match progress of the session

    :param base_url: Broker service base URL
    :param session_secret: Secret of the session to get match progress for
    :return: Match progress of session
    """
    url = _append_path(base_url, "progress")
    r = requests.get(url, headers=_append_auth({}, session_secret))

    _check_response_errors(r, requests.codes.ok, {
        requests.codes.unauthorized: "Session has not been registered at the broker"
    })

    return _deserialize_progress_response(r.json())


def get_results(
        base_url: str,
        client_secret: str
) -> List[BitVectorMatch]:
    """
    Returns the match results for a client.

    :param base_url: Broker service base URL
    :param client_secret: Secret of the client to fetch results for
    :return: List of match results for client
    """
    url = _append_path(base_url, "result")
    r = requests.get(url, headers=_append_auth({}, client_secret))

    _check_response_errors(r, requests.codes.ok, {
        requests.codes.unauthorized: "Session has not been registered at the broker"
    })

    return [
        _deserialize_bit_vector_match(m) for m in r.json()["matches"]
    ]


def get_cancellation_strategies(base_url: str) -> List[str]:
    """
    Returns a list of cancellation strategies supported by the broker.

    :param base_url: Broker service base URL
    :return: List of supported cancellation strategies
    """
    url = _append_path(base_url, "session-cancellation-strategies")
    r = requests.get(url)

    if r.status_code != requests.codes.ok:
        raise ApiError("Couldn't fetch session cancellation strategies", r.status_code)

    return r.json()


class BrokerClient(GenericRestClient):
    """
    Wrapper around the API endpoints exposed by the broker service.
    """

    def create_session(
            self,
            config: Optional[MatchConfiguration] = None,
            session_cancellation: Optional[Union[str, SessionCancellation]] = None,
            metadata_specifications: Optional[List[BitVectorMetadataSpecification]] = None
    ) -> SessionCreationResponse:
        """
        Creates a new session.

        :param config: Match configuration for the session
        :param session_cancellation: Session cancellation strategy name or instance
        :param metadata_specifications: List of metadata specifications to assign
        :return: Generated session secret
        """
        return create_session(self._base_url, config, session_cancellation, metadata_specifications)

    def cancel_session(
            self,
            session_secret: str,
            cancellation_options: Dict[str, str] = None
    ):
        """
        Cancels a session.

        :param session_secret: Secret of the session to cancel
        :param cancellation_options: Additional cancellation options
        """
        cancel_session(self._base_url, session_secret, cancellation_options)

    def create_client(
            self,
            session_secret: str
    ) -> str:
        """
        Creates a new client inside a session.

        :param session_secret: Secret of the session to create a new client in
        :return: Generated client secret
        """
        return create_client(self._base_url, session_secret)

    def submit_bit_vectors(
            self,
            client_secret: str,
            vectors: List[BitVector] = None
    ):
        """
        Submits bit vectors to a session.

        :param client_secret: Secret of the client to submit bit vectors
        :param vectors: Bit vectors to submit
        """
        submit_bit_vectors(self._base_url, client_secret, vectors)

    def get_session_progress(
            self,
            session_secret: str
    ) -> float:
        """
        Returns the match progress of the session

        :param session_secret: Secret of the session to get match progress for
        :return: Match progress of session
        """
        return get_session_progress(self._base_url, session_secret)

    def get_results(
            self,
            client_secret: str
    ) -> List[BitVectorMatch]:
        """
        Returns the match results for a client.

        :param client_secret: Secret of the client to fetch results for
        :return: List of match results for client
        """
        return get_results(self._base_url, client_secret)

    def get_cancellation_strategies(self) -> List[str]:
        """
        Returns a list of cancellation strategies supported by the broker.

        :return: List of supported cancellation strategies
        """
        return get_cancellation_strategies(self._base_url)
