#!/usr/bin/env python
"""
@author: metalcorebear
"""

# Determine word affect based on the NRC emotional lexicon
# Library is built on TextBlob

from textblob import TextBlob
from collections import Counter


def build_word_affect(self):
    # Build word affect function
    affect_list = []
    affect_dict = dict()
    affect_frequencies = Counter()
    lexicon_keys = self.lexicon.keys()
    for word in self.words:
        if word in lexicon_keys:
            affect_list.extend(self.lexicon[word])
            affect_dict.update({word: self.lexicon[word]})
    for word in affect_list:
        affect_frequencies[word] += 1
    sum_values = sum(affect_frequencies.values())
    affect_percent = {'fear': 0.0, 'anger': 0.0, 'anticip': 0.0, 'trust': 0.0, 'surprise': 0.0, 'positive': 0.0,
                      'negative': 0.0, 'sadness': 0.0, 'disgust': 0.0, 'joy': 0.0}
    for key in affect_frequencies.keys():
        affect_percent.update({key: float(affect_frequencies[key]) / float(sum_values)})
    self.affect_list = affect_list
    self.affect_dict = affect_dict
    self.raw_emotion_scores = dict(affect_frequencies)
    self.affect_frequencies = affect_percent


def top_emotions(self):
    emo_dict = self.affect_frequencies
    max_value = max(emo_dict.values())
    top_emotions = []
    for key in emo_dict.keys():
        if emo_dict[key] == max_value:
            top_emotions.append((key, max_value))
    self.top_emotions = top_emotions


class NRCLex:
    """Lexicon source is (C) 2016 National Research Council Canada (NRC) and library is for research purposes only.  Source: http://sentiment.nrc.ca/lexicons-for-research/"""

    lexicon = {'abacus': ['trust'], 'abandon': ['fear', 'negative', 'sadness'],
               'abandoned': ['anger', 'fear', 'negative', 'sadness'],
               'abandonment': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'abba': ['positive'],
               'abbot': ['trust'], 'abduction': ['fear', 'negative', 'sadness', 'surprise'], 'aberrant': ['negative'],
               'aberration': ['disgust', 'negative'], 'abhor': ['anger', 'disgust', 'fear', 'negative'],
               'abhorrent': ['anger', 'disgust', 'fear', 'negative'], 'ability': ['positive'],
               'abject': ['disgust', 'negative'], 'abnormal': ['disgust', 'negative'], 'abolish': ['anger', 'negative'],
               'abolition': ['negative'], 'abominable': ['disgust', 'fear', 'negative'],
               'abomination': ['anger', 'disgust', 'fear', 'negative'], 'abort': ['negative'],
               'abortion': ['disgust', 'fear', 'negative', 'sadness'], 'abortive': ['negative', 'sadness'],
               'abovementioned': ['positive'], 'abrasion': ['negative'], 'abrogate': ['negative'],
               'abrupt': ['surprise'], 'abscess': ['negative', 'sadness'], 'absence': ['fear', 'negative', 'sadness'],
               'absent': ['negative', 'sadness'], 'absentee': ['negative', 'sadness'], 'absenteeism': ['negative'],
               'absolute': ['positive'], 'absolution': ['joy', 'positive', 'trust'], 'absorbed': ['positive'],
               'absurd': ['negative'], 'absurdity': ['negative'],
               'abundance': ['anticipation', 'disgust', 'joy', 'negative', 'positive', 'trust'],
               'abundant': ['joy', 'positive'], 'abuse': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'abysmal': ['negative', 'sadness'], 'abyss': ['fear', 'negative', 'sadness'],
               'academic': ['positive', 'trust'], 'academy': ['positive'], 'accelerate': ['anticipation'],
               'acceptable': ['positive'], 'acceptance': ['positive'], 'accessible': ['positive'],
               'accident': ['fear', 'negative', 'sadness', 'surprise'], 'accidental': ['fear', 'negative', 'surprise'],
               'accidentally': ['surprise'], 'accolade': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'accommodation': ['positive'], 'accompaniment': ['anticipation', 'joy', 'positive', 'trust'],
               'accomplish': ['joy', 'positive'], 'accomplished': ['joy', 'positive'], 'accomplishment': ['positive'],
               'accord': ['positive', 'trust'], 'account': ['trust'], 'accountability': ['positive', 'trust'],
               'accountable': ['positive', 'trust'], 'accountant': ['trust'], 'accounts': ['trust'],
               'accredited': ['positive', 'trust'], 'accueil': ['positive'], 'accurate': ['positive', 'trust'],
               'accursed': ['anger', 'fear', 'negative', 'sadness'], 'accusation': ['anger', 'disgust', 'negative'],
               'accusative': ['negative'], 'accused': ['anger', 'fear', 'negative'],
               'accuser': ['anger', 'fear', 'negative'], 'accusing': ['anger', 'fear', 'negative'], 'ace': ['positive'],
               'ache': ['negative', 'sadness'], 'achieve': ['joy', 'positive', 'trust'],
               'achievement': ['anticipation', 'joy', 'positive', 'trust'], 'aching': ['negative', 'sadness'],
               'acid': ['negative'], 'acknowledgment': ['positive'], 'acquire': ['positive'],
               'acquiring': ['anticipation', 'positive'], 'acrobat': ['fear', 'joy', 'positive', 'trust'],
               'action': ['positive'], 'actionable': ['anger', 'disgust', 'negative'], 'actual': ['positive'],
               'acuity': ['positive'], 'acumen': ['positive'], 'adapt': ['positive'], 'adaptable': ['positive'],
               'adder': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'addiction': ['negative'],
               'addresses': ['anticipation', 'positive'], 'adept': ['positive'], 'adequacy': ['positive'],
               'adhering': ['trust'], 'adipose': ['negative'], 'adjudicate': ['fear', 'negative'],
               'adjunct': ['positive'], 'administrative': ['trust'], 'admirable': ['joy', 'positive', 'trust'],
               'admiral': ['positive', 'trust'], 'admiration': ['joy', 'positive', 'trust'],
               'admire': ['positive', 'trust'], 'admirer': ['positive'], 'admissible': ['positive', 'trust'],
               'admonition': ['fear', 'negative'], 'adorable': ['joy', 'positive'],
               'adoration': ['joy', 'positive', 'trust'], 'adore': ['anticipation', 'joy', 'positive', 'trust'],
               'adrift': ['anticipation', 'fear', 'negative', 'sadness'], 'adulterated': ['negative'],
               'adultery': ['disgust', 'negative', 'sadness'],
               'advance': ['anticipation', 'fear', 'joy', 'positive', 'surprise'], 'advanced': ['positive'],
               'advancement': ['positive'], 'advantage': ['positive'], 'advantageous': ['positive'],
               'advent': ['anticipation', 'joy', 'positive', 'trust'], 'adventure': ['anticipation', 'positive'],
               'adventurous': ['positive'], 'adversary': ['anger', 'negative'],
               'adverse': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'adversity': ['anger', 'fear', 'negative', 'sadness'], 'advice': ['trust'],
               'advisable': ['positive', 'trust'], 'advise': ['positive', 'trust'], 'advised': ['trust'],
               'adviser': ['positive', 'trust'], 'advocacy': ['anger', 'anticipation', 'joy', 'positive', 'trust'],
               'advocate': ['trust'], 'aesthetic': ['positive'], 'aesthetics': ['joy', 'positive'],
               'affable': ['positive'], 'affection': ['joy', 'positive', 'trust'], 'affiliated': ['positive'],
               'affirm': ['positive', 'trust'], 'affirmation': ['positive'], 'affirmative': ['positive'],
               'affirmatively': ['positive', 'trust'], 'afflict': ['fear', 'negative', 'sadness'],
               'afflicted': ['negative'], 'affliction': ['disgust', 'fear', 'negative', 'sadness'],
               'affluence': ['joy', 'positive'], 'affluent': ['positive'], 'afford': ['positive'],
               'affront': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'afraid': ['fear', 'negative'], 'aftermath': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'aftertaste': ['negative'], 'aga': ['fear', 'positive', 'trust'], 'aggravated': ['anger', 'negative'],
               'aggravating': ['anger', 'negative', 'sadness'], 'aggravation': ['anger', 'disgust', 'negative'],
               'aggression': ['anger', 'fear', 'negative'], 'aggressive': ['anger', 'fear', 'negative'],
               'aggressor': ['anger', 'fear', 'negative'], 'aghast': ['disgust', 'fear', 'negative', 'surprise'],
               'agile': ['positive'], 'agility': ['positive'], 'agitated': ['anger', 'negative'],
               'agitation': ['anger', 'negative'], 'agonizing': ['fear', 'negative'],
               'agony': ['anger', 'fear', 'negative', 'sadness'], 'agree': ['positive'],
               'agreeable': ['positive', 'trust'], 'agreed': ['positive', 'trust'], 'agreeing': ['positive', 'trust'],
               'agreement': ['positive', 'trust'], 'agriculture': ['positive'], 'aground': ['negative'],
               'ahead': ['positive'], 'aid': ['positive'], 'aiding': ['positive'], 'ail': ['negative', 'sadness'],
               'ailing': ['fear', 'negative', 'sadness'], 'aimless': ['negative'], 'airport': ['anticipation'],
               'airs': ['disgust', 'negative'], 'akin': ['trust'], 'alabaster': ['positive'],
               'alarm': ['fear', 'negative', 'surprise'], 'alarming': ['fear', 'negative', 'surprise'],
               'alb': ['trust'], 'alcoholism': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'alertness': ['anticipation', 'fear', 'positive', 'surprise'],
               'alerts': ['anticipation', 'fear', 'surprise'], 'alien': ['disgust', 'fear', 'negative'],
               'alienate': ['anger', 'disgust', 'negative'], 'alienated': ['negative', 'sadness'],
               'alienation': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'alimentation': ['positive'],
               'alimony': ['negative'], 'alive': ['anticipation', 'joy', 'positive', 'trust'], 'allay': ['positive'],
               'allegation': ['anger', 'negative'], 'allege': ['negative'], 'allegiance': ['positive', 'trust'],
               'allegro': ['positive'], 'alleviate': ['positive'], 'alleviation': ['positive'], 'alliance': ['trust'],
               'allied': ['positive', 'trust'], 'allowable': ['positive'],
               'allure': ['anticipation', 'joy', 'positive', 'surprise'], 'alluring': ['positive'],
               'ally': ['positive', 'trust'], 'almighty': ['positive'], 'aloha': ['anticipation', 'joy', 'positive'],
               'aloof': ['negative'], 'altercation': ['anger', 'negative'], 'amaze': ['surprise'],
               'amazingly': ['joy', 'positive', 'surprise'], 'ambassador': ['positive', 'trust'],
               'ambiguous': ['negative'], 'ambition': ['anticipation', 'joy', 'positive', 'trust'],
               'ambulance': ['fear', 'trust'], 'ambush': ['anger', 'fear', 'negative', 'surprise'],
               'ameliorate': ['positive'], 'amen': ['joy', 'positive', 'trust'], 'amenable': ['positive'],
               'amend': ['positive'], 'amends': ['positive'], 'amenity': ['positive'], 'amiable': ['positive'],
               'amicable': ['joy', 'positive'], 'ammonia': ['disgust'], 'amnesia': ['negative'],
               'amnesty': ['joy', 'positive'], 'amortization': ['trust'],
               'amour': ['anticipation', 'joy', 'positive', 'trust'], 'amphetamines': ['disgust', 'negative'],
               'amuse': ['joy', 'positive'], 'amused': ['joy', 'positive'], 'amusement': ['joy', 'positive'],
               'amusing': ['joy', 'positive'], 'anaconda': ['disgust', 'fear', 'negative'], 'anal': ['negative'],
               'analyst': ['anticipation', 'positive', 'trust'], 'anarchism': ['anger', 'fear', 'negative'],
               'anarchist': ['anger', 'fear', 'negative'], 'anarchy': ['anger', 'fear', 'negative'],
               'anathema': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'ancestral': ['trust'],
               'anchor': ['positive'], 'anchorage': ['positive', 'sadness'], 'ancient': ['negative'],
               'angel': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'angelic': ['joy', 'positive', 'trust'], 'anger': ['anger', 'negative'], 'angina': ['fear', 'negative'],
               'angling': ['anticipation', 'negative'], 'angry': ['anger', 'disgust', 'negative'],
               'anguish': ['anger', 'fear', 'negative', 'sadness'], 'animate': ['positive'],
               'animated': ['joy', 'positive'], 'animosity': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'animus': ['anger', 'negative'], 'annihilate': ['anger', 'fear', 'negative'],
               'annihilated': ['anger', 'fear', 'negative', 'sadness'],
               'annihilation': ['anger', 'fear', 'negative', 'sadness'], 'announcement': ['anticipation'],
               'annoy': ['anger', 'disgust', 'negative'], 'annoyance': ['anger', 'disgust', 'negative'],
               'annoying': ['anger', 'negative'], 'annul': ['negative'], 'annulment': ['negative', 'sadness'],
               'anomaly': ['fear', 'negative', 'surprise'], 'anonymous': ['negative'], 'answerable': ['trust'],
               'antagonism': ['anger', 'negative'], 'antagonist': ['anger', 'negative'],
               'antagonistic': ['anger', 'disgust', 'negative'], 'anthrax': ['disgust', 'fear', 'negative', 'sadness'],
               'antibiotics': ['positive'], 'antichrist': ['anger', 'disgust', 'fear', 'negative'],
               'anticipation': ['anticipation'], 'anticipatory': ['anticipation'],
               'antidote': ['anticipation', 'positive', 'trust'], 'antifungal': ['positive', 'trust'],
               'antipathy': ['anger', 'disgust', 'negative'], 'antiquated': ['negative'], 'antique': ['positive'],
               'antiseptic': ['positive', 'trust'], 'antisocial': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'antithesis': ['anger', 'negative'], 'anxiety': ['anger', 'anticipation', 'fear', 'negative', 'sadness'],
               'anxious': ['anticipation', 'fear', 'negative'], 'apache': ['fear', 'negative'],
               'apathetic': ['negative', 'sadness'], 'apathy': ['negative', 'sadness'],
               'aphid': ['disgust', 'negative'], 'aplomb': ['positive'], 'apologetic': ['positive', 'trust'],
               'apologize': ['positive', 'sadness', 'trust'], 'apology': ['positive'], 'apostle': ['positive', 'trust'],
               'apostolic': ['trust'], 'appalling': ['disgust', 'fear', 'negative'], 'apparition': ['fear', 'surprise'],
               'appeal': ['anticipation'], 'appendicitis': ['fear', 'negative', 'sadness'],
               'applause': ['joy', 'positive', 'surprise', 'trust'], 'applicant': ['anticipation'],
               'appreciation': ['joy', 'positive', 'trust'], 'apprehend': ['fear'],
               'apprehension': ['fear', 'negative'], 'apprehensive': ['anticipation', 'fear', 'negative'],
               'apprentice': ['trust'], 'approaching': ['anticipation'], 'approbation': ['positive', 'trust'],
               'appropriation': ['negative'], 'approval': ['positive'], 'approve': ['joy', 'positive', 'trust'],
               'approving': ['positive'], 'apt': ['positive'], 'aptitude': ['positive'], 'arbiter': ['trust'],
               'arbitration': ['anticipation'], 'arbitrator': ['trust'], 'archaeology': ['anticipation', 'positive'],
               'archaic': ['negative'], 'architecture': ['trust'], 'ardent': ['anticipation', 'joy', 'positive'],
               'ardor': ['positive'], 'arduous': ['negative'], 'argue': ['anger', 'negative'],
               'argument': ['anger', 'negative'], 'argumentation': ['anger'], 'argumentative': ['negative'],
               'arguments': ['anger'], 'arid': ['negative', 'sadness'], 'aristocracy': ['positive'],
               'aristocratic': ['positive'], 'armament': ['anger', 'fear'], 'armaments': ['fear', 'negative'],
               'armed': ['anger', 'fear', 'negative', 'positive'], 'armor': ['fear', 'positive', 'trust'],
               'armored': ['fear'], 'armory': ['trust'], 'aroma': ['positive'], 'arouse': ['anticipation', 'positive'],
               'arraignment': ['anger', 'fear', 'negative', 'sadness'], 'array': ['positive'], 'arrears': ['negative'],
               'arrest': ['negative'], 'arrival': ['anticipation'], 'arrive': ['anticipation'],
               'arrogance': ['negative'], 'arrogant': ['anger', 'disgust', 'negative'],
               'arsenic': ['disgust', 'fear', 'negative', 'sadness'], 'arson': ['anger', 'fear', 'negative'],
               'art': ['anticipation', 'joy', 'positive', 'sadness', 'surprise'], 'articulate': ['positive'],
               'articulation': ['positive'], 'artillery': ['fear', 'negative'], 'artisan': ['positive'],
               'artiste': ['positive'], 'artistic': ['positive'], 'ascendancy': ['positive'], 'ascent': ['positive'],
               'ash': ['negative'], 'ashamed': ['disgust', 'negative', 'sadness'], 'ashes': ['negative', 'sadness'],
               'asp': ['fear'], 'aspiration': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'aspire': ['anticipation', 'joy', 'positive'], 'aspiring': ['anticipation', 'joy', 'positive', 'trust'],
               'ass': ['negative'], 'assail': ['anger', 'fear', 'negative', 'surprise'],
               'assailant': ['anger', 'fear', 'negative', 'sadness'],
               'assassin': ['anger', 'fear', 'negative', 'sadness'], 'assassinate': ['anger', 'fear', 'negative'],
               'assassination': ['anger', 'fear', 'negative', 'sadness'], 'assault': ['anger', 'fear', 'negative'],
               'assembly': ['positive', 'trust'], 'assent': ['positive'], 'asserting': ['positive', 'trust'],
               'assessment': ['surprise', 'trust'], 'assessor': ['trust'], 'assets': ['positive'],
               'asshole': ['anger', 'disgust', 'negative'], 'assignee': ['trust'], 'assist': ['positive', 'trust'],
               'assistance': ['positive'], 'associate': ['positive', 'trust'], 'association': ['trust'],
               'assuage': ['positive'], 'assurance': ['positive', 'trust'], 'assure': ['trust'],
               'assured': ['positive', 'trust'], 'assuredly': ['trust'], 'astonishingly': ['positive', 'surprise'],
               'astonishment': ['joy', 'positive', 'surprise'], 'astray': ['fear', 'negative'],
               'astringent': ['negative'], 'astrologer': ['anticipation', 'positive'], 'astronaut': ['positive'],
               'astronomer': ['anticipation', 'positive'], 'astute': ['positive'], 'asylum': ['fear', 'negative'],
               'asymmetry': ['disgust'], 'atheism': ['negative'], 'atherosclerosis': ['fear', 'negative', 'sadness'],
               'athlete': ['positive'], 'athletic': ['positive'], 'atom': ['positive'],
               'atone': ['anticipation', 'joy', 'positive', 'trust'], 'atonement': ['positive'],
               'atrocious': ['anger', 'disgust', 'negative'],
               'atrocity': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'atrophy': ['disgust', 'fear', 'negative', 'sadness'], 'attachment': ['positive'],
               'attack': ['anger', 'fear', 'negative'],
               'attacking': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'attainable': ['anticipation', 'positive'], 'attainment': ['positive'], 'attempt': ['anticipation'],
               'attendance': ['anticipation'], 'attendant': ['positive', 'trust'], 'attention': ['positive'],
               'attentive': ['positive', 'trust'], 'attenuated': ['negative'], 'attenuation': ['negative', 'sadness'],
               'attest': ['positive', 'trust'], 'attestation': ['trust'],
               'attorney': ['anger', 'fear', 'positive', 'trust'], 'attraction': ['positive'],
               'attractiveness': ['positive'], 'auction': ['anticipation'], 'audacity': ['negative'],
               'audience': ['anticipation'], 'auditor': ['fear', 'trust'], 'augment': ['positive'],
               'august': ['positive'], 'aunt': ['positive', 'trust'], 'aura': ['positive'],
               'auspicious': ['anticipation', 'joy', 'positive'], 'austere': ['fear', 'negative', 'sadness'],
               'austerity': ['negative'], 'authentic': ['joy', 'positive', 'trust'], 'authenticate': ['trust'],
               'authentication': ['trust'], 'authenticity': ['positive', 'trust'], 'author': ['positive', 'trust'],
               'authoritative': ['positive', 'trust'], 'authority': ['positive', 'trust'],
               'authorization': ['positive', 'trust'], 'authorize': ['trust'], 'authorized': ['positive'],
               'autocratic': ['negative'], 'automatic': ['trust'],
               'autopsy': ['disgust', 'fear', 'negative', 'sadness'],
               'avalanche': ['fear', 'negative', 'sadness', 'surprise'], 'avarice': ['anger', 'disgust', 'negative'],
               'avatar': ['positive'], 'avenger': ['anger', 'negative'],
               'averse': ['anger', 'disgust', 'fear', 'negative'], 'aversion': ['anger', 'disgust', 'fear', 'negative'],
               'avoid': ['fear', 'negative'], 'avoidance': ['fear', 'negative'], 'avoiding': ['fear'],
               'await': ['anticipation'], 'award': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'awful': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'awkwardness': ['disgust', 'negative'],
               'awry': ['negative'], 'axiom': ['trust'], 'axiomatic': ['trust'], 'ay': ['positive'],
               'aye': ['positive'], 'babble': ['negative'], 'babbling': ['negative'], 'baboon': ['disgust', 'negative'],
               'baby': ['joy', 'positive'], 'babysitter': ['trust'], 'baccalaureate': ['positive'],
               'backbone': ['anger', 'positive', 'trust'], 'backer': ['trust'], 'backward': ['negative'],
               'backwards': ['disgust', 'negative'], 'backwater': ['negative', 'sadness'],
               'bacteria': ['disgust', 'fear', 'negative', 'sadness'], 'bacterium': ['disgust', 'fear', 'negative'],
               'bad': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'badge': ['trust'],
               'badger': ['anger', 'negative'], 'badly': ['negative', 'sadness'],
               'badness': ['anger', 'disgust', 'fear', 'negative'], 'bailiff': ['fear', 'negative', 'trust'],
               'bait': ['fear', 'negative', 'trust'], 'balance': ['positive'], 'balanced': ['positive'],
               'bale': ['fear', 'negative'], 'balk': ['negative'], 'ballad': ['positive'], 'ballet': ['positive'],
               'ballot': ['anticipation', 'positive', 'trust'], 'balm': ['anticipation', 'joy', 'negative', 'positive'],
               'balsam': ['positive'], 'ban': ['negative'], 'bandit': ['negative'],
               'bane': ['anger', 'disgust', 'fear', 'negative'],
               'bang': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'banger': ['anger', 'anticipation', 'fear', 'negative', 'surprise'],
               'banish': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'banished': ['anger', 'fear', 'negative', 'sadness'],
               'banishment': ['anger', 'disgust', 'negative', 'sadness'], 'bank': ['trust'], 'banker': ['trust'],
               'bankrupt': ['fear', 'negative', 'sadness'],
               'bankruptcy': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'banquet': ['anticipation', 'joy', 'positive'],
               'banshee': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'baptism': ['positive'],
               'baptismal': ['joy', 'positive'], 'barb': ['anger', 'negative'], 'barbarian': ['fear', 'negative'],
               'barbaric': ['anger', 'disgust', 'fear', 'negative'], 'barbarism': ['negative'], 'bard': ['positive'],
               'barf': ['disgust'], 'bargain': ['positive', 'trust'], 'bark': ['anger', 'negative'],
               'barred': ['negative'], 'barren': ['negative', 'sadness'], 'barricade': ['fear', 'negative'],
               'barrier': ['anger', 'negative'], 'barrow': ['disgust'], 'bartender': ['trust'], 'barter': ['trust'],
               'base': ['trust'], 'baseless': ['negative'], 'basketball': ['anticipation', 'joy', 'positive'],
               'bastard': ['disgust', 'negative', 'sadness'], 'bastion': ['anger', 'positive'], 'bath': ['positive'],
               'battalion': ['anger'], 'batter': ['anger', 'fear', 'negative'],
               'battered': ['fear', 'negative', 'sadness'], 'battery': ['anger', 'negative'],
               'battle': ['anger', 'negative'], 'battled': ['anger', 'fear', 'negative', 'sadness'],
               'battlefield': ['fear', 'negative'], 'bawdy': ['negative'], 'bayonet': ['anger', 'fear', 'negative'],
               'beach': ['joy'], 'beam': ['joy', 'positive'], 'beaming': ['anticipation', 'joy', 'positive'],
               'bear': ['anger', 'fear'], 'bearer': ['negative'], 'bearish': ['anger', 'fear'],
               'beast': ['anger', 'fear', 'negative'], 'beastly': ['disgust', 'fear', 'negative'],
               'beating': ['anger', 'fear', 'negative', 'sadness'], 'beautification': ['joy', 'positive', 'trust'],
               'beautiful': ['joy', 'positive'], 'beautify': ['joy', 'positive'], 'beauty': ['joy', 'positive'],
               'bedrock': ['positive', 'trust'], 'bee': ['anger', 'fear'], 'beer': ['joy', 'positive'],
               'befall': ['negative'], 'befitting': ['positive'], 'befriend': ['joy', 'positive', 'trust'],
               'beg': ['negative', 'sadness'], 'beggar': ['negative', 'sadness'], 'begging': ['negative'],
               'begun': ['anticipation'], 'behemoth': ['fear', 'negative'], 'beholden': ['negative'],
               'belated': ['negative'], 'believed': ['trust'], 'believer': ['trust'],
               'believing': ['positive', 'trust'], 'belittle': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'belligerent': ['anger', 'fear', 'negative'], 'bellows': ['anger'],
               'belt': ['anger', 'fear', 'negative'], 'bender': ['negative'], 'benefactor': ['positive', 'trust'],
               'beneficial': ['positive'], 'benefit': ['positive'], 'benevolence': ['joy', 'positive', 'trust'],
               'benign': ['joy', 'positive'], 'bequest': ['trust'], 'bereaved': ['negative', 'sadness'],
               'bereavement': ['negative', 'sadness'], 'bereft': ['negative'], 'berserk': ['anger', 'negative'],
               'berth': ['positive'], 'bestial': ['disgust', 'fear', 'negative'],
               'betray': ['anger', 'disgust', 'negative', 'sadness', 'surprise'],
               'betrayal': ['anger', 'disgust', 'negative', 'sadness'],
               'betrothed': ['anticipation', 'joy', 'positive', 'trust'], 'betterment': ['positive'],
               'beverage': ['positive'], 'beware': ['anticipation', 'fear', 'negative'],
               'bewildered': ['fear', 'negative', 'surprise'], 'bewilderment': ['fear', 'surprise'],
               'bias': ['anger', 'negative'], 'biased': ['negative'], 'biblical': ['positive'],
               'bickering': ['anger', 'disgust', 'negative'], 'biennial': ['anticipation'],
               'bier': ['fear', 'negative', 'sadness'], 'bigot': ['anger', 'disgust', 'fear', 'negative'],
               'bigoted': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'bile': ['anger', 'disgust', 'negative'],
               'bilingual': ['positive'], 'biopsy': ['fear', 'negative'],
               'birch': ['anger', 'disgust', 'fear', 'negative'],
               'birth': ['anticipation', 'fear', 'joy', 'positive', 'trust'],
               'birthday': ['anticipation', 'joy', 'positive', 'surprise'], 'birthplace': ['anger', 'negative'],
               'bitch': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'bite': ['negative'],
               'bitterly': ['anger', 'disgust', 'negative', 'sadness'],
               'bitterness': ['anger', 'disgust', 'negative', 'sadness'], 'bizarre': ['negative', 'surprise'],
               'black': ['negative', 'sadness'], 'blackjack': ['negative'], 'blackmail': ['anger', 'fear', 'negative'],
               'blackness': ['fear', 'negative', 'sadness'], 'blame': ['anger', 'disgust', 'negative'],
               'blameless': ['positive'], 'bland': ['negative'], 'blanket': ['trust'],
               'blasphemous': ['anger', 'disgust', 'negative'], 'blasphemy': ['anger', 'negative'],
               'blast': ['anger', 'fear', 'negative', 'surprise'], 'blatant': ['anger', 'disgust', 'negative'],
               'blather': ['negative'], 'blaze': ['anger', 'negative'], 'bleak': ['negative', 'sadness'],
               'bleeding': ['disgust', 'fear', 'negative', 'sadness'],
               'blemish': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'bless': ['anticipation', 'joy', 'positive', 'trust'], 'blessed': ['joy', 'positive'],
               'blessing': ['anticipation', 'joy', 'positive', 'trust'],
               'blessings': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'blight': ['disgust', 'fear', 'negative', 'sadness'], 'blighted': ['disgust', 'negative', 'sadness'],
               'blind': ['negative'], 'blinded': ['negative'], 'blindfold': ['anticipation', 'fear', 'surprise'],
               'blindly': ['negative', 'sadness'], 'blindness': ['negative', 'sadness'], 'bliss': ['joy', 'positive'],
               'blissful': ['joy', 'positive'], 'blister': ['disgust', 'negative'], 'blitz': ['surprise'],
               'bloated': ['disgust', 'negative'], 'blob': ['disgust', 'fear', 'negative'],
               'blockade': ['anger', 'fear', 'negative', 'sadness'], 'bloodless': ['positive'],
               'bloodshed': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'bloodthirsty': ['anger', 'disgust', 'fear', 'negative'],
               'bloody': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'bloom': ['anticipation', 'joy', 'positive', 'trust'], 'blossom': ['joy', 'positive'],
               'blot': ['negative'], 'blower': ['negative'], 'blowout': ['negative'], 'blue': ['sadness'],
               'blues': ['fear', 'negative', 'sadness'], 'bluff': ['negative'],
               'blunder': ['disgust', 'negative', 'sadness'], 'blur': ['negative'], 'blurred': ['negative'],
               'blush': ['negative'], 'board': ['anticipation'], 'boast': ['negative', 'positive'],
               'boasting': ['negative'], 'bodyguard': ['positive', 'trust'], 'bog': ['negative'],
               'bogus': ['anger', 'disgust', 'negative'], 'boil': ['disgust', 'negative'], 'boilerplate': ['negative'],
               'boisterous': ['anger', 'anticipation', 'joy', 'negative', 'positive'], 'bold': ['positive'],
               'boldness': ['positive'], 'bolster': ['positive'],
               'bomb': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'bombard': ['anger', 'fear', 'negative'],
               'bombardment': ['anger', 'fear', 'negative'], 'bombed': ['disgust', 'negative'],
               'bomber': ['fear', 'sadness'], 'bonanza': ['joy', 'positive'],
               'bondage': ['fear', 'negative', 'sadness'], 'bonds': ['negative'], 'bonne': ['positive'],
               'bonus': ['anticipation', 'joy', 'positive', 'surprise'], 'boo': ['negative'], 'booby': ['negative'],
               'bookish': ['positive'], 'bookshop': ['positive'], 'bookworm': ['negative', 'positive'],
               'boomerang': ['anticipation', 'trust'], 'boon': ['positive'], 'booze': ['negative'],
               'bore': ['negative'], 'boredom': ['negative', 'sadness'], 'boring': ['negative'],
               'borrower': ['negative'], 'bother': ['negative'], 'bothering': ['anger', 'negative', 'sadness'],
               'bottom': ['negative', 'sadness'], 'bottomless': ['fear'], 'bound': ['negative'],
               'bountiful': ['anticipation', 'joy', 'positive'], 'bounty': ['anticipation', 'joy', 'positive', 'trust'],
               'bouquet': ['joy', 'positive', 'trust'], 'bout': ['anger', 'negative'],
               'bovine': ['disgust', 'negative'], 'bowels': ['disgust'], 'boxing': ['anger'],
               'boy': ['disgust', 'negative'], 'boycott': ['negative'], 'brag': ['negative'], 'brains': ['positive'],
               'bran': ['disgust'], 'brandy': ['negative'], 'bravado': ['negative'], 'bravery': ['positive'],
               'brawl': ['anger', 'disgust', 'fear', 'negative'], 'brazen': ['anger', 'negative'],
               'breach': ['negative'], 'break': ['surprise'], 'breakdown': ['negative'], 'breakfast': ['positive'],
               'breakneck': ['negative'], 'breakup': ['negative', 'sadness'], 'bribe': ['negative'],
               'bribery': ['disgust', 'negative'], 'bridal': ['anticipation', 'joy', 'positive', 'trust'],
               'bride': ['anticipation', 'joy', 'positive', 'trust'],
               'bridegroom': ['anticipation', 'joy', 'positive', 'trust'], 'bridesmaid': ['joy', 'positive', 'trust'],
               'brigade': ['fear', 'negative'], 'brighten': ['joy', 'positive', 'surprise', 'trust'],
               'brightness': ['positive'], 'brilliant': ['anticipation', 'joy', 'positive', 'trust'],
               'brimstone': ['anger', 'fear', 'negative'], 'bristle': ['negative'],
               'broadside': ['anticipation', 'negative'], 'brocade': ['positive'], 'broil': ['anger', 'negative'],
               'broke': ['fear', 'negative', 'sadness'], 'broken': ['anger', 'fear', 'negative', 'sadness'],
               'brothel': ['disgust', 'negative'], 'brother': ['positive', 'trust'],
               'brotherhood': ['positive', 'trust'], 'brotherly': ['anticipation', 'joy', 'positive', 'trust'],
               'bruise': ['anticipation', 'negative'], 'brunt': ['anger', 'negative'],
               'brutal': ['anger', 'fear', 'negative'], 'brutality': ['anger', 'fear', 'negative'],
               'brute': ['anger', 'fear', 'negative', 'sadness'], 'buck': ['fear', 'negative', 'positive', 'surprise'],
               'buddy': ['anticipation', 'joy', 'positive', 'trust'], 'budget': ['trust'],
               'buffet': ['anger', 'negative'], 'bug': ['disgust', 'fear', 'negative'],
               'bugaboo': ['anger', 'fear', 'negative', 'sadness'], 'bugle': ['anticipation'], 'build': ['positive'],
               'building': ['positive'], 'bulbous': ['negative'], 'bulldog': ['positive'], 'bulletproof': ['positive'],
               'bully': ['anger', 'fear', 'negative'], 'bum': ['disgust', 'negative', 'sadness'],
               'bummer': ['anger', 'disgust', 'negative'], 'bunker': ['fear'], 'buoy': ['positive'],
               'burdensome': ['fear', 'negative', 'sadness'], 'bureaucracy': ['negative', 'trust'],
               'bureaucrat': ['disgust', 'negative'], 'burglar': ['disgust', 'fear', 'negative'],
               'burglary': ['negative'], 'burial': ['anger', 'fear', 'negative', 'sadness'],
               'buried': ['fear', 'negative', 'sadness'], 'burke': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'burlesque': ['surprise'], 'burnt': ['disgust', 'negative'], 'bursary': ['trust'], 'bury': ['sadness'],
               'buss': ['joy', 'positive'], 'busted': ['anger', 'fear', 'negative'],
               'butcher': ['anger', 'disgust', 'fear', 'negative'], 'butler': ['positive', 'trust'],
               'butt': ['negative'], 'buttery': ['positive'], 'buxom': ['positive'],
               'buzz': ['anticipation', 'fear', 'positive'], 'buzzed': ['negative'], 'bye': ['anticipation'],
               'bylaw': ['trust'], 'cab': ['positive'], 'cabal': ['fear', 'negative'], 'cabinet': ['positive', 'trust'],
               'cable': ['surprise'], 'cacophony': ['anger', 'disgust', 'negative'],
               'cad': ['anger', 'disgust', 'negative'],
               'cadaver': ['disgust', 'fear', 'negative', 'sadness', 'surprise'], 'cafe': ['positive'],
               'cage': ['negative', 'sadness'], 'calamity': ['sadness'], 'calculating': ['negative'],
               'calculation': ['anticipation'], 'calculator': ['positive', 'trust'],
               'calf': ['joy', 'positive', 'trust'], 'callous': ['anger', 'disgust', 'negative'],
               'calls': ['anticipation', 'negative', 'trust'], 'calm': ['positive'], 'camouflage': ['surprise'],
               'camouflaged': ['surprise'], 'campaigning': ['anger', 'fear', 'negative'], 'canary': ['positive'],
               'cancel': ['negative', 'sadness'], 'cancer': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'candid': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'candidate': ['positive'],
               'candied': ['positive'], 'cane': ['anger', 'fear'], 'canker': ['anger', 'disgust', 'negative'],
               'cannibal': ['disgust', 'fear', 'negative'], 'cannibalism': ['disgust', 'negative'],
               'cannon': ['anger', 'fear', 'negative'], 'canons': ['trust'], 'cap': ['anticipation', 'trust'],
               'capitalist': ['positive'], 'captain': ['positive'],
               'captivate': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'captivating': ['positive'],
               'captive': ['fear', 'negative', 'sadness'], 'captivity': ['negative', 'sadness'],
               'captor': ['fear', 'negative'], 'capture': ['negative'],
               'carcass': ['disgust', 'fear', 'negative', 'sadness'], 'carcinoma': ['fear', 'negative', 'sadness'],
               'cardiomyopathy': ['fear', 'negative', 'sadness'], 'career': ['anticipation', 'positive'],
               'careful': ['positive'], 'carefully': ['positive'], 'carelessness': ['anger', 'disgust', 'negative'],
               'caress': ['positive'], 'caretaker': ['positive', 'trust'], 'caricature': ['negative'],
               'caries': ['disgust', 'negative'],
               'carnage': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'carnal': ['negative'],
               'carnivorous': ['fear', 'negative'], 'carol': ['joy', 'positive', 'trust'], 'cartel': ['negative'],
               'cartridge': ['fear'], 'cascade': ['positive'], 'case': ['fear', 'negative', 'sadness'],
               'cash': ['anger', 'anticipation', 'fear', 'joy', 'positive', 'trust'], 'cashier': ['trust'],
               'casket': ['fear', 'negative', 'sadness'], 'caste': ['negative'],
               'casualty': ['anger', 'fear', 'negative', 'sadness'],
               'cataract': ['anticipation', 'fear', 'negative', 'sadness'],
               'catastrophe': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'catch': ['surprise'],
               'catechism': ['disgust'], 'categorical': ['positive'], 'cater': ['positive'], 'cathartic': ['positive'],
               'cathedral': ['joy', 'positive', 'trust'], 'catheter': ['negative'],
               'caution': ['anger', 'anticipation', 'fear', 'negative'], 'cautionary': ['fear'],
               'cautious': ['anticipation', 'fear', 'positive', 'trust'], 'cautiously': ['fear', 'positive'],
               'cede': ['negative'], 'celebrated': ['anticipation', 'joy', 'positive'],
               'celebrating': ['anticipation', 'joy', 'positive'],
               'celebration': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'celebrity': ['anger', 'anticipation', 'disgust', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'celestial': ['anticipation', 'joy', 'positive'], 'cement': ['anticipation', 'trust'],
               'cemetery': ['fear', 'negative', 'sadness'], 'censor': ['anger', 'disgust', 'fear', 'negative', 'trust'],
               'censure': ['negative'], 'center': ['positive', 'trust'], 'centurion': ['positive'],
               'cerebral': ['positive'], 'ceremony': ['joy', 'positive', 'surprise'], 'certainty': ['positive'],
               'certify': ['trust'], 'cess': ['disgust', 'negative'], 'cessation': ['negative'],
               'chaff': ['anger', 'fear', 'negative'], 'chafing': ['negative'],
               'chagrin': ['disgust', 'negative', 'sadness'], 'chairman': ['positive', 'trust'],
               'chairwoman': ['positive', 'trust'], 'challenge': ['anger', 'fear', 'negative'],
               'champion': ['anticipation', 'joy', 'positive', 'trust'], 'chance': ['surprise'],
               'chancellor': ['trust'], 'change': ['fear'], 'changeable': ['anticipation', 'surprise'],
               'chant': ['anger', 'anticipation', 'joy', 'positive', 'surprise'],
               'chaos': ['anger', 'fear', 'negative', 'sadness'], 'chaotic': ['anger', 'negative'],
               'chaplain': ['trust'], 'charade': ['negative'], 'chargeable': ['fear', 'negative', 'sadness'],
               'charger': ['positive'], 'charitable': ['anticipation', 'joy', 'positive', 'trust'],
               'charity': ['joy', 'positive'], 'charm': ['positive'], 'charmed': ['joy', 'negative', 'positive'],
               'charming': ['positive'], 'chart': ['trust'], 'chase': ['negative'], 'chasm': ['fear'],
               'chastisement': ['negative'], 'chastity': ['anticipation', 'positive', 'trust'],
               'chattering': ['positive'], 'chatty': ['negative'], 'cheap': ['negative'],
               'cheat': ['anger', 'disgust', 'negative'], 'checklist': ['positive', 'trust'],
               'cheer': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'cheerful': ['joy', 'positive', 'surprise'],
               'cheerfulness': ['anticipation', 'joy', 'positive', 'trust'], 'cheering': ['joy', 'positive'],
               'cheery': ['anticipation', 'joy', 'positive'], 'cheesecake': ['negative'],
               'chemist': ['positive', 'trust'], 'cherish': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'cherry': ['positive'], 'chicane': ['anticipation', 'negative', 'surprise', 'trust'],
               'chicken': ['fear'], 'chieftain': ['positive'], 'child': ['anticipation', 'joy', 'positive'],
               'childhood': ['joy', 'positive'], 'childish': ['negative'], 'chilly': ['negative'],
               'chimera': ['fear', 'surprise'], 'chirp': ['joy', 'positive'], 'chisel': ['positive'],
               'chivalry': ['positive'], 'chloroform': ['negative'],
               'chocolate': ['anticipation', 'joy', 'positive', 'trust'], 'choice': ['positive'],
               'choir': ['joy', 'positive', 'trust'], 'choke': ['anger', 'negative', 'sadness'],
               'cholera': ['disgust', 'fear', 'negative', 'sadness'], 'chop': ['negative'],
               'choral': ['joy', 'positive'], 'chore': ['negative'], 'chorus': ['positive'], 'chosen': ['positive'],
               'chowder': ['positive'], 'chronic': ['negative', 'sadness'], 'chronicle': ['positive', 'trust'],
               'chuckle': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'church': ['anticipation', 'joy', 'positive', 'trust'], 'cider': ['positive'], 'cigarette': ['negative'],
               'circumcision': ['positive'], 'circumvention': ['negative', 'positive'], 'citizen': ['positive'],
               'civil': ['positive'], 'civility': ['positive'], 'civilization': ['positive', 'trust'],
               'civilized': ['joy', 'positive', 'trust'], 'claimant': ['anger', 'disgust'], 'clairvoyant': ['positive'],
               'clamor': ['anger', 'anticipation', 'disgust', 'negative', 'surprise'], 'clan': ['trust'],
               'clap': ['anticipation', 'joy', 'positive', 'trust'], 'clarify': ['positive'],
               'clash': ['anger', 'negative'], 'clashing': ['anger', 'fear', 'negative'], 'classic': ['positive'],
               'classical': ['positive'], 'classics': ['joy', 'positive'], 'classify': ['positive'],
               'claw': ['anger', 'fear', 'negative'], 'clean': ['joy', 'positive', 'trust'], 'cleaning': ['positive'],
               'cleanliness': ['positive'], 'cleanly': ['positive'], 'cleanse': ['positive'], 'cleansing': ['positive'],
               'clearance': ['positive', 'trust'], 'clearness': ['positive'], 'cleave': ['fear'],
               'clerical': ['positive', 'trust'], 'clever': ['positive'], 'cleverness': ['positive'], 'cliff': ['fear'],
               'climax': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'clock': ['anticipation'],
               'cloister': ['negative'], 'closeness': ['joy', 'positive', 'trust'],
               'closure': ['anticipation', 'joy', 'positive', 'sadness'], 'clothe': ['positive'],
               'clouded': ['negative', 'sadness'], 'cloudiness': ['fear', 'negative'], 'cloudy': ['sadness'],
               'clown': ['anticipation', 'joy', 'positive', 'surprise'], 'clue': ['anticipation'],
               'clump': ['negative'], 'clumsy': ['disgust', 'negative'], 'coach': ['trust'], 'coalesce': ['trust'],
               'coalition': ['positive'], 'coast': ['positive'], 'coax': ['trust'], 'cobra': ['fear'],
               'cocaine': ['negative', 'sadness'], 'coerce': ['anger', 'disgust', 'fear', 'negative'],
               'coercion': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'coexist': ['positive', 'trust'],
               'coexisting': ['trust'], 'coffin': ['fear', 'negative', 'sadness'], 'cogent': ['positive', 'trust'],
               'cognitive': ['positive'], 'coherence': ['positive'], 'coherent': ['positive'], 'cohesion': ['trust'],
               'cohesive': ['positive', 'trust'], 'coincidence': ['surprise'], 'cold': ['negative'],
               'coldly': ['negative'], 'coldness': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'colic': ['negative'], 'collaborator': ['trust'], 'collapse': ['disgust', 'fear', 'negative', 'sadness'],
               'collateral': ['trust'], 'collectively': ['positive', 'trust'], 'collision': ['anger', 'negative'],
               'collusion': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'colonel': ['positive', 'trust'],
               'colossal': ['positive'], 'coma': ['fear', 'negative', 'sadness'],
               'comatose': ['fear', 'negative', 'sadness'], 'combat': ['anger', 'fear', 'negative'],
               'combatant': ['anger', 'fear', 'negative'], 'combative': ['anger', 'fear', 'negative'],
               'comfort': ['anticipation', 'joy', 'positive', 'trust'], 'coming': ['anticipation'],
               'commandant': ['positive', 'trust'], 'commanding': ['positive', 'trust'],
               'commemorate': ['anticipation', 'joy', 'positive', 'sadness'],
               'commemoration': ['anticipation', 'joy', 'positive'], 'commemorative': ['anticipation', 'positive'],
               'commend': ['positive'], 'commendable': ['joy', 'positive', 'trust'], 'commentator': ['positive'],
               'commerce': ['trust'], 'commission': ['trust'], 'committal': ['negative', 'sadness'],
               'committed': ['positive', 'trust'], 'committee': ['trust'], 'commodore': ['positive', 'trust'],
               'commonplace': ['anticipation', 'trust'], 'commonwealth': ['positive', 'trust'],
               'commotion': ['anger', 'negative'], 'communicate': ['positive', 'trust'], 'communication': ['trust'],
               'communicative': ['positive'], 'communion': ['joy', 'positive', 'trust'],
               'communism': ['anger', 'fear', 'negative', 'sadness'], 'communist': ['negative'],
               'community': ['positive'], 'commutation': ['positive'], 'commute': ['positive'], 'compact': ['trust'],
               'companion': ['joy', 'positive', 'trust'], 'compass': ['trust'], 'compassion': ['fear', 'positive'],
               'compassionate': ['positive'], 'compatibility': ['positive'], 'compatible': ['positive'],
               'compelling': ['positive'], 'compensate': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'compensatory': ['positive'], 'competence': ['positive', 'trust'], 'competency': ['positive', 'trust'],
               'competent': ['positive', 'trust'], 'competition': ['anticipation', 'negative'],
               'complacency': ['positive'], 'complain': ['anger', 'negative', 'sadness'],
               'complaint': ['anger', 'negative'],
               'complement': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'complementary': ['positive'],
               'completely': ['positive'], 'completeness': ['positive'],
               'completing': ['anticipation', 'joy', 'positive'], 'completion': ['anticipation', 'joy', 'positive'],
               'complexed': ['negative'], 'complexity': ['negative'], 'compliance': ['positive', 'trust'],
               'compliant': ['positive'], 'complicate': ['anger', 'negative'], 'complicated': ['negative'],
               'complication': ['negative'], 'complicity': ['negative', 'positive'],
               'compliment': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'composed': ['positive'],
               'composer': ['positive'], 'compost': ['disgust', 'negative'], 'composure': ['positive'],
               'comprehend': ['positive'], 'comprehensive': ['positive'], 'compress': ['anger'],
               'comptroller': ['trust'], 'compulsion': ['anger', 'negative'], 'compulsory': ['negative'],
               'comrade': ['positive', 'trust'], 'conceal': ['negative', 'sadness'],
               'concealed': ['anticipation', 'fear', 'negative', 'surprise'],
               'concealment': ['anger', 'anticipation', 'fear', 'negative'], 'conceit': ['negative'],
               'conceited': ['negative'], 'concentric': ['positive'], 'concerned': ['fear', 'sadness'],
               'conciliation': ['joy', 'positive', 'trust'], 'concluding': ['positive'],
               'concord': ['positive', 'trust'], 'concordance': ['positive', 'trust'],
               'concussion': ['anger', 'negative', 'sadness'], 'condemn': ['anger', 'negative'],
               'condemnation': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'condescending': ['negative'], 'condescension': ['anger', 'disgust', 'negative', 'sadness'],
               'condolence': ['positive', 'sadness'], 'condone': ['positive'], 'conducive': ['positive'],
               'conductivity': ['positive'], 'confederate': ['positive', 'trust'],
               'confess': ['negative', 'positive', 'trust'],
               'confession': ['anticipation', 'fear', 'negative', 'sadness', 'surprise'],
               'confessional': ['fear', 'trust'], 'confide': ['trust'],
               'confidence': ['fear', 'joy', 'positive', 'trust'], 'confident': ['joy', 'positive', 'trust'],
               'confidential': ['trust'], 'confidentially': ['trust'],
               'confine': ['anger', 'fear', 'negative', 'sadness'],
               'confined': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'confinement': ['anger', 'fear', 'negative', 'sadness'], 'confirmation': ['trust'],
               'confirmed': ['positive', 'trust'], 'confiscate': ['anger', 'negative', 'sadness'],
               'confiscation': ['negative'], 'conflagration': ['anger', 'fear', 'negative'],
               'conflict': ['anger', 'fear', 'negative', 'sadness'], 'conflicting': ['negative'],
               'conformance': ['positive'], 'conformity': ['trust'], 'confound': ['negative'],
               'confounded': ['negative'], 'confront': ['anger'], 'confuse': ['negative'],
               'confusion': ['anger', 'fear', 'negative'], 'congenial': ['positive'], 'congestion': ['negative'],
               'conglomerate': ['trust'], 'congratulatory': ['joy', 'positive'], 'congregation': ['positive', 'trust'],
               'congress': ['disgust', 'trust'], 'congressman': ['trust'], 'congruence': ['positive', 'trust'],
               'conjecture': ['anticipation'], 'conjure': ['anticipation', 'surprise'], 'conjuring': ['negative'],
               'connective': ['trust'], 'connoisseur': ['joy', 'positive', 'trust'],
               'conquest': ['anger', 'fear', 'negative'], 'conscience': ['positive', 'trust'],
               'conscientious': ['positive', 'trust'], 'consciousness': ['positive'],
               'consecration': ['anticipation', 'joy', 'positive', 'sadness', 'trust'], 'consequent': ['anticipation'],
               'conservation': ['anticipation', 'positive', 'trust'], 'conserve': ['positive'],
               'considerable': ['positive'], 'considerate': ['positive', 'trust'], 'consistency': ['positive', 'trust'],
               'console': ['positive', 'sadness'], 'consonant': ['positive'], 'consort': ['trust'],
               'conspiracy': ['fear'], 'conspirator': ['anger', 'anticipation', 'disgust', 'fear', 'negative'],
               'conspire': ['fear', 'negative'], 'constable': ['trust'], 'constancy': ['positive', 'trust'],
               'constant': ['positive', 'trust'], 'constantly': ['trust'],
               'consternation': ['anger', 'fear', 'negative'], 'constipation': ['disgust', 'negative'],
               'constitute': ['trust'], 'constitutional': ['positive', 'trust'], 'constrain': ['fear', 'negative'],
               'constrained': ['negative'], 'constraint': ['anger', 'fear', 'negative', 'sadness'],
               'construct': ['positive'], 'consul': ['trust'], 'consult': ['trust'], 'consummate': ['positive'],
               'contact': ['positive'], 'contagion': ['anticipation', 'disgust', 'fear', 'negative'],
               'contagious': ['disgust', 'fear', 'negative'], 'contaminate': ['disgust', 'negative'],
               'contaminated': ['disgust', 'fear', 'negative', 'sadness'], 'contamination': ['disgust', 'negative'],
               'contemplation': ['positive'], 'contempt': ['anger', 'disgust', 'fear', 'negative'],
               'contemptible': ['anger', 'disgust', 'negative'], 'contemptuous': ['anger', 'negative'],
               'content': ['joy', 'positive', 'trust'], 'contentious': ['anger', 'disgust', 'fear', 'negative'],
               'contingent': ['anticipation'], 'continuation': ['anticipation'],
               'continue': ['anticipation', 'positive', 'trust'], 'contour': ['positive'],
               'contraband': ['anger', 'disgust', 'fear', 'negative'], 'contracted': ['negative'],
               'contradict': ['anger', 'negative'], 'contradiction': ['negative'], 'contradictory': ['negative'],
               'contrary': ['negative'], 'contrasted': ['negative'], 'contravene': ['negative'],
               'contravention': ['negative'], 'contribute': ['positive'], 'contributor': ['positive', 'trust'],
               'controversial': ['anger', 'negative'], 'controversy': ['negative'], 'convenience': ['positive'],
               'convenient': ['positive'], 'convent': ['positive', 'trust'], 'convention': ['positive'],
               'convergence': ['anticipation'], 'conversant': ['positive'], 'conversational': ['positive'],
               'convert': ['positive'], 'conveyancing': ['trust'],
               'convict': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'conviction': ['negative'],
               'convince': ['anticipation', 'positive', 'trust'], 'convinced': ['trust'], 'convincing': ['trust'],
               'cool': ['positive'], 'coolness': ['positive'], 'coop': ['anger', 'disgust', 'negative'],
               'cooperate': ['positive'], 'cooperating': ['positive', 'trust'], 'cooperation': ['positive', 'trust'],
               'cooperative': ['positive', 'trust'], 'cop': ['fear', 'trust'], 'copy': ['negative'],
               'copycat': ['anger', 'disgust', 'negative'], 'core': ['positive'],
               'coronation': ['joy', 'positive', 'trust'], 'coroner': ['negative'], 'corporal': ['negative'],
               'corporation': ['positive', 'trust'], 'corporeal': ['positive'],
               'corpse': ['disgust', 'negative', 'sadness'], 'correction': ['negative'], 'corrective': ['positive'],
               'correctness': ['trust'], 'correspondence': ['anticipation', 'positive'],
               'corroborate': ['positive', 'trust'], 'corroboration': ['trust'], 'corrosion': ['negative'],
               'corrosive': ['fear', 'negative'], 'corrupt': ['negative'],
               'corrupting': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'corruption': ['disgust', 'negative'],
               'corse': ['sadness'], 'cosmopolitan': ['positive', 'trust'], 'cosy': ['positive'], 'couch': ['sadness'],
               'cough': ['disgust', 'negative'], 'council': ['anticipation', 'positive', 'trust'],
               'counsel': ['positive', 'trust'], 'counsellor': ['anger', 'fear', 'negative', 'trust'],
               'counselor': ['positive', 'trust'], 'count': ['positive', 'trust'], 'countdown': ['anticipation'],
               'countess': ['positive'], 'countryman': ['trust'], 'county': ['trust'], 'coup': ['anger', 'surprise'],
               'courage': ['positive'], 'courageous': ['fear', 'positive'], 'courier': ['trust'],
               'coursing': ['negative'], 'court': ['anger', 'anticipation', 'fear'], 'courteous': ['positive'],
               'courtesy': ['positive'], 'courtship': ['anticipation', 'joy', 'positive', 'trust'],
               'cove': ['anticipation', 'disgust', 'fear', 'joy', 'positive'], 'covenant': ['positive', 'trust'],
               'cover': ['trust'], 'covet': ['negative'], 'coward': ['disgust', 'fear', 'negative', 'sadness'],
               'cowardice': ['fear', 'negative'], 'cowardly': ['fear', 'negative'], 'coy': ['fear'], 'coyote': ['fear'],
               'crabby': ['anger', 'negative'], 'crack': ['negative'], 'cracked': ['anger', 'fear', 'negative'],
               'cracking': ['negative'], 'cradle': ['anticipation', 'joy', 'positive', 'trust'], 'craft': ['positive'],
               'craftsman': ['positive'], 'cramp': ['anticipation', 'negative'], 'cramped': ['negative'],
               'crank': ['negative'], 'cranky': ['anger', 'negative'], 'crap': ['disgust', 'negative'],
               'craps': ['anticipation'], 'crash': ['fear', 'negative', 'sadness', 'surprise'],
               'crave': ['anticipation'], 'craving': ['anticipation'], 'crawl': ['disgust', 'negative'],
               'crazed': ['anger', 'fear', 'negative'], 'crazy': ['anger', 'fear', 'negative', 'sadness'],
               'creaking': ['negative'], 'cream': ['anticipation', 'joy', 'positive', 'surprise'],
               'create': ['joy', 'positive'], 'creative': ['positive'], 'creature': ['disgust', 'fear', 'negative'],
               'credence': ['positive', 'trust'], 'credential': ['positive', 'trust'],
               'credibility': ['positive', 'trust'], 'credible': ['positive', 'trust'], 'credit': ['positive', 'trust'],
               'creditable': ['positive', 'trust'], 'credited': ['positive'], 'creep': ['negative'],
               'creeping': ['anticipation'], 'cremation': ['sadness'],
               'crescendo': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'crew': ['trust'],
               'crime': ['anger', 'negative'], 'criminal': ['anger', 'disgust', 'fear', 'negative'],
               'criminality': ['anger', 'disgust', 'fear', 'negative'],
               'cringe': ['disgust', 'fear', 'negative', 'sadness'], 'cripple': ['fear', 'negative', 'sadness'],
               'crippled': ['negative', 'sadness'], 'crisis': ['negative'], 'crisp': ['negative', 'trust'],
               'critic': ['negative'], 'criticism': ['anger', 'negative', 'sadness'],
               'criticize': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'critique': ['positive'],
               'critter': ['disgust'], 'crocodile': ['fear'], 'crook': ['negative'],
               'cross': ['anger', 'fear', 'negative', 'sadness'], 'crouch': ['fear'], 'crouching': ['fear', 'negative'],
               'crowning': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'crucial': ['positive', 'trust'],
               'cruciate': ['negative'], 'crucifixion': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'crude': ['disgust', 'negative'], 'cruel': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'cruelly': ['anger', 'fear', 'negative'], 'cruelty': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'crumbling': ['negative', 'sadness'], 'crunch': ['anger', 'negative'],
               'crusade': ['anger', 'fear', 'negative'], 'crushed': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'crushing': ['anger', 'disgust', 'fear', 'negative'], 'crusty': ['disgust', 'negative'],
               'cry': ['negative', 'sadness'], 'crying': ['negative', 'sadness'],
               'crypt': ['fear', 'negative', 'sadness'], 'crystal': ['positive'], 'cube': ['trust'],
               'cuckold': ['disgust', 'negative'], 'cuckoo': ['negative'], 'cuddle': ['joy', 'positive', 'trust'],
               'cue': ['anticipation'], 'culinary': ['positive', 'trust'], 'cull': ['negative'],
               'culmination': ['positive'], 'culpability': ['negative'], 'culpable': ['negative'],
               'culprit': ['negative'], 'cult': ['fear', 'negative'],
               'cultivate': ['anticipation', 'positive', 'trust'], 'cultivated': ['positive'],
               'cultivation': ['positive'], 'culture': ['positive'], 'cumbersome': ['negative', 'sadness'],
               'cunning': ['negative', 'positive'], 'cupping': ['disgust', 'fear', 'negative', 'sadness'],
               'cur': ['anger', 'disgust', 'fear', 'negative'], 'curable': ['positive', 'trust'],
               'curiosity': ['anticipation', 'positive', 'surprise'], 'curl': ['positive'],
               'curse': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'cursed': ['anger', 'fear', 'negative', 'sadness'], 'cursing': ['anger', 'disgust', 'negative'],
               'cursory': ['negative'], 'cushion': ['positive'], 'cussed': ['anger'], 'custodian': ['trust'],
               'custody': ['trust'], 'customer': ['positive'], 'cute': ['positive'], 'cutter': ['fear', 'negative'],
               'cutters': ['positive'], 'cutthroat': ['anger', 'fear', 'negative'],
               'cutting': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'cyanide': ['fear', 'negative'],
               'cyclone': ['fear', 'negative', 'surprise'], 'cyst': ['fear', 'negative', 'sadness'],
               'cystic': ['disgust'], 'cytomegalovirus': ['negative', 'sadness'],
               'dabbling': ['anger', 'disgust', 'negative'],
               'daemon': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'daft': ['disgust', 'negative'], 'dagger': ['fear', 'negative'], 'daily': ['anticipation'],
               'damage': ['anger', 'disgust', 'negative', 'sadness'], 'damages': ['negative', 'sadness'],
               'dame': ['anger', 'disgust', 'positive', 'trust'], 'damn': ['anger', 'disgust', 'negative'],
               'damnation': ['anger', 'fear', 'negative', 'sadness'], 'damned': ['negative'], 'damper': ['negative'],
               'dance': ['joy', 'positive', 'trust'], 'dandruff': ['negative'], 'dandy': ['disgust', 'negative'],
               'danger': ['fear', 'negative', 'sadness'], 'dangerous': ['fear', 'negative'], 'dank': ['disgust'],
               'dare': ['anticipation', 'trust'], 'daring': ['positive'], 'dark': ['sadness'],
               'darken': ['fear', 'negative', 'sadness'], 'darkened': ['fear', 'negative', 'sadness'],
               'darkness': ['anger', 'fear', 'negative', 'sadness'], 'darling': ['joy', 'positive', 'trust'],
               'dart': ['fear'], 'dashed': ['anger', 'fear', 'negative', 'sadness'], 'dashing': ['positive'],
               'dastardly': ['anger', 'disgust', 'fear', 'negative'], 'daughter': ['joy', 'positive'],
               'dawn': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'dazed': ['negative'],
               'deacon': ['trust'], 'deactivate': ['negative'], 'deadlock': ['negative'],
               'deadly': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'deaf': ['negative'],
               'deal': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'dealings': ['trust'],
               'dear': ['positive'],
               'death': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'debacle': ['fear', 'negative', 'sadness'], 'debate': ['positive'],
               'debauchery': ['disgust', 'fear', 'negative'], 'debenture': ['anticipation'],
               'debris': ['disgust', 'negative'], 'debt': ['negative', 'sadness'], 'debtor': ['negative'],
               'decay': ['fear', 'negative', 'sadness'], 'decayed': ['disgust', 'negative', 'sadness'],
               'deceased': ['negative', 'sadness'],
               'deceit': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'deceitful': ['disgust', 'negative', 'sadness'], 'deceive': ['anger', 'disgust', 'negative', 'sadness'],
               'deceived': ['anger', 'negative'], 'deceiving': ['negative', 'trust'], 'decency': ['positive'],
               'decent': ['positive'], 'deception': ['negative'], 'deceptive': ['negative'],
               'declaratory': ['positive'], 'declination': ['negative'], 'decline': ['negative'],
               'declining': ['negative'], 'decompose': ['disgust'], 'decomposed': ['sadness'],
               'decomposition': ['disgust', 'fear', 'negative', 'sadness'], 'decoy': ['surprise'],
               'decrease': ['negative'], 'decrement': ['negative'], 'decrepit': ['negative'],
               'decry': ['anger', 'negative'], 'dedication': ['positive'], 'deduct': ['negative'], 'deed': ['trust'],
               'defamation': ['disgust', 'fear', 'negative'], 'defamatory': ['anger', 'negative'],
               'default': ['disgust', 'fear', 'negative', 'sadness'], 'defeat': ['negative'],
               'defeated': ['negative', 'sadness'], 'defect': ['anger', 'negative'], 'defection': ['fear', 'negative'],
               'defective': ['disgust', 'negative'], 'defend': ['fear', 'positive'],
               'defendant': ['anger', 'fear', 'sadness'], 'defended': ['positive', 'trust'],
               'defender': ['positive', 'trust'], 'defending': ['positive'],
               'defense': ['anger', 'anticipation', 'fear', 'positive'], 'defenseless': ['fear', 'negative', 'sadness'],
               'deference': ['positive', 'trust'], 'deferral': ['negative'],
               'defiance': ['anger', 'disgust', 'fear', 'negative'], 'defiant': ['anger', 'negative'],
               'deficiency': ['negative'], 'deficit': ['negative'], 'definitive': ['positive', 'trust'],
               'deflate': ['anger', 'negative', 'sadness'], 'deflation': ['fear', 'negative'],
               'deform': ['disgust', 'negative'], 'deformed': ['disgust', 'negative', 'sadness'],
               'deformity': ['disgust', 'fear', 'negative', 'sadness'], 'defraud': ['anger', 'disgust', 'negative'],
               'defunct': ['negative', 'sadness'], 'defy': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'degeneracy': ['anger', 'disgust', 'negative', 'sadness'], 'degenerate': ['negative'],
               'degradation': ['negative'], 'degrade': ['disgust', 'negative'],
               'degrading': ['disgust', 'fear', 'negative', 'sadness'], 'degree': ['positive'],
               'delay': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'delayed': ['negative'],
               'delectable': ['positive'], 'delegate': ['positive', 'trust'],
               'deleterious': ['anger', 'disgust', 'fear', 'negative'], 'deletion': ['negative'],
               'deliberate': ['positive'], 'delicious': ['joy', 'positive'],
               'delight': ['anticipation', 'joy', 'positive'],
               'delighted': ['anticipation', 'joy', 'positive', 'surprise'],
               'delightful': ['anticipation', 'joy', 'positive', 'trust'], 'delinquency': ['negative'],
               'delinquent': ['anger', 'disgust', 'negative'], 'delirious': ['negative', 'sadness'],
               'delirium': ['disgust', 'negative', 'sadness'],
               'deliverance': ['anticipation', 'joy', 'positive', 'trust'], 'delivery': ['anticipation', 'positive'],
               'deluge': ['fear', 'negative', 'sadness', 'surprise'],
               'delusion': ['anger', 'fear', 'negative', 'sadness'], 'delusional': ['anger', 'fear', 'negative'],
               'demand': ['anger', 'negative'], 'demanding': ['negative'], 'demented': ['fear', 'negative'],
               'dementia': ['fear', 'negative', 'sadness'], 'demise': ['fear', 'negative', 'sadness'],
               'democracy': ['positive'], 'demolish': ['anger', 'negative', 'sadness'], 'demolition': ['negative'],
               'demon': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'demonic': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'demonstrable': ['positive'],
               'demonstrative': ['joy', 'positive', 'sadness'], 'demoralized': ['fear', 'negative', 'sadness'],
               'denial': ['negative'], 'denied': ['negative', 'sadness'], 'denounce': ['anger', 'disgust', 'negative'],
               'dentistry': ['fear'], 'denunciation': ['anger', 'disgust', 'fear', 'negative'],
               'deny': ['anger', 'negative'], 'denying': ['anticipation', 'negative'],
               'depart': ['anticipation', 'sadness'], 'departed': ['negative', 'sadness'],
               'departure': ['negative', 'sadness'], 'depend': ['anticipation', 'trust'],
               'dependence': ['fear', 'negative', 'sadness'], 'dependency': ['negative'],
               'dependent': ['negative', 'positive', 'trust'],
               'deplorable': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'deplore': ['anger', 'disgust', 'negative', 'sadness'], 'deport': ['fear', 'negative', 'sadness'],
               'deportation': ['anger', 'fear', 'negative', 'sadness'], 'depository': ['trust'],
               'depraved': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'depravity': ['anger', 'disgust', 'negative'], 'depreciate': ['anger', 'disgust', 'negative'],
               'depreciated': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'depreciation': ['fear', 'negative'],
               'depress': ['fear', 'negative', 'sadness'], 'depressed': ['anger', 'fear', 'negative', 'sadness'],
               'depressing': ['disgust', 'negative', 'sadness'], 'depression': ['negative', 'sadness'],
               'depressive': ['negative', 'sadness'],
               'deprivation': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'depth': ['positive', 'trust'],
               'deputy': ['trust'], 'deranged': ['anger', 'disgust', 'fear', 'negative'], 'derelict': ['negative'],
               'derision': ['anger', 'disgust', 'negative'], 'dermatologist': ['trust'],
               'derogation': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'derogatory': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'descent': ['fear', 'sadness'],
               'descriptive': ['positive'], 'desecration': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'desert': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'deserted': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'desertion': ['negative'],
               'deserve': ['anger', 'anticipation', 'positive', 'trust'], 'deserved': ['positive'],
               'designation': ['trust'], 'designer': ['positive'], 'desirable': ['positive'], 'desiring': ['positive'],
               'desirous': ['positive'], 'desist': ['anger', 'disgust', 'negative'],
               'desolation': ['fear', 'negative', 'sadness'],
               'despair': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'despairing': ['fear', 'negative', 'sadness'], 'desperate': ['negative'],
               'despicable': ['anger', 'disgust', 'negative'], 'despise': ['anger', 'disgust', 'negative'],
               'despotic': ['fear', 'negative'], 'despotism': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'destination': ['anticipation', 'fear', 'joy', 'positive', 'sadness', 'surprise'],
               'destined': ['anticipation'], 'destitute': ['fear', 'negative', 'sadness'],
               'destroyed': ['anger', 'fear', 'negative', 'sadness'], 'destroyer': ['anger', 'fear', 'negative'],
               'destroying': ['anger', 'fear', 'negative', 'sadness'], 'destruction': ['anger', 'negative'],
               'destructive': ['anger', 'disgust', 'fear', 'negative'], 'detachment': ['negative'],
               'detain': ['negative'], 'detainee': ['anger', 'anticipation', 'fear', 'negative', 'sadness'],
               'detect': ['positive'], 'detection': ['positive'], 'detention': ['negative', 'sadness'],
               'deteriorate': ['fear', 'negative', 'sadness'], 'deteriorated': ['disgust', 'negative', 'sadness'],
               'deterioration': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'determinate': ['anticipation', 'trust'], 'determination': ['positive', 'trust'],
               'determined': ['positive'], 'detest': ['anger', 'disgust', 'negative'],
               'detonate': ['fear', 'negative', 'surprise'], 'detonation': ['anger'], 'detract': ['anger', 'negative'],
               'detriment': ['negative'], 'detrimental': ['negative'], 'detritus': ['negative'],
               'devastate': ['anger', 'fear', 'negative', 'sadness'],
               'devastating': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'trust'],
               'devastation': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'develop': ['anticipation', 'positive'], 'deviation': ['sadness'],
               'devil': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'devilish': ['disgust', 'fear', 'negative'], 'devious': ['negative'], 'devolution': ['negative'],
               'devotional': ['positive', 'trust'], 'devour': ['negative'],
               'devout': ['anticipation', 'joy', 'positive', 'trust'], 'dexterity': ['positive'],
               'diabolical': ['anger', 'disgust', 'fear', 'negative'],
               'diagnosis': ['anticipation', 'fear', 'negative', 'trust'], 'diamond': ['joy', 'positive'],
               'diaper': ['disgust'], 'diarrhoea': ['disgust'], 'diary': ['joy', 'positive', 'trust'],
               'diatribe': ['anger', 'disgust', 'negative'], 'dictator': ['fear', 'negative'],
               'dictatorial': ['anger', 'negative'],
               'dictatorship': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'dictionary': ['positive', 'trust'], 'dictum': ['trust'], 'didactic': ['positive'],
               'die': ['fear', 'negative', 'sadness'], 'dietary': ['anticipation', 'positive'],
               'differential': ['trust'], 'differently': ['surprise'], 'difficult': ['fear'],
               'difficulties': ['negative', 'sadness'], 'difficulty': ['anger', 'fear', 'negative', 'sadness'],
               'digit': ['trust'], 'dignified': ['positive'], 'dignity': ['positive', 'trust'],
               'digress': ['anticipation', 'negative'], 'dike': ['fear'],
               'dilapidated': ['disgust', 'negative', 'sadness'], 'diligence': ['positive', 'trust'],
               'dilute': ['negative'], 'diminish': ['negative', 'sadness'], 'diminished': ['negative'],
               'din': ['negative'], 'dinner': ['positive'], 'dinosaur': ['fear'],
               'diplomacy': ['anticipation', 'positive', 'trust'], 'diplomatic': ['positive', 'trust'],
               'dire': ['disgust', 'fear', 'negative', 'sadness', 'surprise'], 'director': ['positive', 'trust'],
               'dirt': ['disgust', 'negative'], 'dirty': ['disgust', 'negative'], 'disability': ['negative', 'sadness'],
               'disable': ['fear', 'negative', 'sadness'], 'disabled': ['fear', 'negative', 'sadness'],
               'disaffected': ['negative'], 'disagree': ['anger', 'negative'],
               'disagreeing': ['anger', 'negative', 'sadness'], 'disagreement': ['anger', 'negative', 'sadness'],
               'disallowed': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'disappear': ['fear'],
               'disappoint': ['anger', 'disgust', 'negative', 'sadness'],
               'disappointed': ['anger', 'disgust', 'negative', 'sadness'], 'disappointing': ['negative', 'sadness'],
               'disappointment': ['disgust', 'negative', 'sadness'], 'disapproval': ['negative', 'sadness'],
               'disapprove': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'disapproved': ['anger', 'negative', 'sadness'],
               'disapproving': ['anger', 'disgust', 'negative', 'sadness'],
               'disaster': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'disastrous': ['anger', 'fear', 'negative', 'sadness'], 'disbelieve': ['negative'],
               'discards': ['negative'], 'discharge': ['negative'], 'disciple': ['trust'],
               'discipline': ['fear', 'negative'], 'disclaim': ['anger', 'disgust', 'negative', 'trust'],
               'disclosed': ['trust'], 'discoloration': ['disgust', 'negative'],
               'discolored': ['disgust', 'negative', 'sadness'], 'discomfort': ['negative', 'sadness'],
               'disconnect': ['negative', 'sadness'], 'disconnected': ['negative', 'sadness'],
               'disconnection': ['negative'], 'discontent': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'discontinue': ['negative'], 'discontinuity': ['disgust', 'fear', 'negative', 'sadness'],
               'discord': ['anger', 'disgust', 'negative'], 'discourage': ['fear', 'negative', 'sadness'],
               'discouragement': ['negative'], 'discovery': ['positive'], 'discredit': ['negative'],
               'discreet': ['anticipation', 'positive'], 'discretion': ['anticipation', 'positive', 'trust'],
               'discretionary': ['positive'], 'discriminate': ['anger', 'negative', 'sadness'],
               'discriminating': ['disgust', 'negative'],
               'discrimination': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'discussion': ['positive'],
               'disdain': ['anger', 'disgust', 'negative'],
               'disease': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'diseased': ['disgust', 'fear', 'negative', 'sadness'], 'disembodied': ['fear', 'negative', 'sadness'],
               'disengagement': ['negative'], 'disfigured': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'disgrace': ['anger', 'disgust', 'negative', 'sadness'],
               'disgraced': ['anger', 'disgust', 'negative', 'sadness'],
               'disgraceful': ['anger', 'disgust', 'negative'],
               'disgruntled': ['anger', 'disgust', 'negative', 'sadness'],
               'disgust': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'disgusting': ['anger', 'disgust', 'fear', 'negative'], 'disheartened': ['negative', 'sadness'],
               'disheartening': ['negative', 'sadness'], 'dishonest': ['anger', 'disgust', 'negative', 'sadness'],
               'dishonesty': ['disgust', 'negative'], 'dishonor': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'disillusionment': ['anger', 'disgust', 'negative', 'sadness'], 'disinfection': ['positive'],
               'disinformation': ['anger', 'fear', 'negative'], 'disingenuous': ['disgust', 'negative'],
               'disintegrate': ['disgust', 'fear', 'negative'], 'disintegration': ['negative'],
               'disinterested': ['negative'], 'dislike': ['anger', 'disgust', 'negative'],
               'disliked': ['anger', 'negative', 'sadness'],
               'dislocated': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'dismal': ['disgust', 'fear', 'negative', 'sadness'],
               'dismay': ['anger', 'anticipation', 'fear', 'negative', 'sadness', 'surprise'],
               'dismemberment': ['disgust', 'fear', 'negative', 'sadness'],
               'dismissal': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'disobedience': ['anger', 'disgust', 'negative'], 'disobedient': ['anger', 'negative'],
               'disobey': ['anger', 'disgust', 'negative'], 'disorder': ['fear', 'negative'],
               'disorderly': ['negative'], 'disorganized': ['negative'],
               'disparage': ['anger', 'disgust', 'negative', 'sadness'],
               'disparaging': ['anger', 'disgust', 'negative', 'sadness'],
               'disparity': ['anger', 'disgust', 'negative', 'sadness'], 'dispassionate': ['negative', 'sadness'],
               'dispel': ['negative', 'sadness'], 'dispersion': ['negative'], 'displace': ['negative'],
               'displaced': ['anger', 'fear', 'sadness'],
               'displeased': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'displeasure': ['disgust', 'negative'], 'disposal': ['negative'], 'dispose': ['disgust'],
               'disposed': ['anticipation', 'positive', 'trust'],
               'dispossessed': ['anger', 'fear', 'negative', 'sadness'], 'dispute': ['anger', 'negative'],
               'disqualification': ['negative'], 'disqualified': ['anger', 'disgust', 'negative', 'sadness'],
               'disqualify': ['negative', 'sadness'], 'disregard': ['negative'], 'disregarded': ['disgust', 'negative'],
               'disreputable': ['anger', 'disgust', 'fear', 'negative'], 'disrespect': ['anger', 'negative'],
               'disrespectful': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'disruption': ['anger', 'fear', 'negative', 'surprise'], 'dissatisfaction': ['negative'],
               'dissection': ['disgust'], 'disseminate': ['positive'], 'dissension': ['anger', 'negative'],
               'dissenting': ['negative'], 'disservice': ['anger', 'disgust', 'negative', 'sadness'],
               'dissident': ['anger', 'fear', 'negative'],
               'dissolution': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'dissonance': ['anger', 'negative'],
               'distaste': ['disgust', 'negative'], 'distasteful': ['disgust', 'negative'],
               'distillation': ['positive'], 'distinction': ['positive'], 'distorted': ['disgust', 'negative'],
               'distortion': ['negative'], 'distract': ['negative'], 'distracted': ['anger', 'negative'],
               'distracting': ['anger', 'anticipation', 'negative'], 'distraction': ['negative'],
               'distraught': ['negative', 'sadness'],
               'distress': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'distressed': ['fear', 'negative'], 'distressing': ['anger', 'fear', 'negative'],
               'distrust': ['anger', 'disgust', 'fear', 'negative'],
               'disturbance': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'disturbed': ['anger', 'negative', 'sadness'], 'disuse': ['negative'], 'disused': ['anger', 'negative'],
               'ditty': ['joy', 'positive'], 'divan': ['trust'], 'divergent': ['negative', 'surprise'],
               'diverse': ['negative', 'positive'], 'diversified': ['positive'], 'diversion': ['positive', 'surprise'],
               'divested': ['negative'], 'divestment': ['negative'], 'divination': ['anticipation'],
               'divinity': ['positive'],
               'divorce': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise', 'trust'],
               'dizziness': ['negative'], 'dizzy': ['negative'], 'docked': ['negative'],
               'doctor': ['positive', 'trust'], 'doctrine': ['trust'], 'doer': ['positive'], 'dogged': ['positive'],
               'dogma': ['trust'], 'doit': ['negative'], 'doldrums': ['negative', 'sadness'],
               'dole': ['negative', 'sadness'], 'doll': ['joy'], 'dolor': ['negative', 'sadness'],
               'dolphin': ['joy', 'positive', 'surprise', 'trust'], 'dominant': ['fear', 'negative'],
               'dominate': ['anger', 'fear', 'negative', 'positive'],
               'domination': ['anger', 'fear', 'negative', 'sadness'], 'dominion': ['fear', 'trust'],
               'don': ['positive', 'trust'], 'donation': ['positive'], 'donkey': ['disgust', 'negative'],
               'doodle': ['negative'], 'doom': ['fear', 'negative'], 'doomed': ['fear', 'negative', 'sadness'],
               'doomsday': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'doubt': ['fear', 'negative', 'sadness', 'trust'], 'doubtful': ['negative'], 'doubting': ['negative'],
               'doubtless': ['positive', 'trust'], 'douche': ['negative'], 'dour': ['negative'],
               'dove': ['anticipation', 'joy', 'positive', 'trust'], 'downfall': ['fear', 'negative', 'sadness'],
               'downright': ['trust'], 'downy': ['positive'], 'drab': ['negative', 'sadness'],
               'draft': ['anticipation'], 'dragon': ['fear'], 'drainage': ['negative'], 'drawback': ['negative'],
               'dread': ['anticipation', 'fear', 'negative'],
               'dreadful': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'dreadfully': ['disgust', 'fear', 'negative', 'sadness', 'surprise'], 'dreary': ['negative', 'sadness'],
               'drinking': ['negative'], 'drivel': ['disgust', 'negative'], 'drone': ['negative'], 'drool': ['disgust'],
               'drooping': ['negative'], 'drought': ['negative'], 'drown': ['fear', 'negative', 'sadness'],
               'drowsiness': ['negative'], 'drudgery': ['negative'], 'drugged': ['sadness'],
               'drunken': ['disgust', 'negative'], 'drunkenness': ['negative'],
               'dubious': ['fear', 'negative', 'trust'], 'duel': ['anger', 'anticipation', 'fear'],
               'duet': ['positive'], 'duke': ['positive'], 'dull': ['negative', 'sadness'], 'dumb': ['negative'],
               'dummy': ['negative'], 'dumps': ['anger', 'negative', 'sadness'], 'dun': ['negative'],
               'dung': ['disgust'], 'dungeon': ['fear', 'negative'], 'dupe': ['anger', 'negative'],
               'duplicity': ['anger', 'negative'], 'durability': ['positive', 'trust'],
               'durable': ['positive', 'trust'], 'duress': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'dust': ['negative'], 'dutiful': ['anticipation', 'positive', 'trust'],
               'dwarfed': ['fear', 'negative', 'sadness'], 'dying': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'dynamic': ['surprise'], 'dysentery': ['disgust', 'negative', 'sadness'],
               'eager': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'eagerness': ['anticipation', 'joy', 'positive', 'trust'], 'eagle': ['trust'], 'earl': ['positive'],
               'earn': ['positive'], 'earnest': ['positive'], 'earnestly': ['positive'], 'earnestness': ['positive'],
               'earthquake': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'ease': ['positive'],
               'easement': ['positive'], 'easygoing': ['positive'], 'eat': ['positive'], 'eavesdropping': ['negative'],
               'economy': ['trust'], 'ecstasy': ['anticipation', 'joy', 'positive'],
               'ecstatic': ['anticipation', 'joy', 'positive', 'surprise'], 'edict': ['fear', 'negative'],
               'edification': ['anticipation', 'joy', 'positive', 'trust'], 'edition': ['anticipation'],
               'educate': ['positive'], 'educated': ['positive'], 'educational': ['positive', 'trust'], 'eel': ['fear'],
               'effective': ['positive', 'trust'], 'effeminate': ['negative'], 'efficacy': ['positive'],
               'efficiency': ['positive'], 'efficient': ['anticipation', 'positive', 'trust'], 'effigy': ['anger'],
               'effort': ['positive'], 'egotistical': ['disgust', 'negative'],
               'egregious': ['anger', 'disgust', 'negative'],
               'ejaculation': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'eject': ['negative'],
               'ejection': ['negative'], 'elaboration': ['positive'], 'elated': ['joy', 'positive'], 'elbow': ['anger'],
               'elder': ['positive', 'trust'], 'elders': ['positive', 'trust'], 'elect': ['positive', 'trust'],
               'electorate': ['trust'], 'electric': ['joy', 'positive', 'surprise'], 'electricity': ['positive'],
               'elegance': ['anticipation', 'joy', 'positive', 'trust'], 'elegant': ['joy', 'positive'],
               'elevation': ['anticipation', 'fear', 'joy', 'positive', 'trust'], 'elf': ['anger', 'disgust', 'fear'],
               'eligible': ['positive'], 'elimination': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'elite': ['anticipation', 'joy', 'positive', 'trust'], 'eloquence': ['positive'],
               'eloquent': ['positive'], 'elucidate': ['positive', 'trust'], 'elusive': ['negative', 'surprise'],
               'emaciated': ['fear', 'negative', 'sadness'], 'emancipation': ['anticipation', 'joy', 'positive'],
               'embargo': ['negative'], 'embarrass': ['negative', 'sadness'], 'embarrassing': ['negative'],
               'embarrassment': ['fear', 'negative', 'sadness', 'surprise'], 'embezzlement': ['negative'],
               'embolism': ['fear', 'negative', 'sadness'],
               'embrace': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'embroiled': ['negative'],
               'emergency': ['fear', 'negative', 'sadness', 'surprise'], 'emeritus': ['positive'],
               'eminence': ['positive', 'trust'], 'eminent': ['positive'], 'eminently': ['positive'],
               'emir': ['positive'], 'empathy': ['positive'], 'emphasize': ['trust'], 'employ': ['trust'],
               'empower': ['positive'], 'emptiness': ['sadness'], 'emulate': ['positive'],
               'enable': ['positive', 'trust'], 'enablement': ['positive', 'trust'],
               'enchant': ['anticipation', 'joy', 'positive', 'surprise'], 'enchanted': ['joy', 'positive', 'trust'],
               'enchanting': ['anticipation', 'joy', 'positive'], 'enclave': ['negative'], 'encore': ['positive'],
               'encourage': ['joy', 'positive', 'trust'], 'encouragement': ['positive'],
               'encroachment': ['fear', 'negative'], 'encumbrance': ['anger', 'fear', 'negative', 'sadness'],
               'encyclopedia': ['positive', 'trust'], 'endanger': ['anticipation', 'fear', 'negative'],
               'endangered': ['fear', 'negative'], 'endeavor': ['anticipation', 'positive'],
               'endemic': ['disgust', 'fear', 'negative', 'sadness'],
               'endless': ['anger', 'fear', 'joy', 'negative', 'positive', 'sadness', 'trust'],
               'endocarditis': ['fear', 'sadness'], 'endow': ['positive', 'trust'], 'endowed': ['positive'],
               'endowment': ['positive', 'trust'], 'endurance': ['positive'], 'endure': ['positive'],
               'enema': ['disgust'], 'enemy': ['anger', 'disgust', 'fear', 'negative'], 'energetic': ['positive'],
               'enforce': ['anger', 'fear', 'negative', 'positive'], 'enforcement': ['negative'],
               'engaged': ['anticipation', 'joy', 'positive', 'trust'], 'engaging': ['joy', 'positive', 'trust'],
               'engulf': ['anticipation'], 'enhance': ['positive'], 'enigmatic': ['fear', 'negative'],
               'enjoy': ['anticipation', 'joy', 'positive', 'trust'],
               'enjoying': ['anticipation', 'joy', 'positive', 'trust'], 'enlighten': ['joy', 'positive', 'trust'],
               'enlightenment': ['joy', 'positive', 'trust'], 'enliven': ['joy', 'positive', 'surprise', 'trust'],
               'enmity': ['anger', 'fear', 'negative', 'sadness'], 'enrich': ['positive'],
               'enroll': ['anticipation', 'trust'], 'ensemble': ['positive', 'trust'], 'ensign': ['positive'],
               'enslave': ['negative'], 'enslaved': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'enslavement': ['negative'], 'entangled': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'entanglement': ['negative'], 'enterprising': ['positive'], 'entertain': ['joy', 'positive'],
               'entertained': ['joy', 'positive'], 'entertaining': ['anticipation', 'joy', 'positive'],
               'entertainment': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'enthusiasm': ['anticipation', 'joy', 'positive', 'surprise'],
               'enthusiast': ['anticipation', 'joy', 'positive', 'surprise'], 'entrails': ['disgust', 'negative'],
               'entrust': ['trust'], 'envious': ['negative'], 'environ': ['positive'], 'ephemeris': ['positive'],
               'epic': ['positive'],
               'epidemic': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'epilepsy': ['negative'], 'episcopal': ['trust'], 'epitaph': ['sadness'], 'epitome': ['positive'],
               'equality': ['joy', 'positive', 'trust'], 'equally': ['positive'], 'equilibrium': ['positive'],
               'equity': ['positive'], 'eradicate': ['anger', 'negative'],
               'eradication': ['anger', 'disgust', 'fear', 'negative'], 'erase': ['fear', 'negative'],
               'erosion': ['negative'], 'erotic': ['anticipation', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'err': ['negative'], 'errand': ['anticipation', 'positive', 'trust'], 'errant': ['negative'],
               'erratic': ['negative', 'surprise'], 'erratum': ['negative'], 'erroneous': ['negative'],
               'error': ['negative', 'sadness'], 'erudite': ['positive'], 'erupt': ['anger', 'negative', 'surprise'],
               'eruption': ['anger', 'fear', 'negative', 'surprise'], 'escalate': ['anger', 'negative'],
               'escape': ['anticipation', 'fear', 'negative', 'positive'], 'escaped': ['fear'],
               'eschew': ['anger', 'negative', 'sadness'], 'escort': ['trust'], 'espionage': ['negative'],
               'esprit': ['positive'], 'essential': ['positive'], 'establish': ['trust'],
               'established': ['joy', 'positive'], 'esteem': ['joy', 'positive', 'sadness', 'trust'],
               'esthetic': ['positive'], 'estranged': ['negative'], 'ethereal': ['fear'], 'ethical': ['positive'],
               'ethics': ['positive'], 'euthanasia': ['fear', 'negative', 'sadness'], 'evacuate': ['fear', 'negative'],
               'evacuation': ['negative'], 'evade': ['anger', 'disgust', 'fear', 'negative'],
               'evanescence': ['sadness', 'surprise'], 'evasion': ['fear', 'negative', 'sadness'],
               'eventual': ['anticipation'], 'eventuality': ['anticipation', 'fear'],
               'evergreen': ['joy', 'positive', 'trust'], 'everlasting': ['positive'], 'evict': ['negative', 'sadness'],
               'eviction': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'evident': ['positive', 'trust'],
               'evil': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'evolution': ['positive'],
               'exacerbate': ['negative'], 'exacerbation': ['anger', 'fear', 'negative'], 'exacting': ['negative'],
               'exaggerate': ['anger', 'negative'], 'exaggerated': ['negative'],
               'exalt': ['anticipation', 'joy', 'positive', 'trust'], 'exaltation': ['joy', 'positive', 'trust'],
               'exalted': ['joy', 'positive', 'trust'], 'examination': ['fear', 'negative', 'surprise'],
               'exasperation': ['anger', 'disgust', 'negative'], 'excavation': ['anticipation', 'negative', 'surprise'],
               'exceed': ['anticipation', 'joy', 'positive'],
               'excel': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'excellence': ['disgust', 'joy', 'positive', 'trust'], 'excellent': ['joy', 'positive', 'trust'],
               'excess': ['negative'], 'exchange': ['positive', 'trust'], 'excise': ['negative'],
               'excitable': ['positive'],
               'excitation': ['anger', 'anticipation', 'fear', 'joy', 'positive', 'surprise'],
               'excite': ['anger', 'anticipation', 'fear', 'joy', 'positive', 'surprise'],
               'excited': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'excitement': ['anticipation', 'joy', 'positive', 'surprise'],
               'exciting': ['anticipation', 'joy', 'positive', 'surprise'], 'exclaim': ['surprise'],
               'excluded': ['disgust', 'negative', 'sadness'], 'excluding': ['negative', 'sadness'],
               'exclusion': ['disgust', 'fear', 'negative', 'sadness'], 'excrement': ['disgust', 'negative'],
               'excretion': ['disgust'], 'excruciating': ['fear', 'negative'], 'excuse': ['negative'],
               'execution': ['anger', 'fear', 'negative', 'sadness', 'trust'],
               'executioner': ['anger', 'fear', 'negative', 'sadness'], 'executor': ['trust'],
               'exemplary': ['positive'], 'exemption': ['positive'], 'exhaust': ['negative'],
               'exhausted': ['negative', 'sadness'], 'exhaustion': ['anticipation', 'negative', 'sadness'],
               'exhaustive': ['trust'], 'exhilaration': ['joy', 'positive', 'surprise'], 'exhort': ['positive'],
               'exhortation': ['positive'], 'exigent': ['anticipation', 'disgust', 'fear', 'negative', 'surprise'],
               'exile': ['anger', 'fear', 'negative', 'sadness'], 'existence': ['positive'],
               'exorcism': ['fear', 'negative', 'sadness'], 'exotic': ['positive'], 'expatriate': ['negative'],
               'expect': ['anticipation', 'positive', 'surprise', 'trust'], 'expectancy': ['anticipation'],
               'expectant': ['anticipation'], 'expectation': ['anticipation', 'positive'], 'expected': ['anticipation'],
               'expecting': ['anticipation'], 'expedient': ['joy', 'positive', 'trust'],
               'expedition': ['anticipation', 'positive'], 'expel': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'expenditure': ['negative'], 'expenses': ['negative'], 'experienced': ['positive', 'trust'],
               'experiment': ['anticipation', 'surprise'], 'expert': ['positive', 'trust'],
               'expertise': ['positive', 'trust'], 'expire': ['negative', 'sadness'], 'expired': ['negative'],
               'explain': ['positive', 'trust'], 'expletive': ['anger', 'negative'],
               'explode': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'explore': ['anticipation'],
               'explosion': ['fear', 'negative', 'surprise'],
               'explosive': ['anger', 'anticipation', 'fear', 'negative', 'surprise'],
               'expose': ['anticipation', 'fear'], 'exposed': ['negative'], 'expropriation': ['negative'],
               'expulsion': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'exquisite': ['joy', 'positive'],
               'exquisitely': ['positive'], 'extend': ['positive'], 'extensive': ['positive'],
               'exterminate': ['fear', 'negative', 'sadness'], 'extermination': ['anger', 'fear', 'negative'],
               'extinct': ['negative', 'sadness'], 'extinguish': ['anger', 'negative'], 'extra': ['positive'],
               'extrajudicial': ['fear', 'negative'], 'extraordinary': ['positive'], 'extremity': ['negative'],
               'extricate': ['anticipation', 'positive'], 'exuberance': ['joy', 'positive'], 'eyewitness': ['trust'],
               'fabricate': ['negative'], 'fabrication': ['negative', 'trust'], 'facilitate': ['positive'],
               'fact': ['trust'], 'facts': ['positive', 'trust'], 'faculty': ['positive', 'trust'],
               'fade': ['negative'], 'faeces': ['disgust', 'negative'],
               'failing': ['anger', 'anticipation', 'fear', 'negative', 'sadness'],
               'failure': ['disgust', 'fear', 'negative', 'sadness'],
               'fain': ['anticipation', 'joy', 'positive', 'trust'],
               'fainting': ['fear', 'negative', 'sadness', 'surprise'], 'fair': ['positive'],
               'fairly': ['positive', 'trust'], 'faith': ['anticipation', 'joy', 'positive', 'trust'],
               'faithful': ['positive', 'trust'], 'faithless': ['negative', 'sadness'], 'fake': ['negative'],
               'fall': ['negative', 'sadness'], 'fallacious': ['anger', 'negative'], 'fallacy': ['negative'],
               'fallible': ['negative'], 'falling': ['negative', 'sadness'], 'fallow': ['negative'],
               'falsehood': ['anger', 'negative', 'trust'], 'falsely': ['negative'],
               'falsification': ['anger', 'negative'], 'falsify': ['disgust', 'negative'],
               'falsity': ['disgust', 'negative'], 'falter': ['fear', 'negative'], 'fame': ['positive'],
               'familiar': ['positive', 'trust'], 'familiarity': ['anticipation', 'joy', 'positive', 'trust'],
               'famine': ['negative', 'sadness'], 'famous': ['positive'], 'famously': ['positive'],
               'fanatic': ['negative'], 'fanaticism': ['fear'], 'fancy': ['anticipation', 'joy', 'positive'],
               'fanfare': ['anticipation', 'joy', 'positive', 'surprise'], 'fang': ['fear'], 'fangs': ['fear'],
               'farce': ['negative'], 'farcical': ['disgust', 'negative'], 'farm': ['anticipation'],
               'fascinating': ['positive'], 'fascination': ['positive'], 'fashionable': ['positive'],
               'fasting': ['negative', 'sadness'], 'fat': ['disgust', 'negative', 'sadness'],
               'fatal': ['anger', 'fear', 'negative', 'sadness'], 'fatality': ['fear', 'negative', 'sadness'],
               'fate': ['anticipation', 'negative'], 'father': ['trust'], 'fatigue': ['negative'],
               'fatigued': ['negative', 'sadness'], 'fatty': ['disgust', 'negative', 'sadness'],
               'fault': ['negative', 'sadness'], 'faultless': ['positive', 'trust'], 'faulty': ['negative'],
               'favorable': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'favorite': ['joy', 'positive', 'trust'], 'favoritism': ['positive'], 'fawn': ['negative'],
               'fear': ['anger', 'fear', 'negative'], 'fearful': ['fear', 'negative', 'sadness'],
               'fearfully': ['fear', 'negative', 'sadness', 'surprise'], 'fearing': ['fear', 'negative'],
               'fearless': ['fear', 'positive'], 'feat': ['anticipation', 'joy', 'positive', 'surprise'],
               'feature': ['positive'], 'febrile': ['negative'], 'fecal': ['disgust', 'negative'],
               'feces': ['disgust', 'negative'], 'fee': ['anger', 'negative'], 'feeble': ['negative', 'sadness'],
               'feeling': ['anger', 'anticipation', 'disgust', 'fear', 'joy', 'negative', 'positive', 'sadness',
                           'surprise', 'trust'], 'feigned': ['negative'], 'felicity': ['joy', 'positive'],
               'fell': ['negative', 'sadness'], 'fellow': ['positive', 'trust'], 'fellowship': ['positive'],
               'felon': ['fear', 'negative'], 'felony': ['negative'], 'female': ['positive'], 'fenced': ['anger'],
               'fender': ['trust'], 'ferment': ['anticipation', 'negative'], 'fermentation': ['anticipation'],
               'ferocious': ['anger', 'disgust', 'fear', 'negative'], 'ferocity': ['anger', 'negative'],
               'fertile': ['positive'], 'fervor': ['anger', 'joy', 'positive'],
               'festival': ['anticipation', 'joy', 'positive', 'surprise'], 'festive': ['joy', 'positive'],
               'fete': ['anticipation', 'joy', 'positive', 'surprise'], 'feud': ['anger', 'negative'],
               'feudal': ['negative'], 'feudalism': ['anger', 'negative', 'sadness'], 'fever': ['fear'],
               'feverish': ['negative'], 'fiasco': ['negative'], 'fib': ['anger', 'negative'], 'fickle': ['negative'],
               'fictitious': ['negative'], 'fidelity': ['joy', 'positive', 'trust'],
               'fiend': ['anger', 'disgust', 'fear', 'negative'], 'fierce': ['anger', 'disgust', 'fear', 'negative'],
               'fiesta': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'fight': ['anger', 'fear', 'negative'], 'fighting': ['anger', 'negative'], 'filibuster': ['negative'],
               'fill': ['trust'], 'filth': ['disgust', 'negative'], 'filthy': ['disgust', 'negative'],
               'finally': ['anticipation', 'disgust', 'joy', 'positive', 'surprise', 'trust'], 'finery': ['positive'],
               'finesse': ['positive'], 'fire': ['fear'], 'firearms': ['anger', 'fear', 'negative'],
               'fireball': ['positive'], 'fireman': ['trust'], 'fireproof': ['positive'],
               'firmness': ['positive', 'trust'], 'firstborn': ['anticipation', 'joy', 'positive', 'trust'],
               'fits': ['anger', 'negative'], 'fitting': ['anticipation', 'joy', 'positive', 'trust'],
               'fixed': ['trust'], 'fixture': ['positive'], 'flabby': ['disgust', 'negative'],
               'flaccid': ['negative', 'sadness'], 'flagging': ['negative'],
               'flagrant': ['anger', 'disgust', 'negative'], 'flagship': ['trust'], 'flake': ['negative'],
               'flange': ['trust'], 'flap': ['negative'], 'flattering': ['joy', 'positive'],
               'flatulence': ['disgust', 'negative'], 'flaunt': ['negative'], 'flaw': ['negative', 'sadness'],
               'flea': ['disgust', 'negative'], 'fled': ['fear'], 'flee': ['fear', 'negative'],
               'fleece': ['anger', 'disgust', 'negative', 'sadness'], 'fleet': ['positive'], 'flesh': ['disgust'],
               'fleshy': ['negative'], 'flexibility': ['positive'],
               'flinch': ['anticipation', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'flirt': ['anticipation', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'flog': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'flood': ['fear'],
               'flop': ['disgust', 'negative', 'sadness'], 'floral': ['positive'],
               'flounder': ['fear', 'negative', 'sadness'], 'flow': ['positive'], 'flowery': ['positive'],
               'flu': ['fear', 'negative'], 'fluctuation': ['anger', 'fear', 'negative'], 'fluke': ['surprise'],
               'flush': ['positive'], 'flying': ['fear', 'positive'], 'focus': ['positive'],
               'foe': ['anger', 'fear', 'negative'], 'foiled': ['negative'], 'follower': ['trust'],
               'folly': ['negative'], 'fondness': ['joy', 'positive'], 'food': ['joy', 'positive', 'trust'],
               'fool': ['disgust', 'negative'], 'foolish': ['negative'], 'foolishness': ['negative'],
               'football': ['anticipation', 'joy', 'positive'], 'footing': ['trust'], 'forage': ['negative'],
               'foray': ['anger', 'negative'], 'forbearance': ['positive'], 'forbid': ['negative', 'sadness'],
               'forbidding': ['anger', 'fear', 'negative'], 'force': ['anger', 'fear', 'negative'],
               'forced': ['fear', 'negative'], 'forcibly': ['anger', 'fear', 'negative'], 'fore': ['positive'],
               'forearm': ['anger', 'anticipation'], 'foreboding': ['anticipation', 'fear', 'negative'],
               'forecast': ['anticipation', 'trust'], 'foreclose': ['fear', 'negative', 'sadness'],
               'forefathers': ['joy', 'positive', 'trust'], 'forego': ['negative'], 'foregoing': ['positive'],
               'foreign': ['negative'], 'foreigner': ['fear', 'negative'], 'foreman': ['positive'],
               'forerunner': ['positive'], 'foresee': ['anticipation', 'positive', 'surprise', 'trust'],
               'foreseen': ['anticipation', 'positive'], 'foresight': ['anticipation', 'positive', 'trust'],
               'forewarned': ['anticipation', 'fear', 'negative'], 'forfeit': ['anger', 'negative', 'sadness'],
               'forfeited': ['negative'], 'forfeiture': ['fear', 'negative', 'sadness'], 'forge': ['positive'],
               'forgery': ['negative'], 'forget': ['negative'], 'forgive': ['positive'], 'forgiven': ['positive'],
               'forgiving': ['positive', 'trust'], 'forgotten': ['fear', 'negative', 'sadness'],
               'forlorn': ['negative', 'sadness'], 'formality': ['trust'], 'formative': ['positive', 'trust'],
               'formidable': ['fear'], 'forming': ['anticipation'], 'formless': ['negative'],
               'formula': ['positive', 'trust'], 'fornication': ['negative'], 'forsake': ['negative', 'sadness'],
               'forsaken': ['anger', 'negative', 'sadness'], 'fort': ['trust'], 'forte': ['positive'],
               'forthcoming': ['positive'], 'fortify': ['positive'], 'fortitude': ['joy', 'positive', 'trust'],
               'fortress': ['fear', 'positive', 'sadness', 'trust'], 'fortunate': ['positive'],
               'fortune': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'forward': ['positive'],
               'foul': ['anger', 'disgust', 'fear', 'negative'], 'found': ['joy', 'positive', 'trust'],
               'foundation': ['positive'], 'fracture': ['negative'], 'fragile': ['fear', 'negative', 'sadness'],
               'fragrant': ['positive'], 'frailty': ['negative', 'sadness'], 'framework': ['trust'],
               'frank': ['positive', 'trust'], 'frankness': ['positive', 'trust'],
               'frantic': ['anticipation', 'disgust', 'fear', 'negative', 'surprise'],
               'fraternal': ['joy', 'positive', 'trust'], 'fraud': ['anger', 'negative'],
               'fraudulent': ['anger', 'disgust', 'negative'], 'fraught': ['fear', 'negative', 'sadness'],
               'fray': ['negative'], 'frayed': ['negative', 'sadness'],
               'freakish': ['disgust', 'fear', 'negative', 'surprise'], 'freedom': ['joy', 'positive', 'trust'],
               'freely': ['joy', 'positive', 'trust'], 'freezing': ['negative'],
               'frenetic': ['anger', 'fear', 'negative', 'surprise'], 'frenzied': ['anger', 'fear', 'negative'],
               'frenzy': ['negative'], 'fret': ['fear', 'negative'], 'friction': ['anger'],
               'friend': ['joy', 'positive', 'trust'], 'friendliness': ['joy', 'positive', 'trust'],
               'friendly': ['anticipation', 'joy', 'positive', 'trust'], 'friendship': ['joy', 'positive', 'trust'],
               'frigate': ['fear'], 'fright': ['fear', 'negative', 'surprise'],
               'frighten': ['fear', 'negative', 'sadness', 'surprise'], 'frightened': ['fear', 'negative', 'surprise'],
               'frightful': ['anger', 'fear', 'negative', 'sadness'], 'frigid': ['negative'],
               'frisky': ['anticipation', 'joy', 'positive', 'surprise'], 'frivolous': ['negative'],
               'frolic': ['joy', 'positive'], 'frostbite': ['negative'], 'froth': ['negative'],
               'frown': ['negative', 'sadness'], 'frowning': ['anger', 'disgust', 'negative', 'sadness'],
               'frugal': ['positive'], 'fruitful': ['positive'], 'fruitless': ['negative', 'sadness'],
               'frustrate': ['anger', 'disgust', 'negative', 'sadness'], 'frustrated': ['anger', 'negative'],
               'frustration': ['anger', 'negative'], 'fudge': ['negative'],
               'fugitive': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'trust'], 'fulfill': ['joy', 'positive'],
               'fulfillment': ['anticipation', 'joy', 'positive', 'trust'], 'full': ['positive'],
               'fully': ['positive', 'trust'], 'fumble': ['negative'], 'fume': ['negative'],
               'fuming': ['anger', 'negative'], 'fun': ['anticipation', 'joy', 'positive'],
               'fundamental': ['positive', 'trust'], 'funeral': ['sadness'], 'fungus': ['disgust', 'negative'],
               'funk': ['negative', 'sadness'], 'furious': ['anger', 'disgust', 'negative'], 'furiously': ['anger'],
               'furnace': ['anger'], 'furor': ['anger', 'negative'], 'furrow': ['sadness'],
               'fury': ['anger', 'fear', 'negative', 'sadness'], 'fuse': ['positive', 'trust'], 'fusion': ['positive'],
               'fuss': ['anger', 'negative', 'sadness'], 'fussy': ['disgust', 'negative'],
               'futile': ['negative', 'sadness'], 'futility': ['negative'], 'gaby': ['disgust', 'negative'],
               'gag': ['disgust', 'negative'], 'gage': ['trust'], 'gain': ['anticipation', 'joy', 'positive'],
               'gaining': ['positive'], 'gall': ['anger', 'disgust', 'negative'], 'gallant': ['positive'],
               'gallantry': ['positive'], 'gallows': ['anger', 'fear', 'negative', 'sadness'], 'galore': ['positive'],
               'gamble': ['negative'], 'gambler': ['anticipation', 'negative', 'surprise'],
               'gambling': ['anticipation', 'negative', 'surprise'], 'gang': ['anger', 'fear', 'negative'],
               'gaol': ['negative'], 'gap': ['negative'], 'gape': ['surprise'], 'garbage': ['disgust', 'negative'],
               'garden': ['joy', 'positive'], 'garish': ['disgust', 'negative', 'surprise'], 'garnet': ['positive'],
               'garnish': ['negative'], 'garrison': ['positive', 'trust'], 'gash': ['fear', 'negative'],
               'gasp': ['surprise'], 'gasping': ['fear', 'negative'], 'gate': ['trust'], 'gateway': ['trust'],
               'gauche': ['negative'], 'gauging': ['trust'], 'gaunt': ['negative'], 'gawk': ['surprise'],
               'gazette': ['positive', 'trust'], 'gear': ['positive'], 'gelatin': ['disgust'],
               'gem': ['joy', 'positive'], 'general': ['positive', 'trust'],
               'generosity': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'generous': ['joy', 'positive', 'trust'], 'genial': ['joy', 'positive'], 'genius': ['positive'],
               'gent': ['anger', 'disgust', 'fear', 'negative'], 'genteel': ['positive'],
               'gentleman': ['positive', 'trust'], 'gentleness': ['positive'], 'gentry': ['positive', 'trust'],
               'genuine': ['positive', 'trust'], 'geranium': ['positive'], 'geriatric': ['negative', 'sadness'],
               'germ': ['anticipation', 'disgust'], 'germination': ['anticipation'],
               'ghastly': ['disgust', 'fear', 'negative'], 'ghetto': ['disgust', 'fear', 'negative', 'sadness'],
               'ghost': ['fear'], 'ghostly': ['fear', 'negative'], 'giant': ['fear'],
               'gibberish': ['anger', 'negative'], 'gift': ['anticipation', 'joy', 'positive', 'surprise'],
               'gifted': ['positive'], 'gigantic': ['positive'], 'giggle': ['joy', 'positive'],
               'girder': ['positive', 'trust'], 'giving': ['positive'], 'glacial': ['negative'],
               'glad': ['anticipation', 'joy', 'positive'], 'gladiator': ['fear'], 'gladness': ['joy', 'positive'],
               'glare': ['anger', 'fear', 'negative'], 'glaring': ['anger', 'negative'], 'glee': ['joy', 'positive'],
               'glib': ['negative'], 'glide': ['anticipation', 'joy', 'positive'],
               'glimmer': ['anticipation', 'joy', 'positive', 'surprise'], 'glitter': ['disgust', 'joy'],
               'glittering': ['positive'], 'gloom': ['negative', 'sadness'], 'gloomy': ['negative', 'sadness'],
               'glorification': ['joy', 'positive'],
               'glorify': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'glory': ['anticipation', 'joy', 'positive', 'trust'], 'gloss': ['positive'],
               'glow': ['anticipation', 'joy', 'positive', 'trust'], 'glowing': ['positive'],
               'glum': ['negative', 'sadness'], 'glut': ['disgust', 'negative'], 'gluttony': ['disgust', 'negative'],
               'gnome': ['anger', 'disgust', 'fear', 'negative'], 'gob': ['disgust'],
               'goblin': ['disgust', 'fear', 'negative'], 'god': ['anticipation', 'fear', 'joy', 'positive', 'trust'],
               'godless': ['anger', 'negative'], 'godly': ['joy', 'positive', 'trust'],
               'godsend': ['joy', 'positive', 'surprise'], 'gold': ['positive'],
               'gonorrhea': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'goo': ['disgust', 'negative'],
               'good': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'goodly': ['positive'],
               'goodness': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'goods': ['positive'],
               'goodwill': ['positive'], 'gore': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'gorge': ['disgust', 'negative'], 'gorgeous': ['joy', 'positive'], 'gorilla': ['negative'],
               'gory': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'gospel': ['positive', 'trust'],
               'gossip': ['negative'], 'gouge': ['negative'], 'gout': ['negative'], 'govern': ['positive', 'trust'],
               'governess': ['trust'], 'government': ['fear', 'negative'], 'governor': ['trust'],
               'grab': ['anger', 'negative'], 'grace': ['positive'], 'graceful': ['positive'], 'gracious': ['positive'],
               'graciously': ['positive'], 'gradual': ['anticipation'],
               'graduation': ['anticipation', 'fear', 'joy', 'positive', 'surprise', 'trust'], 'grammar': ['trust'],
               'grandchildren': ['anticipation', 'joy', 'positive', 'trust'], 'grandeur': ['positive'],
               'grandfather': ['trust'], 'grandmother': ['positive'],
               'grant': ['anticipation', 'joy', 'positive', 'trust'], 'granted': ['positive'],
               'grasping': ['anticipation', 'negative'], 'grate': ['negative'], 'grated': ['anger', 'negative'],
               'grateful': ['positive'], 'gratify': ['joy', 'positive', 'surprise'],
               'grating': ['anger', 'disgust', 'negative'], 'gratitude': ['joy', 'positive'], 'gratuitous': ['disgust'],
               'grave': ['fear', 'negative', 'sadness'], 'gravitate': ['anticipation'], 'gray': ['disgust', 'sadness'],
               'greasy': ['disgust'], 'greater': ['positive'], 'greatness': ['joy', 'positive', 'surprise', 'trust'],
               'greed': ['anger', 'disgust', 'negative'], 'greedy': ['disgust', 'negative'],
               'green': ['joy', 'positive', 'trust'], 'greeting': ['positive', 'surprise'], 'gregarious': ['positive'],
               'grenade': ['fear', 'negative'], 'grief': ['negative', 'sadness'],
               'grievance': ['anger', 'disgust', 'negative', 'sadness'], 'grieve': ['fear', 'negative', 'sadness'],
               'grievous': ['anger', 'fear', 'negative', 'sadness'],
               'grim': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'grime': ['disgust', 'negative'], 'grimy': ['disgust', 'negative'],
               'grin': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'grinding': ['negative'],
               'grisly': ['disgust', 'fear', 'negative'], 'grist': ['positive'], 'grit': ['positive', 'trust'],
               'grizzly': ['fear', 'negative'], 'groan': ['disgust', 'negative', 'sadness'],
               'grope': ['anger', 'disgust', 'fear', 'negative'], 'gross': ['disgust', 'negative'],
               'grotesque': ['disgust', 'negative'], 'ground': ['trust'], 'grounded': ['fear', 'negative', 'sadness'],
               'groundless': ['negative'], 'groundwork': ['positive'],
               'grow': ['anticipation', 'joy', 'positive', 'trust'], 'growl': ['anger', 'fear', 'negative'],
               'growling': ['anger', 'disgust', 'fear', 'negative'], 'growth': ['positive'],
               'grudge': ['anger', 'negative'], 'grudgingly': ['negative'], 'gruesome': ['disgust', 'negative'],
               'gruff': ['anger', 'disgust', 'negative'], 'grumble': ['anger', 'disgust', 'negative'],
               'grumpy': ['anger', 'disgust', 'negative', 'sadness'], 'guarantee': ['positive', 'trust'],
               'guard': ['fear', 'positive', 'trust'], 'guarded': ['trust'], 'guardian': ['positive', 'trust'],
               'guardianship': ['positive', 'trust'], 'gubernatorial': ['positive', 'trust'],
               'guerilla': ['fear', 'negative'], 'guess': ['surprise'], 'guidance': ['positive', 'trust'],
               'guide': ['positive', 'trust'], 'guidebook': ['positive', 'trust'], 'guile': ['negative'],
               'guillotine': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'guilt': ['disgust', 'negative', 'sadness'], 'guilty': ['anger', 'negative', 'sadness'],
               'guise': ['negative'], 'gull': ['negative'], 'gullible': ['negative', 'sadness', 'trust'],
               'gulp': ['fear', 'surprise'], 'gun': ['anger', 'fear', 'negative'], 'gunpowder': ['fear'],
               'guru': ['positive', 'trust'], 'gush': ['disgust', 'joy', 'negative'], 'gusset': ['trust'],
               'gut': ['disgust'], 'guts': ['positive'], 'gutter': ['disgust'], 'guzzling': ['negative'],
               'gymnast': ['positive'], 'gypsy': ['negative'], 'habitat': ['positive'], 'habitual': ['anticipation'],
               'hag': ['disgust', 'fear', 'negative'], 'haggard': ['negative', 'sadness'],
               'hail': ['negative', 'positive', 'trust'], 'hairy': ['disgust', 'negative'], 'hale': ['positive'],
               'halfway': ['negative'], 'hallucination': ['fear', 'negative'],
               'halter': ['anger', 'fear', 'negative', 'sadness'], 'halting': ['fear', 'negative', 'sadness'],
               'hamper': ['negative'], 'hamstring': ['anger', 'negative', 'sadness'], 'handbook': ['trust'],
               'handicap': ['negative', 'sadness'], 'handy': ['positive'],
               'hanging': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'hangman': ['fear', 'negative'],
               'hankering': ['anticipation'], 'hap': ['anticipation', 'surprise'], 'happen': ['anticipation'],
               'happily': ['joy', 'positive'], 'happiness': ['anticipation', 'joy', 'positive'],
               'happy': ['anticipation', 'joy', 'positive', 'trust'], 'harass': ['anger', 'disgust', 'negative'],
               'harassing': ['anger'], 'harbinger': ['anger', 'anticipation', 'fear', 'negative'], 'harbor': ['trust'],
               'hardened': ['anger', 'disgust', 'fear', 'negative'], 'hardness': ['negative'],
               'hardship': ['negative', 'sadness'], 'hardy': ['joy', 'positive', 'trust'],
               'harlot': ['disgust', 'negative'], 'harm': ['fear', 'negative'],
               'harmful': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'harmoniously': ['joy', 'positive', 'trust'], 'harmony': ['joy', 'positive', 'trust'],
               'harrowing': ['fear', 'negative'], 'harry': ['anger', 'negative', 'sadness'],
               'harshness': ['anger', 'fear', 'negative'], 'harvest': ['anticipation', 'joy', 'positive'],
               'hash': ['negative'], 'hashish': ['negative'], 'haste': ['anticipation'], 'hasty': ['negative'],
               'hate': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'hateful': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'hating': ['anger', 'negative'],
               'hatred': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'haughty': ['anger', 'negative'],
               'haunt': ['fear', 'negative'], 'haunted': ['fear', 'negative', 'sadness'],
               'haven': ['positive', 'trust'], 'havoc': ['anger', 'fear', 'negative'], 'hawk': ['fear'],
               'hazard': ['fear', 'negative'], 'hazardous': ['fear', 'negative'], 'haze': ['fear', 'negative'],
               'headache': ['negative'], 'headlight': ['anticipation'], 'headway': ['positive'], 'heady': ['negative'],
               'heal': ['joy', 'positive', 'trust'], 'healing': ['anticipation', 'joy', 'positive', 'trust'],
               'healthful': ['joy', 'positive'], 'healthy': ['positive'], 'hearing': ['fear', 'negative'],
               'hearsay': ['negative'], 'hearse': ['fear', 'negative', 'sadness'], 'heartache': ['negative', 'sadness'],
               'heartburn': ['negative'], 'heartfelt': ['joy', 'positive', 'sadness', 'trust'], 'hearth': ['positive'],
               'heartily': ['joy', 'positive'], 'heartless': ['negative', 'sadness'], 'heartworm': ['disgust'],
               'heathen': ['fear', 'negative'], 'heavenly': ['anticipation', 'joy', 'positive', 'trust'],
               'heavens': ['joy', 'positive', 'trust'], 'heavily': ['negative'],
               'hedonism': ['joy', 'negative', 'positive'], 'heel': ['negative'],
               'heft': ['anticipation', 'fear', 'positive'], 'heighten': ['fear', 'negative'], 'heinous': ['negative'],
               'hell': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'hellish': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'helmet': ['fear', 'positive'],
               'helper': ['positive', 'trust'], 'helpful': ['joy', 'positive', 'trust'],
               'helpless': ['fear', 'negative', 'sadness'], 'helplessness': ['fear', 'negative', 'sadness'],
               'hemorrhage': ['disgust', 'fear', 'negative', 'sadness'], 'hemorrhoids': ['negative'],
               'herbal': ['positive'], 'heresy': ['negative'], 'heretic': ['disgust', 'negative'],
               'heritage': ['trust'], 'hermaphrodite': ['negative', 'surprise'], 'hermit': ['sadness', 'trust'],
               'hero': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'heroic': ['joy', 'positive', 'surprise', 'trust'], 'heroics': ['positive'], 'heroin': ['negative'],
               'heroine': ['positive', 'trust'], 'heroism': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'herpes': ['disgust', 'negative'], 'herpesvirus': ['disgust', 'negative'],
               'hesitation': ['fear', 'negative'], 'heyday': ['anticipation', 'joy', 'positive', 'trust'],
               'hidden': ['negative'], 'hide': ['fear'], 'hideous': ['disgust', 'fear', 'negative', 'sadness'],
               'hiding': ['fear'], 'highest': ['anticipation', 'fear', 'joy', 'negative', 'positive', 'surprise'],
               'hilarious': ['joy', 'positive', 'surprise'], 'hilarity': ['joy', 'positive'],
               'hindering': ['negative', 'sadness'], 'hindrance': ['negative'], 'hippie': ['negative'],
               'hire': ['anticipation', 'joy', 'positive', 'trust'], 'hiss': ['anger', 'fear', 'negative'],
               'hissing': ['negative'], 'hit': ['anger', 'negative'], 'hitherto': ['negative'], 'hive': ['negative'],
               'hoarse': ['negative'], 'hoary': ['negative', 'sadness'],
               'hoax': ['anger', 'disgust', 'negative', 'sadness', 'surprise'], 'hobby': ['joy', 'positive'],
               'hobo': ['negative', 'sadness'], 'hog': ['disgust', 'negative'],
               'holiday': ['anticipation', 'joy', 'positive'],
               'holiness': ['anticipation', 'fear', 'joy', 'positive', 'surprise', 'trust'],
               'hollow': ['negative', 'sadness'], 'holocaust': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'holy': ['positive'], 'homage': ['positive'],
               'homeless': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'homesick': ['negative', 'sadness'], 'homework': ['fear'], 'homicidal': ['anger', 'fear', 'negative'],
               'homicide': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'homology': ['positive'],
               'homosexuality': ['negative'],
               'honest': ['anger', 'disgust', 'fear', 'joy', 'positive', 'sadness', 'trust'],
               'honesty': ['positive', 'trust'], 'honey': ['positive'],
               'honeymoon': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'honor': ['positive', 'trust'],
               'honorable': ['positive', 'trust'], 'hood': ['anger', 'disgust', 'fear', 'negative'], 'hooded': ['fear'],
               'hooked': ['negative'], 'hoot': ['anger', 'disgust', 'negative'],
               'hope': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'hopeful': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'hopeless': ['fear', 'negative', 'sadness'],
               'hopelessness': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'horde': ['negative', 'surprise'],
               'horizon': ['anticipation', 'positive'], 'horoscope': ['anticipation'],
               'horrible': ['anger', 'disgust', 'fear', 'negative'], 'horribly': ['negative'],
               'horrid': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'horrific': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'horrified': ['fear', 'negative'],
               'horrifying': ['disgust', 'fear', 'negative', 'sadness'],
               'horror': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'horrors': ['fear', 'negative', 'sadness'], 'horse': ['trust'], 'hospice': ['sadness'],
               'hospital': ['fear', 'sadness', 'trust'], 'hospitality': ['positive'],
               'hostage': ['anger', 'fear', 'negative'], 'hostile': ['anger', 'disgust', 'fear', 'negative'],
               'hostilities': ['anger', 'fear', 'negative'], 'hostility': ['anger', 'disgust', 'negative'],
               'hot': ['anger'], 'household': ['positive'], 'housekeeping': ['positive'],
               'howl': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'huff': ['anger', 'disgust', 'negative'], 'hug': ['joy', 'positive', 'trust'], 'hulk': ['disgust'],
               'humane': ['positive'], 'humanitarian': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'humanity': ['joy', 'positive', 'trust'], 'humble': ['disgust', 'negative', 'positive', 'sadness'],
               'humbled': ['positive', 'sadness'], 'humbly': ['positive'],
               'humbug': ['anger', 'disgust', 'negative', 'sadness'], 'humiliate': ['anger', 'negative', 'sadness'],
               'humiliating': ['disgust', 'negative'], 'humiliation': ['disgust', 'negative', 'sadness'],
               'humility': ['positive', 'trust'], 'humorist': ['positive'], 'humorous': ['joy', 'positive'],
               'hunch': ['negative'], 'hungry': ['anticipation', 'negative'],
               'hunter': ['anticipation', 'fear', 'negative', 'sadness'],
               'hunting': ['anger', 'anticipation', 'fear', 'negative'], 'hurrah': ['joy', 'positive'],
               'hurricane': ['fear', 'negative'], 'hurried': ['anticipation', 'negative'], 'hurry': ['anticipation'],
               'hurt': ['anger', 'fear', 'negative', 'sadness'],
               'hurtful': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'hurting': ['anger', 'fear', 'negative', 'sadness'], 'husbandry': ['positive', 'trust'],
               'hush': ['positive'], 'hustler': ['negative'], 'hut': ['positive', 'sadness'],
               'hydra': ['fear', 'negative'], 'hydrocephalus': ['disgust', 'fear', 'negative', 'sadness'],
               'hygienic': ['positive'], 'hymn': ['anticipation', 'joy', 'positive', 'sadness', 'trust'],
               'hype': ['anticipation', 'negative'], 'hyperbole': ['negative'],
               'hypertrophy': ['disgust', 'fear', 'surprise'], 'hypocrisy': ['negative'],
               'hypocrite': ['disgust', 'negative'], 'hypocritical': ['disgust', 'negative'],
               'hypothesis': ['anticipation', 'surprise'], 'hysteria': ['fear', 'negative'],
               'hysterical': ['anger', 'fear', 'negative'], 'idealism': ['positive'],
               'idiocy': ['anger', 'disgust', 'negative', 'sadness'], 'idiot': ['disgust', 'negative'],
               'idiotic': ['anger', 'disgust', 'negative'], 'idler': ['negative'], 'idol': ['positive'],
               'idolatry': ['disgust', 'fear', 'negative'], 'ignorance': ['negative'],
               'ignorant': ['disgust', 'negative'], 'ignore': ['negative'],
               'ill': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'illegal': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'illegality': ['anger', 'disgust', 'fear', 'negative'], 'illegible': ['negative'],
               'illegitimate': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'illicit': ['anger', 'disgust', 'fear', 'negative'], 'illiterate': ['disgust', 'negative'],
               'illness': ['fear', 'negative', 'sadness'], 'illogical': ['negative'],
               'illuminate': ['anticipation', 'joy', 'positive', 'surprise'],
               'illumination': ['joy', 'positive', 'surprise', 'trust'], 'illusion': ['negative', 'surprise'],
               'illustrate': ['positive'], 'illustrious': ['positive'], 'imaginative': ['positive'],
               'imitated': ['negative'], 'imitation': ['negative'], 'immaculate': ['joy', 'positive', 'trust'],
               'immature': ['anticipation', 'negative'], 'immaturity': ['anger', 'anticipation', 'negative'],
               'immediacy': ['surprise'], 'immediately': ['anticipation', 'negative', 'positive'],
               'immense': ['positive'], 'immerse': ['anticipation', 'fear', 'joy', 'positive', 'surprise', 'trust'],
               'immigrant': ['fear'], 'imminent': ['anticipation', 'fear'],
               'immoral': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'immorality': ['anger', 'disgust', 'negative'], 'immortal': ['positive'],
               'immortality': ['anticipation'], 'immovable': ['negative', 'positive', 'trust'],
               'immunization': ['trust'], 'impair': ['negative'], 'impairment': ['negative'],
               'impart': ['positive', 'trust'], 'impartial': ['positive', 'trust'],
               'impartiality': ['positive', 'trust'], 'impassable': ['negative'], 'impatience': ['negative'],
               'impatient': ['anticipation', 'negative'], 'impeach': ['disgust', 'fear', 'negative'],
               'impeachment': ['negative'], 'impeccable': ['positive', 'trust'], 'impede': ['negative'],
               'impending': ['anticipation', 'fear'], 'impenetrable': ['trust'], 'imperfection': ['negative'],
               'imperfectly': ['negative'], 'impermeable': ['anger', 'fear'], 'impersonate': ['negative'],
               'impersonation': ['negative'], 'impervious': ['positive'], 'implacable': ['negative'],
               'implicate': ['anger', 'negative'], 'impolite': ['disgust', 'negative'],
               'importance': ['anticipation', 'positive'], 'important': ['positive', 'trust'],
               'imposition': ['negative'], 'impossible': ['negative', 'sadness'],
               'impotence': ['anger', 'fear', 'negative', 'sadness'], 'impotent': ['negative'], 'impound': ['negative'],
               'impracticable': ['negative'], 'impress': ['positive'], 'impression': ['positive'],
               'impressionable': ['trust'], 'imprison': ['negative'],
               'imprisoned': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'imprisonment': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'impropriety': ['negative'],
               'improve': ['anticipation', 'joy', 'positive', 'trust'], 'improved': ['positive'],
               'improvement': ['joy', 'positive', 'trust'], 'improving': ['positive'], 'improvisation': ['surprise'],
               'improvise': ['anticipation', 'positive', 'surprise'], 'imprudent': ['negative', 'sadness'],
               'impure': ['disgust', 'negative'], 'impurity': ['disgust', 'negative'], 'imputation': ['negative'],
               'inability': ['negative', 'sadness'], 'inaccessible': ['negative'], 'inaccurate': ['negative'],
               'inaction': ['negative'], 'inactivity': ['negative'], 'inadequacy': ['negative'],
               'inadequate': ['negative', 'sadness'], 'inadmissible': ['anger', 'disgust', 'negative'],
               'inalienable': ['positive'], 'inane': ['negative'], 'inapplicable': ['negative'],
               'inappropriate': ['anger', 'disgust', 'negative', 'sadness'], 'inattention': ['anger', 'negative'],
               'inaudible': ['negative'], 'inaugural': ['anticipation'],
               'inauguration': ['anticipation', 'joy', 'positive', 'trust'], 'incalculable': ['negative'],
               'incapacity': ['negative'], 'incarceration': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'incase': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'incendiary': ['anger', 'fear', 'negative', 'surprise'], 'incense': ['anger', 'negative'],
               'incessant': ['negative'], 'incest': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'incestuous': ['disgust', 'negative'], 'incident': ['surprise'], 'incineration': ['negative'],
               'incisive': ['positive'], 'incite': ['anger', 'anticipation', 'fear', 'negative'],
               'inclement': ['negative'], 'incline': ['trust'], 'include': ['positive'], 'included': ['positive'],
               'including': ['positive'], 'inclusion': ['trust'], 'inclusive': ['positive'], 'incoherent': ['negative'],
               'income': ['anticipation', 'joy', 'negative', 'positive', 'sadness', 'trust'],
               'incompatible': ['anger', 'disgust', 'negative', 'sadness'], 'incompetence': ['negative'],
               'incompetent': ['anger', 'negative', 'sadness'], 'incompleteness': ['negative'],
               'incomprehensible': ['negative'], 'incongruous': ['anger', 'negative'],
               'inconsequential': ['negative', 'sadness'], 'inconsiderate': ['anger', 'disgust', 'negative', 'sadness'],
               'inconsistency': ['negative'], 'incontinence': ['surprise'],
               'inconvenient': ['anger', 'disgust', 'negative', 'sadness'], 'incorrect': ['negative'],
               'increase': ['positive'], 'incredulous': ['anger', 'disgust', 'negative'],
               'incrimination': ['fear', 'negative', 'sadness'], 'incubus': ['disgust', 'fear', 'negative'],
               'incur': ['negative'], 'incurable': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'incursion': ['fear', 'negative'], 'indecency': ['anger', 'disgust'],
               'indecent': ['disgust', 'negative'], 'indecision': ['negative'], 'indecisive': ['negative'],
               'indefensible': ['fear', 'negative'], 'indelible': ['positive', 'trust'], 'indemnify': ['negative'],
               'indemnity': ['positive', 'trust'], 'indent': ['trust'], 'indenture': ['anger', 'negative'],
               'independence': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'indestructible': ['positive', 'trust'], 'indeterminate': ['negative'],
               'indict': ['anger', 'fear', 'negative'], 'indictment': ['fear', 'negative'],
               'indifference': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'indigent': ['negative', 'sadness'],
               'indignant': ['anger', 'negative'], 'indignation': ['anger', 'disgust', 'negative'],
               'indistinct': ['negative'], 'individuality': ['positive'], 'indivisible': ['trust'],
               'indoctrination': ['anger', 'fear', 'negative'], 'indolent': ['negative'],
               'indomitable': ['fear', 'positive'], 'ineffable': ['positive'], 'ineffective': ['negative'],
               'ineffectual': ['disgust', 'negative'], 'inefficiency': ['disgust', 'negative', 'sadness'],
               'inefficient': ['negative', 'sadness'], 'inept': ['anger', 'disgust', 'negative'],
               'ineptitude': ['disgust', 'fear', 'negative', 'sadness'],
               'inequality': ['anger', 'fear', 'negative', 'sadness'], 'inequitable': ['negative'],
               'inert': ['negative'], 'inexcusable': ['anger', 'disgust', 'negative', 'sadness'],
               'inexpensive': ['positive'], 'inexperience': ['negative'], 'inexperienced': ['negative'],
               'inexplicable': ['negative', 'surprise'], 'infallibility': ['trust'], 'infallible': ['positive'],
               'infamous': ['anger', 'disgust', 'fear', 'negative'], 'infamy': ['negative', 'sadness'],
               'infant': ['anticipation', 'fear', 'joy', 'positive', 'surprise'],
               'infanticide': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'infantile': ['anger', 'disgust', 'negative'], 'infarct': ['fear', 'negative', 'surprise'],
               'infect': ['disgust', 'negative'], 'infection': ['fear', 'negative'],
               'infectious': ['disgust', 'fear', 'negative', 'sadness'], 'inferior': ['negative', 'sadness'],
               'inferiority': ['negative'], 'inferno': ['anger', 'fear', 'negative'],
               'infertility': ['negative', 'sadness'], 'infestation': ['disgust', 'fear', 'negative'],
               'infidel': ['anger', 'disgust', 'fear', 'negative'],
               'infidelity': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'infiltration': ['negative', 'positive'], 'infinite': ['positive'],
               'infinity': ['anticipation', 'joy', 'positive', 'trust'], 'infirm': ['negative'],
               'infirmity': ['fear', 'negative'], 'inflammation': ['negative'], 'inflated': ['negative'],
               'inflation': ['fear', 'negative'], 'inflict': ['anger', 'fear', 'negative', 'sadness'],
               'infliction': ['fear', 'negative', 'sadness'], 'influence': ['negative', 'positive'],
               'influential': ['positive', 'trust'], 'influenza': ['negative'], 'inform': ['trust'],
               'information': ['positive'], 'informer': ['negative'], 'infraction': ['anger', 'negative'],
               'infrequent': ['surprise'], 'infrequently': ['negative'], 'infringement': ['negative'],
               'ingenious': ['positive'], 'inheritance': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'inhibit': ['anger', 'disgust', 'negative', 'sadness'], 'inhospitable': ['negative', 'sadness'],
               'inhuman': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'inhumanity': ['negative', 'sadness'],
               'inimical': ['anger', 'negative', 'sadness'], 'inimitable': ['positive', 'trust'],
               'iniquity': ['disgust', 'negative'], 'injection': ['fear'], 'injunction': ['negative'],
               'injure': ['anger', 'fear', 'negative', 'sadness'], 'injured': ['fear', 'negative', 'sadness'],
               'injurious': ['anger', 'fear', 'negative', 'sadness'],
               'injury': ['anger', 'fear', 'negative', 'sadness'], 'injustice': ['anger', 'negative'],
               'inmate': ['disgust', 'fear', 'negative'], 'innocence': ['positive'], 'innocent': ['positive', 'trust'],
               'innocently': ['positive'], 'innocuous': ['positive'], 'innovate': ['positive'],
               'innovation': ['positive'], 'inoculation': ['anticipation', 'trust'],
               'inoperative': ['anger', 'negative'], 'inquirer': ['positive'], 'inquiry': ['anticipation', 'positive'],
               'inquisitive': ['positive'], 'insane': ['anger', 'fear', 'negative'],
               'insanity': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'insecure': ['anger', 'fear', 'negative', 'sadness'], 'insecurity': ['fear', 'negative'],
               'inseparable': ['joy', 'positive', 'trust'], 'insidious': ['anger', 'disgust', 'fear', 'negative'],
               'insignia': ['positive'], 'insignificance': ['negative'],
               'insignificant': ['anger', 'negative', 'sadness'], 'insipid': ['negative'], 'insolent': ['negative'],
               'insolvency': ['fear', 'negative', 'sadness', 'surprise'],
               'insolvent': ['fear', 'negative', 'sadness', 'trust'], 'inspector': ['positive'],
               'inspiration': ['anticipation', 'joy', 'positive'],
               'inspire': ['anticipation', 'joy', 'positive', 'trust'],
               'inspired': ['joy', 'positive', 'surprise', 'trust'], 'instability': ['disgust', 'fear', 'negative'],
               'install': ['anticipation'], 'instigate': ['negative'], 'instigation': ['negative'],
               'instinctive': ['anger', 'disgust', 'fear', 'positive'], 'institute': ['trust'],
               'instruct': ['positive', 'trust'], 'instruction': ['positive', 'trust'],
               'instructions': ['anticipation', 'trust'], 'instructor': ['anticipation', 'positive', 'trust'],
               'instrumental': ['positive'], 'insufficiency': ['anger', 'negative'], 'insufficient': ['negative'],
               'insufficiently': ['negative'], 'insulation': ['trust'],
               'insult': ['anger', 'disgust', 'negative', 'sadness', 'surprise'],
               'insulting': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'insure': ['positive', 'trust'],
               'insurgent': ['negative'], 'insurmountable': ['fear', 'negative', 'sadness'],
               'insurrection': ['anger', 'negative'], 'intact': ['positive', 'trust'],
               'integrity': ['positive', 'trust'], 'intellect': ['positive'], 'intellectual': ['positive'],
               'intelligence': ['fear', 'joy', 'positive', 'trust'], 'intelligent': ['positive', 'trust'],
               'intend': ['trust'], 'intended': ['anticipation', 'positive'],
               'intense': ['anger', 'disgust', 'fear', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'inter': ['negative', 'sadness'], 'intercede': ['fear', 'sadness'], 'intercession': ['trust'],
               'intercourse': ['positive'], 'interdiction': ['negative'], 'interest': ['positive'],
               'interested': ['disgust', 'positive', 'sadness'], 'interesting': ['positive'],
               'interference': ['negative'], 'interim': ['anticipation'], 'interior': ['disgust', 'positive', 'trust'],
               'interlocutory': ['positive'], 'interlude': ['positive'], 'interment': ['negative', 'sadness'],
               'interminable': ['anger', 'anticipation', 'negative', 'positive'], 'intermission': ['anticipation'],
               'interrogate': ['fear'], 'interrogation': ['fear'], 'interrupt': ['anger', 'negative', 'surprise'],
               'interrupted': ['negative', 'sadness'], 'intervention': ['negative', 'positive', 'sadness'],
               'interviewer': ['fear'], 'intestate': ['negative'], 'intestinal': ['disgust'],
               'intimate': ['anticipation', 'joy', 'positive', 'trust'], 'intimately': ['anticipation', 'fear', 'joy'],
               'intimidate': ['fear', 'negative'], 'intimidation': ['anger', 'fear', 'negative'],
               'intolerable': ['anger', 'negative'], 'intolerance': ['anger', 'disgust', 'fear', 'negative'],
               'intolerant': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'intonation': ['positive'],
               'intoxicated': ['disgust', 'negative'], 'intractable': ['anger', 'negative'], 'intrepid': ['positive'],
               'intrigue': ['anticipation', 'fear', 'negative', 'surprise'],
               'intruder': ['anger', 'fear', 'negative', 'surprise'], 'intrusion': ['fear', 'negative'],
               'intrusive': ['anger', 'disgust', 'fear', 'negative', 'surprise'], 'intuition': ['positive', 'trust'],
               'intuitive': ['positive'], 'intuitively': ['anticipation'],
               'invade': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'invader': ['anger', 'fear', 'negative', 'sadness'], 'invalid': ['sadness'], 'invalidate': ['negative'],
               'invalidation': ['negative'], 'invalidity': ['negative'], 'invariably': ['positive'],
               'invasion': ['anger', 'negative'], 'inventive': ['positive'], 'inventor': ['positive'],
               'investigate': ['positive'], 'investigation': ['anticipation'], 'invigorate': ['positive'],
               'invitation': ['anticipation', 'positive'],
               'invite': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'inviting': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'invocation': ['anticipation', 'trust'], 'invoke': ['anticipation'], 'involuntary': ['negative'],
               'involution': ['anger', 'negative'], 'involvement': ['anger'], 'irate': ['anger', 'negative'],
               'ire': ['anger', 'negative'], 'iris': ['fear'], 'iron': ['positive', 'trust'], 'irons': ['negative'],
               'irrational': ['disgust', 'fear', 'negative'], 'irrationality': ['negative'],
               'irreconcilable': ['anger', 'fear', 'negative', 'sadness'], 'irreducible': ['positive'],
               'irrefutable': ['positive', 'trust'], 'irregular': ['negative'], 'irregularity': ['negative'],
               'irrelevant': ['negative'], 'irreparable': ['fear', 'negative', 'sadness'],
               'irresponsible': ['negative'], 'irreverent': ['negative'], 'irrevocable': ['negative'],
               'irritability': ['anger', 'negative'], 'irritable': ['anger', 'negative'],
               'irritating': ['anger', 'disgust', 'negative'],
               'irritation': ['anger', 'disgust', 'negative', 'sadness'], 'isolate': ['sadness'],
               'isolated': ['fear', 'negative', 'sadness'], 'isolation': ['negative', 'sadness'], 'jab': ['anger'],
               'jabber': ['negative'], 'jackpot': ['anticipation', 'joy', 'positive', 'surprise'],
               'jail': ['fear', 'negative', 'sadness'], 'jam': ['positive'], 'janitor': ['disgust'],
               'jargon': ['negative'], 'jarring': ['fear', 'negative', 'sadness'], 'jaundice': ['fear', 'negative'],
               'jaws': ['fear'], 'jealous': ['anger', 'disgust', 'negative'],
               'jealousy': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'jeopardize': ['anger', 'fear', 'negative'], 'jeopardy': ['anticipation', 'fear', 'negative'],
               'jerk': ['anger', 'surprise'], 'jest': ['joy', 'positive', 'surprise'], 'job': ['positive'],
               'john': ['disgust', 'negative'], 'join': ['positive'], 'joined': ['positive'], 'joke': ['negative'],
               'joker': ['joy', 'positive', 'surprise'], 'joking': ['positive'], 'jolt': ['surprise'],
               'jornada': ['negative'], 'journalism': ['trust'], 'journalist': ['positive'],
               'journey': ['anticipation', 'fear', 'joy', 'positive'], 'journeyman': ['trust'],
               'jovial': ['joy', 'positive'], 'joy': ['joy', 'positive'], 'joyful': ['joy', 'positive', 'trust'],
               'joyous': ['joy', 'positive'], 'jubilant': ['joy', 'positive', 'surprise', 'trust'],
               'jubilee': ['joy', 'positive', 'surprise'], 'judgment': ['surprise'],
               'judicial': ['anticipation', 'positive', 'trust'], 'judiciary': ['anticipation', 'trust'],
               'judicious': ['positive', 'trust'], 'jumble': ['negative'], 'jump': ['joy', 'positive'],
               'jungle': ['fear'], 'junk': ['negative'], 'junta': ['negative'], 'jurisprudence': ['sadness'],
               'jurist': ['trust'], 'jury': ['trust'], 'justice': ['positive', 'trust'],
               'justifiable': ['positive', 'trust'], 'justification': ['positive'], 'juvenile': ['negative'],
               'keepsake': ['positive'], 'ken': ['positive'], 'kennel': ['sadness'], 'kern': ['negative'],
               'kerosene': ['fear'], 'keynote': ['positive'], 'keystone': ['positive'], 'khan': ['fear', 'trust'],
               'kick': ['anger', 'negative'], 'kicking': ['anger'],
               'kidnap': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'kill': ['fear', 'negative', 'sadness'],
               'killing': ['anger', 'fear', 'negative', 'sadness'], 'kind': ['joy', 'positive', 'trust'],
               'kindness': ['positive'], 'kindred': ['anticipation', 'joy', 'positive', 'trust'], 'king': ['positive'],
               'kiss': ['anticipation', 'joy', 'positive', 'surprise'], 'kite': ['disgust', 'negative'],
               'kitten': ['joy', 'positive', 'trust'], 'knack': ['positive'], 'knell': ['fear', 'negative', 'sadness'],
               'knickers': ['trust'], 'knight': ['positive'], 'knotted': ['negative'], 'knowing': ['positive'],
               'knowledge': ['positive'], 'kudos': ['joy', 'positive'], 'label': ['trust'],
               'labor': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'labored': ['negative', 'sadness'],
               'laborious': ['negative'], 'labyrinth': ['anticipation', 'negative'],
               'lace': ['anger', 'fear', 'negative', 'positive', 'sadness', 'trust'], 'lack': ['negative'],
               'lacking': ['negative'], 'lackluster': ['negative'], 'laden': ['negative'], 'lag': ['negative'],
               'lagging': ['anger', 'anticipation', 'disgust', 'negative', 'sadness'], 'lair': ['negative'],
               'lamb': ['joy', 'positive', 'trust'], 'lament': ['disgust', 'fear', 'negative', 'sadness'],
               'lamenting': ['sadness'], 'land': ['positive'], 'landed': ['positive'], 'landmark': ['trust'],
               'landslide': ['fear', 'negative', 'sadness'], 'languid': ['negative'], 'languish': ['negative'],
               'languishing': ['fear', 'negative', 'sadness'], 'lapse': ['negative'],
               'larceny': ['disgust', 'negative'], 'larger': ['disgust', 'surprise', 'trust'],
               'laser': ['positive', 'trust'], 'lash': ['anger', 'fear', 'negative'], 'late': ['negative', 'sadness'],
               'lateness': ['negative'], 'latent': ['anger', 'anticipation', 'disgust', 'negative', 'surprise'],
               'latrines': ['disgust', 'negative'], 'laudable': ['positive'], 'laugh': ['joy', 'positive', 'surprise'],
               'laughable': ['disgust', 'negative'], 'laughing': ['joy', 'positive'],
               'laughter': ['anticipation', 'joy', 'positive', 'surprise'], 'launch': ['anticipation', 'positive'],
               'laureate': ['positive', 'trust'], 'laurel': ['positive'], 'laurels': ['joy', 'positive'],
               'lava': ['anger', 'fear', 'negative'], 'lavatory': ['disgust'], 'lavish': ['positive'], 'law': ['trust'],
               'lawful': ['positive', 'trust'], 'lawlessness': ['anger', 'fear', 'negative'],
               'lawsuit': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'lawyer': ['anger', 'disgust', 'fear', 'negative'], 'lax': ['negative', 'sadness'],
               'laxative': ['disgust', 'fear', 'negative'], 'lazy': ['negative'], 'lead': ['positive'],
               'leader': ['positive', 'trust'], 'leading': ['trust'], 'league': ['positive'], 'leak': ['negative'],
               'leakage': ['negative'], 'leaky': ['negative'], 'leaning': ['trust'], 'learn': ['positive'],
               'learning': ['positive'], 'leave': ['negative', 'sadness', 'surprise'], 'lecturer': ['positive'],
               'leech': ['negative'], 'leeches': ['disgust', 'fear', 'negative'], 'leer': ['disgust', 'negative'],
               'leery': ['surprise'], 'leeway': ['positive'], 'legal': ['positive', 'trust'],
               'legalized': ['anger', 'fear', 'joy', 'positive', 'trust'], 'legendary': ['positive'],
               'legibility': ['positive'], 'legible': ['positive'], 'legislator': ['trust'], 'legislature': ['trust'],
               'legitimacy': ['trust'], 'leisure': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'leisurely': ['positive'], 'lemma': ['positive'], 'lemon': ['disgust', 'negative'], 'lender': ['trust'],
               'lenient': ['positive'], 'leprosy': ['disgust', 'fear', 'negative', 'sadness'],
               'lesbian': ['disgust', 'negative', 'sadness'], 'lessen': ['anticipation', 'negative'],
               'lesser': ['disgust', 'negative'], 'lesson': ['anticipation', 'positive', 'trust'],
               'lethal': ['disgust', 'fear', 'negative', 'sadness'], 'lethargy': ['negative', 'sadness'],
               'letter': ['anticipation'], 'lettered': ['anticipation', 'positive', 'trust'],
               'leukemia': ['anger', 'fear', 'negative', 'sadness'], 'levee': ['trust'], 'level': ['positive', 'trust'],
               'leverage': ['positive'], 'levy': ['negative'], 'lewd': ['disgust', 'negative'], 'liaison': ['negative'],
               'liar': ['disgust', 'negative'], 'libel': ['anger', 'fear', 'negative', 'trust'],
               'liberal': ['negative', 'positive'],
               'liberate': ['anger', 'anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'liberation': ['anticipation', 'joy', 'positive', 'surprise'],
               'liberty': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'library': ['positive'],
               'lick': ['disgust', 'negative'], 'lie': ['anger', 'disgust', 'negative', 'sadness'],
               'lieutenant': ['trust'], 'lifeblood': ['positive'], 'lifeless': ['fear', 'negative', 'sadness'],
               'lighthouse': ['positive'], 'lightning': ['anger', 'fear', 'surprise'],
               'liking': ['joy', 'positive', 'trust'], 'limited': ['anger', 'negative', 'sadness'],
               'limp': ['negative'], 'lines': ['fear'], 'linger': ['anticipation'], 'linguist': ['positive', 'trust'],
               'lint': ['negative'], 'lion': ['fear', 'positive'], 'liquor': ['anger', 'joy', 'negative', 'sadness'],
               'listless': ['negative', 'sadness'], 'lithe': ['positive'], 'litigant': ['negative'],
               'litigate': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'litigation': ['negative'], 'litigious': ['anger', 'disgust', 'negative'], 'litter': ['negative'],
               'livid': ['anger', 'disgust', 'negative'], 'loaf': ['negative'], 'loafer': ['negative'],
               'loath': ['anger', 'negative'], 'loathe': ['anger', 'disgust', 'negative'],
               'loathing': ['disgust', 'negative'], 'loathsome': ['anger', 'disgust', 'negative'],
               'lobbyist': ['negative'], 'localize': ['anticipation'], 'lockup': ['fear', 'negative', 'sadness'],
               'locust': ['fear', 'negative'], 'lodging': ['trust'], 'lofty': ['negative'], 'logical': ['positive'],
               'lone': ['sadness'], 'loneliness': ['fear', 'negative', 'sadness'],
               'lonely': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'lonesome': ['negative', 'sadness'],
               'long': ['anticipation'], 'longevity': ['positive'], 'longing': ['anticipation', 'sadness'],
               'loo': ['disgust', 'negative'], 'loom': ['anticipation', 'fear', 'negative'],
               'loon': ['disgust', 'negative'], 'loony': ['negative'], 'loot': ['negative'],
               'lord': ['disgust', 'negative', 'positive', 'trust'], 'lordship': ['positive'],
               'lose': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'losing': ['anger', 'negative', 'sadness'], 'loss': ['anger', 'fear', 'negative', 'sadness'],
               'lost': ['negative', 'sadness'], 'lottery': ['anticipation'], 'loudness': ['anger', 'negative'],
               'lounge': ['negative'], 'louse': ['disgust', 'negative'], 'lovable': ['joy', 'positive', 'trust'],
               'love': ['joy', 'positive'],
               'lovely': ['anticipation', 'joy', 'positive', 'sadness', 'surprise', 'trust'],
               'lovemaking': ['joy', 'positive', 'trust'], 'lover': ['anticipation', 'joy', 'positive', 'trust'],
               'loving': ['joy', 'positive', 'trust'], 'lower': ['negative', 'sadness'], 'lowering': ['negative'],
               'lowest': ['negative', 'sadness'], 'lowlands': ['negative'], 'lowly': ['negative', 'sadness'],
               'loyal': ['fear', 'joy', 'positive', 'surprise', 'trust'], 'loyalty': ['positive', 'trust'],
               'luck': ['anticipation', 'joy', 'positive', 'surprise'], 'lucky': ['joy', 'positive', 'surprise'],
               'ludicrous': ['negative'], 'lull': ['anticipation'], 'lumbering': ['negative'], 'lump': ['negative'],
               'lumpy': ['disgust', 'negative'], 'lunacy': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'lunatic': ['anger', 'disgust', 'fear', 'negative'], 'lunge': ['surprise'], 'lurch': ['negative'],
               'lure': ['negative'], 'lurid': ['disgust', 'negative'], 'lurk': ['negative'], 'lurking': ['negative'],
               'luscious': ['anticipation', 'joy', 'positive'], 'lush': ['disgust', 'negative', 'sadness'],
               'lust': ['anticipation', 'negative'], 'luster': ['joy', 'positive'], 'lustful': ['negative'],
               'lustrous': ['positive'], 'lusty': ['disgust'], 'luxuriant': ['positive'],
               'luxurious': ['joy', 'positive'], 'luxury': ['joy', 'positive'],
               'lying': ['anger', 'disgust', 'negative'], 'lynch': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'lyre': ['joy', 'positive'], 'lyrical': ['joy', 'positive'], 'mace': ['fear', 'negative'],
               'machine': ['trust'], 'mad': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'madden': ['anger', 'fear', 'negative'], 'madman': ['anger', 'fear', 'negative'],
               'madness': ['anger', 'fear', 'negative'], 'mafia': ['fear', 'negative'], 'mage': ['fear'],
               'maggot': ['disgust', 'negative'], 'magical': ['anticipation', 'joy', 'positive', 'surprise'],
               'magician': ['surprise'], 'magnet': ['positive', 'trust'], 'magnetism': ['positive'],
               'magnetite': ['positive'], 'magnificence': ['anticipation', 'joy', 'positive', 'trust'],
               'magnificent': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'maiden': ['positive'],
               'mail': ['anticipation'], 'main': ['positive'], 'mainstay': ['positive', 'trust'],
               'maintenance': ['trust'], 'majestic': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'majesty': ['positive', 'trust'], 'major': ['positive'], 'majority': ['joy', 'positive', 'trust'],
               'makeshift': ['negative'], 'malady': ['negative'], 'malaise': ['negative', 'sadness'],
               'malaria': ['disgust', 'fear', 'negative', 'sadness'],
               'malevolent': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'malfeasance': ['disgust', 'negative'], 'malformation': ['negative'],
               'malice': ['anger', 'fear', 'negative'],
               'malicious': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'malign': ['anger', 'disgust', 'negative'], 'malignancy': ['fear', 'negative', 'sadness'],
               'malignant': ['anger', 'fear', 'negative'], 'malpractice': ['anger', 'negative'], 'mamma': ['trust'],
               'manage': ['positive', 'trust'], 'management': ['positive', 'trust'], 'mandamus': ['fear', 'negative'],
               'mandarin': ['positive', 'trust'], 'mange': ['disgust', 'fear', 'negative'],
               'mangle': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'manhood': ['positive'],
               'mania': ['negative'], 'maniac': ['anger', 'fear', 'negative'], 'maniacal': ['negative'],
               'manifestation': ['fear'], 'manifested': ['positive'], 'manipulate': ['negative'],
               'manipulation': ['anger', 'fear', 'negative'], 'manly': ['positive'], 'manna': ['positive'],
               'mannered': ['positive'], 'manners': ['positive'],
               'manslaughter': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'manual': ['trust'],
               'manufacturer': ['positive'], 'manure': ['disgust', 'negative'], 'mar': ['negative'],
               'march': ['positive'], 'margin': ['negative', 'sadness'], 'marine': ['trust'], 'marked': ['positive'],
               'marketable': ['positive'], 'maroon': ['negative'], 'marquis': ['positive'],
               'marriage': ['anticipation', 'joy', 'positive', 'trust'], 'marrow': ['joy', 'positive', 'trust'],
               'marry': ['anticipation', 'fear', 'joy', 'positive', 'surprise', 'trust'],
               'marshal': ['positive', 'trust'], 'martial': ['anger'], 'martingale': ['negative'],
               'martyr': ['fear', 'negative', 'sadness'], 'martyrdom': ['fear', 'negative', 'sadness'],
               'marvel': ['positive', 'surprise'], 'marvelous': ['joy', 'positive'], 'marvelously': ['joy', 'positive'],
               'masculine': ['positive'], 'masochism': ['anger', 'disgust', 'fear', 'negative'],
               'massacre': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'massage': ['joy', 'positive'],
               'master': ['positive'], 'masterpiece': ['joy', 'positive'],
               'mastery': ['anger', 'joy', 'positive', 'trust'], 'matchmaker': ['anticipation'],
               'mate': ['positive', 'trust'], 'materialism': ['negative'], 'materialist': ['disgust', 'negative'],
               'maternal': ['anticipation', 'negative', 'positive'], 'mathematical': ['trust'],
               'matrimony': ['anticipation', 'joy', 'positive', 'trust'], 'matron': ['positive', 'trust'],
               'mausoleum': ['sadness'], 'maxim': ['trust'], 'maximum': ['positive'], 'mayor': ['positive'],
               'meadow': ['positive'], 'meandering': ['negative'], 'meaningless': ['negative', 'sadness'],
               'measles': ['disgust', 'fear', 'negative', 'sadness'], 'measure': ['trust'],
               'measured': ['positive', 'trust'], 'medal': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'meddle': ['anger', 'negative'], 'mediate': ['anticipation', 'positive', 'trust'],
               'mediation': ['positive'], 'mediator': ['anticipation', 'positive', 'trust'],
               'medical': ['anticipation', 'fear', 'positive', 'trust'], 'mediocre': ['negative'],
               'mediocrity': ['negative'], 'meditate': ['anticipation', 'joy', 'positive', 'trust'],
               'mediterranean': ['positive'], 'medley': ['positive'], 'meek': ['sadness'],
               'melancholic': ['negative', 'sadness'], 'melancholy': ['negative', 'sadness'],
               'melee': ['fear', 'negative'], 'melodrama': ['anger', 'negative', 'sadness'],
               'meltdown': ['negative', 'sadness'], 'memento': ['positive'],
               'memorable': ['joy', 'positive', 'surprise', 'trust'], 'memorials': ['sadness'],
               'menace': ['anger', 'fear', 'negative'], 'menacing': ['anger', 'fear', 'negative'],
               'mending': ['positive'], 'menial': ['negative'], 'menses': ['positive'], 'mentor': ['positive', 'trust'],
               'mercenary': ['fear', 'negative'], 'merchant': ['trust'], 'merciful': ['positive'],
               'merciless': ['fear', 'negative'], 'mercy': ['positive'], 'merge': ['anticipation', 'positive'],
               'merit': ['positive', 'trust'], 'meritorious': ['joy', 'positive', 'trust'],
               'merriment': ['joy', 'positive', 'surprise'], 'merry': ['joy', 'positive'],
               'mess': ['disgust', 'negative'], 'messenger': ['trust'], 'messy': ['disgust', 'negative'],
               'metastasis': ['negative'], 'methanol': ['negative'], 'metropolitan': ['positive'],
               'mettle': ['positive'], 'microscope': ['trust'], 'microscopy': ['positive'],
               'midwife': ['anticipation', 'joy', 'negative', 'positive', 'trust'], 'midwifery': ['positive'],
               'mighty': ['anger', 'fear', 'joy', 'positive', 'trust'], 'mildew': ['disgust', 'negative'],
               'military': ['fear'], 'militia': ['anger', 'fear', 'negative', 'sadness'], 'mill': ['anticipation'],
               'mime': ['positive'], 'mimicry': ['negative', 'surprise'], 'mindful': ['positive'],
               'mindfulness': ['positive'], 'minimize': ['negative'], 'minimum': ['negative'],
               'ministry': ['joy', 'positive', 'trust'], 'minority': ['negative'],
               'miracle': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'miraculous': ['joy', 'positive', 'surprise'], 'mire': ['disgust', 'negative'],
               'mirth': ['joy', 'positive'], 'misbehavior': ['anger', 'disgust', 'negative', 'surprise'],
               'miscarriage': ['fear', 'negative', 'sadness'], 'mischief': ['negative'], 'mischievous': ['negative'],
               'misconception': ['anger', 'disgust', 'fear', 'negative'], 'misconduct': ['disgust', 'negative'],
               'miserable': ['anger', 'disgust', 'negative', 'sadness'], 'miserably': ['negative', 'sadness'],
               'misery': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'misfortune': ['fear', 'negative', 'sadness'], 'misguided': ['disgust', 'negative'],
               'mishap': ['disgust', 'fear', 'negative', 'sadness', 'surprise'], 'misinterpretation': ['negative'],
               'mislead': ['anger', 'fear', 'negative', 'trust'], 'misleading': ['anger', 'disgust', 'negative'],
               'mismanagement': ['negative'], 'mismatch': ['negative'], 'misnomer': ['negative'],
               'misplace': ['anger', 'disgust', 'negative'], 'misplaced': ['negative'], 'misrepresent': ['negative'],
               'misrepresentation': ['negative', 'sadness'], 'misrepresented': ['anger', 'negative'],
               'missile': ['fear'], 'missing': ['fear', 'negative', 'sadness'], 'missionary': ['positive'],
               'misstatement': ['anger', 'disgust', 'negative'], 'mistake': ['negative', 'sadness'],
               'mistaken': ['fear', 'negative'], 'mistress': ['anger', 'disgust', 'negative'],
               'mistrust': ['disgust', 'fear', 'negative'], 'misunderstand': ['negative'],
               'misunderstanding': ['anger', 'negative', 'sadness'], 'misuse': ['negative'],
               'mite': ['disgust', 'negative'], 'moan': ['fear', 'sadness'], 'moat': ['trust'],
               'mob': ['anger', 'fear', 'negative'], 'mobile': ['anticipation'], 'mockery': ['disgust', 'negative'],
               'mocking': ['anger', 'disgust', 'negative', 'sadness'], 'model': ['positive'], 'moderate': ['positive'],
               'moderator': ['positive', 'trust'], 'modest': ['positive', 'trust'], 'modesty': ['positive'],
               'modify': ['surprise'], 'molestation': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'momentum': ['anticipation', 'positive'], 'monetary': ['anticipation', 'positive'],
               'money': ['anger', 'anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'monk': ['positive', 'trust'], 'monochrome': ['disgust', 'negative'], 'monogamy': ['trust'],
               'monopolist': ['negative'], 'monsoon': ['negative', 'sadness'], 'monster': ['fear', 'negative'],
               'monstrosity': ['anger', 'disgust', 'fear', 'negative', 'surprise'], 'monument': ['positive'],
               'moody': ['anger', 'negative', 'sadness'], 'moorings': ['trust'], 'moot': ['negative'],
               'moral': ['anger', 'positive', 'trust'], 'morality': ['positive', 'trust'],
               'morals': ['anger', 'anticipation', 'disgust', 'joy', 'positive', 'surprise', 'trust'],
               'morbid': ['negative', 'sadness'], 'morbidity': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'morgue': ['disgust', 'fear', 'negative', 'sadness'], 'moribund': ['negative', 'sadness'],
               'morn': ['anticipation'], 'moron': ['negative'], 'moronic': ['negative'], 'morrow': ['anticipation'],
               'morsel': ['negative'], 'mortal': ['negative'], 'mortality': ['anger', 'fear', 'negative', 'sadness'],
               'mortar': ['positive'], 'mortgage': ['fear'], 'mortgagee': ['trust'], 'mortgagor': ['fear'],
               'mortification': ['anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'mortuary': ['fear', 'negative', 'sadness'], 'mosque': ['anger'],
               'mosquito': ['anger', 'disgust', 'negative'],
               'mother': ['anticipation', 'joy', 'negative', 'positive', 'sadness', 'trust'],
               'motherhood': ['joy', 'positive', 'trust'], 'motion': ['anticipation'], 'motive': ['positive'],
               'mountain': ['anticipation'], 'mourn': ['negative', 'sadness'],
               'mournful': ['anger', 'fear', 'negative', 'sadness'], 'mourning': ['negative', 'sadness'],
               'mouth': ['surprise'], 'mouthful': ['disgust'], 'movable': ['positive'], 'mover': ['positive'],
               'muck': ['disgust', 'negative'], 'mucous': ['disgust'], 'mucus': ['disgust'], 'mud': ['negative'],
               'muddle': ['negative'], 'muddled': ['negative'], 'muddy': ['disgust', 'negative'],
               'muff': ['anger', 'disgust', 'negative'], 'mug': ['anger', 'fear', 'negative', 'positive', 'sadness'],
               'mule': ['anger', 'negative', 'trust'], 'mum': ['fear', 'negative'], 'mumble': ['negative'],
               'mumps': ['negative'], 'murder': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'murderer': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'murderous': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'murky': ['disgust', 'negative', 'sadness'], 'muscular': ['positive'],
               'music': ['joy', 'positive', 'sadness'],
               'musical': ['anger', 'anticipation', 'joy', 'positive', 'sadness', 'surprise', 'trust'],
               'musket': ['fear'], 'muss': ['negative'], 'musty': ['disgust', 'negative'],
               'mutable': ['anticipation', 'positive'], 'mutant': ['negative'], 'mutilated': ['disgust', 'negative'],
               'mutilation': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'mutiny': ['anger', 'disgust', 'fear', 'negative', 'surprise'], 'mutter': ['anger', 'negative'],
               'mutual': ['positive'], 'muzzle': ['fear', 'negative'], 'myopia': ['anger', 'negative', 'sadness'],
               'mysterious': ['anticipation', 'fear', 'surprise'], 'mystery': ['anticipation', 'surprise'],
               'mystic': ['surprise'], 'nab': ['negative', 'surprise'], 'nadir': ['negative'],
               'nag': ['anger', 'negative'], 'naive': ['negative'], 'nameless': ['disgust', 'negative'],
               'nap': ['joy', 'positive'], 'napkin': ['sadness'], 'nappy': ['disgust', 'negative'],
               'narcotic': ['negative'], 'nascent': ['anticipation'],
               'nasty': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'nation': ['trust'],
               'naturalist': ['positive'], 'naughty': ['negative'], 'nausea': ['disgust', 'negative'],
               'nauseous': ['disgust', 'negative', 'sadness'], 'navigable': ['anticipation', 'positive'],
               'navigator': ['anticipation', 'trust'], 'nay': ['negative'], 'neatly': ['positive'],
               'necessity': ['sadness'], 'nectar': ['positive'], 'needful': ['negative'], 'needle': ['positive'],
               'needy': ['negative'], 'nefarious': ['disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'negation': ['anger', 'negative'], 'negative': ['negative', 'sadness'], 'neglect': ['negative'],
               'neglected': ['anger', 'disgust', 'negative', 'sadness'], 'neglecting': ['negative'],
               'negligence': ['negative'], 'negligent': ['negative'], 'negligently': ['negative'],
               'negotiate': ['positive', 'trust'], 'negotiator': ['positive'], 'negro': ['negative', 'sadness'],
               'neighbor': ['anticipation', 'positive', 'trust'], 'neighborhood': ['anticipation'],
               'nepotism': ['anger', 'disgust', 'negative', 'sadness'], 'nerve': ['positive'],
               'nervous': ['anticipation', 'fear', 'negative'], 'nervousness': ['fear'], 'nest': ['trust'],
               'nestle': ['positive', 'trust'], 'nether': ['anger', 'fear', 'negative', 'sadness'],
               'nettle': ['anger', 'disgust', 'negative'], 'network': ['anticipation'],
               'neuralgia': ['fear', 'negative', 'sadness'], 'neurosis': ['fear', 'negative', 'sadness'],
               'neurotic': ['disgust', 'fear', 'negative'], 'neutral': ['anticipation', 'trust'],
               'neutrality': ['trust'], 'newcomer': ['fear', 'surprise'], 'nicotine': ['disgust', 'negative'],
               'nigger': ['negative'], 'nightmare': ['fear', 'negative'], 'nihilism': ['negative'],
               'nobility': ['anticipation', 'positive', 'trust'], 'noble': ['positive', 'trust'],
               'nobleman': ['positive', 'trust'], 'noise': ['negative'], 'noisy': ['anger', 'negative'],
               'nomination': ['positive'], 'noncompliance': ['anger', 'anticipation', 'fear', 'negative', 'sadness'],
               'nonsense': ['negative'], 'nonsensical': ['negative', 'sadness'], 'noose': ['negative', 'sadness'],
               'normality': ['positive'], 'nose': ['disgust'], 'notable': ['joy', 'positive', 'trust'],
               'notables': ['positive'], 'notary': ['trust'], 'noted': ['positive'],
               'nothingness': ['negative', 'sadness'], 'notification': ['anticipation'], 'notion': ['positive'],
               'notoriety': ['anger', 'disgust', 'fear', 'negative', 'positive'], 'nourishment': ['positive'],
               'noxious': ['disgust', 'fear', 'negative'], 'nuisance': ['anger', 'negative'], 'nul': ['negative'],
               'nullify': ['negative', 'surprise'], 'numb': ['negative'], 'numbers': ['positive'],
               'numbness': ['negative', 'sadness'], 'nun': ['negative', 'trust'], 'nurse': ['positive', 'trust'],
               'nursery': ['joy', 'positive', 'trust'],
               'nurture': ['anger', 'anticipation', 'disgust', 'fear', 'joy', 'positive', 'trust'],
               'nutritious': ['positive', 'sadness'], 'oaf': ['negative'], 'oak': ['positive'],
               'oasis': ['anticipation', 'joy', 'positive', 'trust'], 'oath': ['positive', 'trust'],
               'obedience': ['positive', 'trust'], 'obese': ['disgust', 'negative'],
               'obesity': ['disgust', 'negative', 'sadness'], 'obey': ['fear', 'trust'],
               'obi': ['disgust', 'fear', 'negative'], 'obit': ['negative', 'sadness', 'surprise'],
               'obituary': ['negative', 'sadness'], 'objection': ['anger', 'negative'], 'objectionable': ['negative'],
               'objective': ['anticipation', 'positive', 'trust'], 'oblige': ['negative', 'trust'],
               'obliging': ['anger', 'anticipation', 'disgust', 'fear', 'joy', 'positive', 'surprise', 'trust'],
               'obligor': ['fear', 'negative'], 'obliterate': ['anger', 'fear', 'negative', 'sadness'],
               'obliterated': ['anger', 'fear', 'negative'], 'obliteration': ['fear', 'negative', 'sadness'],
               'oblivion': ['anger', 'fear', 'negative'], 'obnoxious': ['anger', 'disgust', 'negative', 'sadness'],
               'obscene': ['disgust', 'negative'], 'obscenity': ['anger', 'disgust', 'negative'],
               'obscurity': ['negative'], 'observant': ['positive'],
               'obstacle': ['anger', 'fear', 'negative', 'sadness'], 'obstetrician': ['trust'],
               'obstinate': ['negative'], 'obstruct': ['anger', 'fear', 'negative'], 'obstruction': ['negative'],
               'obstructive': ['anger', 'negative'], 'obtainable': ['joy', 'positive'], 'obtuse': ['negative'],
               'obvious': ['positive', 'trust'], 'occasional': ['surprise'], 'occult': ['disgust', 'fear', 'negative'],
               'occupant': ['positive', 'trust'], 'occupation': ['positive'], 'occupy': ['positive'],
               'octopus': ['disgust'], 'oddity': ['disgust', 'negative', 'sadness', 'surprise'],
               'odious': ['anger', 'disgust', 'fear', 'negative'], 'odor': ['negative'],
               'offend': ['anger', 'disgust', 'negative'], 'offended': ['anger', 'negative', 'sadness'],
               'offender': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'offense': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'offensive': ['anger', 'disgust', 'negative'], 'offer': ['positive'], 'offering': ['trust'],
               'offhand': ['negative'], 'officer': ['positive', 'trust'], 'official': ['trust'], 'offing': ['negative'],
               'offset': ['anticipation', 'positive'], 'offshoot': ['positive'],
               'ogre': ['disgust', 'fear', 'negative'], 'older': ['sadness'], 'olfactory': ['anticipation', 'negative'],
               'oligarchy': ['negative'], 'omen': ['anticipation', 'fear', 'negative'],
               'ominous': ['anticipation', 'fear', 'negative'], 'omit': ['negative'],
               'omnipotence': ['fear', 'negative', 'positive'], 'omniscient': ['positive', 'trust'],
               'onerous': ['anger', 'negative', 'sadness'], 'ongoing': ['anticipation'], 'onset': ['anticipation'],
               'onus': ['negative'], 'ooze': ['disgust', 'negative'], 'opaque': ['negative'], 'openness': ['positive'],
               'opera': ['anger', 'anticipation', 'fear', 'joy', 'positive', 'sadness', 'surprise', 'trust'],
               'operatic': ['negative'], 'operation': ['fear', 'trust'], 'opiate': ['negative'],
               'opinionated': ['anger', 'negative'], 'opium': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'opponent': ['anger', 'anticipation', 'disgust', 'fear', 'negative'], 'opportune': ['joy', 'positive'],
               'opportunity': ['anticipation', 'positive'], 'oppose': ['negative'],
               'opposed': ['anger', 'fear', 'negative'], 'opposition': ['anger', 'negative'],
               'oppress': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'oppression': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'oppressive': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'oppressor': ['anger', 'fear', 'negative', 'sadness'],
               'optimism': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'optimist': ['positive', 'trust'],
               'option': ['positive'], 'optional': ['positive'], 'oracle': ['anticipation', 'positive', 'trust'],
               'oration': ['positive'], 'orc': ['anger', 'disgust', 'fear', 'negative'],
               'orchestra': ['anger', 'joy', 'positive', 'sadness', 'trust'], 'ordained': ['positive', 'trust'],
               'ordeal': ['anger', 'negative', 'surprise'], 'orderly': ['positive'], 'ordinance': ['trust'],
               'ordination': ['anticipation', 'joy', 'positive', 'trust'], 'ordnance': ['fear', 'negative'],
               'organ': ['anticipation', 'joy'], 'organic': ['positive'],
               'organization': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'organize': ['positive'],
               'organized': ['positive'], 'orgasm': ['anticipation', 'joy', 'positive'],
               'originality': ['positive', 'surprise'], 'ornamented': ['positive'], 'ornate': ['positive'],
               'orphan': ['fear', 'negative', 'sadness'], 'orthodoxy': ['trust'], 'ostensibly': ['negative'],
               'oust': ['anger', 'negative', 'sadness'],
               'outburst': ['anger', 'fear', 'joy', 'negative', 'positive', 'sadness', 'surprise'],
               'outcast': ['disgust', 'fear', 'negative', 'sadness'], 'outcome': ['positive'],
               'outcry': ['anger', 'fear', 'negative', 'surprise'], 'outdo': ['anticipation', 'positive'],
               'outhouse': ['disgust', 'negative'], 'outlandish': ['negative'], 'outlaw': ['negative'],
               'outpost': ['fear'], 'outrage': ['anger', 'disgust', 'negative'], 'outrageous': ['surprise'],
               'outsider': ['fear'], 'outstanding': ['joy', 'negative', 'positive'], 'outward': ['positive'],
               'ovation': ['negative', 'sadness'], 'overbearing': ['anger', 'negative'], 'overburden': ['negative'],
               'overcast': ['negative'], 'overdo': ['negative'], 'overdose': ['negative'],
               'overdue': ['anticipation', 'negative', 'sadness', 'surprise'], 'overestimate': ['surprise'],
               'overestimated': ['negative'], 'overflow': ['negative'], 'overgrown': ['negative'],
               'overjoyed': ['joy', 'positive'], 'overload': ['negative', 'sadness'], 'overpaid': ['negative'],
               'overpower': ['negative'], 'overpowering': ['anger', 'fear', 'negative'],
               'overpriced': ['anger', 'disgust', 'negative', 'sadness'], 'oversight': ['negative'], 'overt': ['fear'],
               'overthrow': ['anticipation', 'fear', 'negative'], 'overture': ['anticipation'],
               'overturn': ['negative'], 'overwhelm': ['negative'], 'overwhelmed': ['negative', 'sadness'],
               'overwhelming': ['positive'],
               'owing': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness', 'trust'],
               'ownership': ['positive'], 'oxidation': ['negative'], 'pacific': ['positive'], 'pact': ['trust'],
               'pad': ['positive'], 'padding': ['positive'], 'paddle': ['anticipation', 'positive'],
               'pain': ['fear', 'negative', 'sadness'], 'pained': ['fear', 'negative', 'sadness'],
               'painful': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'painfully': ['negative', 'sadness'],
               'pains': ['negative'], 'palatable': ['positive'], 'palliative': ['positive'], 'palpable': ['surprise'],
               'palsy': ['disgust', 'fear', 'negative', 'sadness'], 'panacea': ['positive'], 'panache': ['positive'],
               'pandemic': ['fear', 'negative', 'sadness'], 'pang': ['negative', 'surprise'],
               'panic': ['fear', 'negative'], 'panier': ['positive'], 'paprika': ['positive'], 'parachute': ['fear'],
               'parade': ['anticipation', 'fear', 'joy', 'negative', 'positive', 'surprise'],
               'paragon': ['anticipation', 'joy', 'positive', 'trust'],
               'paralysis': ['anger', 'anticipation', 'fear', 'negative', 'sadness'],
               'paralyzed': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'paramount': ['positive'],
               'paranoia': ['fear', 'negative'], 'paraphrase': ['negative', 'positive'],
               'parasite': ['disgust', 'fear', 'negative'], 'pardon': ['positive'],
               'pare': ['anger', 'anticipation', 'fear', 'negative', 'sadness'], 'parentage': ['positive'],
               'parietal': ['positive', 'trust'], 'parish': ['trust'], 'parliament': ['trust'],
               'parole': ['anticipation'], 'parrot': ['disgust', 'negative'], 'parsimonious': ['negative'],
               'partake': ['positive', 'trust'], 'participation': ['positive'], 'parting': ['sadness'],
               'partisan': ['negative'], 'partner': ['positive'], 'partnership': ['positive', 'trust'],
               'passe': ['negative'], 'passenger': ['anticipation'],
               'passion': ['anticipation', 'joy', 'positive', 'trust'],
               'passionate': ['anticipation', 'joy', 'positive', 'trust'], 'passive': ['negative'],
               'passivity': ['negative'], 'pastime': ['positive'], 'pastor': ['joy', 'positive', 'trust'],
               'pastry': ['joy', 'positive'], 'pasture': ['positive'], 'patch': ['negative'], 'patent': ['positive'],
               'pathetic': ['disgust', 'negative', 'sadness'], 'patience': ['anticipation', 'positive', 'trust'],
               'patient': ['anticipation', 'positive'], 'patriarchal': ['positive', 'trust'], 'patrol': ['trust'],
               'patron': ['positive', 'trust'], 'patronage': ['positive', 'trust'], 'patronizing': ['negative'],
               'patter': ['anger', 'negative'], 'paucity': ['anger', 'disgust', 'negative', 'sadness'],
               'pauper': ['negative', 'sadness'], 'pavement': ['trust'], 'pawn': ['negative', 'trust'],
               'pay': ['anticipation', 'joy', 'positive', 'trust'], 'payback': ['anger', 'negative'],
               'payment': ['negative'], 'peace': ['anticipation', 'joy', 'positive', 'trust'],
               'peaceable': ['positive'], 'peaceful': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'peacock': ['positive'], 'peck': ['positive'], 'peculiarities': ['negative'],
               'peculiarity': ['positive'], 'pedestrian': ['negative'], 'pedigree': ['positive', 'trust'],
               'peerless': ['positive'], 'penal': ['fear', 'negative', 'sadness'],
               'penalty': ['anger', 'fear', 'negative', 'sadness'], 'penance': ['fear', 'sadness'],
               'penchant': ['positive'], 'penetration': ['anger', 'fear', 'negative'],
               'penitentiary': ['anger', 'negative'], 'pensive': ['sadness'], 'perceive': ['positive', 'trust'],
               'perceptible': ['positive'], 'perchance': ['surprise'],
               'perdition': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'perennial': ['positive', 'trust'],
               'perfect': ['anticipation', 'joy', 'positive', 'trust'],
               'perfection': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'performer': ['positive'],
               'peri': ['surprise'], 'peril': ['anticipation', 'fear', 'negative', 'sadness'],
               'perilous': ['anticipation', 'fear', 'negative', 'sadness'], 'periodicity': ['trust'],
               'perish': ['fear', 'negative', 'sadness'], 'perishable': ['negative'],
               'perished': ['negative', 'sadness'], 'perishing': ['fear', 'negative', 'sadness'],
               'perjury': ['fear', 'negative', 'surprise'], 'perk': ['positive'], 'permission': ['positive'],
               'pernicious': ['anger', 'fear', 'negative', 'sadness'],
               'perpetrator': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'perpetuate': ['anticipation'],
               'perpetuation': ['negative'], 'perpetuity': ['anticipation', 'positive', 'trust'],
               'perplexed': ['negative'], 'perplexity': ['negative', 'sadness'],
               'persecute': ['anger', 'fear', 'negative'],
               'persecution': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'persistence': ['positive'],
               'persistent': ['positive'], 'personable': ['positive'], 'personal': ['trust'],
               'perspiration': ['disgust'], 'persuade': ['trust'], 'pertinent': ['positive', 'trust'],
               'perturbation': ['fear', 'negative'], 'pertussis': ['negative'],
               'perverse': ['anger', 'disgust', 'fear', 'negative'],
               'perversion': ['anger', 'disgust', 'negative', 'sadness'], 'pervert': ['anger', 'disgust', 'negative'],
               'perverted': ['disgust', 'negative'], 'pessimism': ['anger', 'fear', 'negative', 'sadness'],
               'pessimist': ['fear', 'negative', 'sadness'], 'pest': ['anger', 'disgust', 'fear', 'negative'],
               'pestilence': ['disgust', 'fear', 'negative'], 'pet': ['negative'],
               'petroleum': ['disgust', 'negative', 'positive'], 'petty': ['negative'], 'phalanx': ['fear', 'trust'],
               'phantom': ['fear', 'negative'], 'pharmaceutical': ['positive'], 'philanthropic': ['trust'],
               'philanthropist': ['positive', 'trust'], 'philanthropy': ['positive'],
               'philosopher': ['positive', 'trust'], 'phlegm': ['disgust', 'negative'], 'phoenix': ['positive'],
               'phonetic': ['positive'], 'phony': ['anger', 'disgust', 'negative'], 'physician': ['positive', 'trust'],
               'physicist': ['trust'], 'physics': ['positive'], 'physiology': ['positive'], 'physique': ['positive'],
               'pick': ['positive'], 'picket': ['anger', 'anticipation', 'fear', 'negative'],
               'picketing': ['anger', 'negative'], 'pickle': ['negative'],
               'picnic': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'picturesque': ['joy', 'positive'],
               'piety': ['positive'], 'pig': ['disgust', 'negative'], 'pigeon': ['negative'],
               'piles': ['disgust', 'negative'], 'pill': ['positive', 'trust'],
               'pillage': ['anger', 'disgust', 'fear', 'negative'], 'pillow': ['positive'],
               'pilot': ['positive', 'trust'], 'pimp': ['negative'], 'pimple': ['disgust', 'negative'],
               'pine': ['negative', 'sadness'], 'pinion': ['fear', 'negative'], 'pinnacle': ['positive'],
               'pioneer': ['positive'], 'pious': ['disgust', 'positive', 'sadness', 'trust'],
               'pique': ['anger', 'negative'], 'piracy': ['negative'], 'pirate': ['anger', 'negative'],
               'pistol': ['negative'], 'pitfall': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'pith': ['positive', 'trust'], 'pity': ['sadness'], 'pivot': ['positive', 'trust'],
               'placard': ['surprise'], 'placid': ['positive'], 'plagiarism': ['disgust', 'negative'],
               'plague': ['disgust', 'fear', 'negative', 'sadness'], 'plaintiff': ['negative'],
               'plaintive': ['sadness'], 'plan': ['anticipation'], 'planning': ['anticipation', 'positive', 'trust'],
               'plated': ['negative'], 'player': ['negative'],
               'playful': ['anger', 'joy', 'positive', 'surprise', 'trust'],
               'playground': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'playhouse': ['joy', 'positive'],
               'plea': ['anticipation', 'fear', 'negative', 'sadness'],
               'pleasant': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'pleased': ['joy', 'positive'],
               'pleasurable': ['joy', 'positive'], 'pledge': ['joy', 'positive', 'trust'], 'plenary': ['positive'],
               'plentiful': ['positive'], 'plexus': ['positive'],
               'plight': ['anticipation', 'disgust', 'fear', 'negative', 'sadness'], 'plodding': ['negative'],
               'plum': ['positive'], 'plumb': ['positive'], 'plummet': ['fear', 'negative'], 'plump': ['anticipation'],
               'plunder': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'plunge': ['fear', 'negative'], 'plush': ['positive'], 'ply': ['positive'],
               'pneumonia': ['fear', 'negative'], 'poaching': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'pointedly': ['positive'], 'pointless': ['negative', 'sadness'],
               'poison': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'poisoned': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'poisoning': ['disgust', 'negative'],
               'poisonous': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'poke': ['anticipation', 'negative'],
               'polarity': ['surprise'], 'polemic': ['anger', 'disgust', 'negative'],
               'police': ['fear', 'positive', 'trust'], 'policeman': ['fear', 'positive', 'trust'], 'policy': ['trust'],
               'polio': ['fear', 'negative', 'sadness'], 'polish': ['positive'], 'polished': ['positive'],
               'polite': ['positive'], 'politeness': ['positive'], 'politic': ['disgust', 'positive'],
               'politics': ['anger'], 'poll': ['trust'], 'pollute': ['disgust', 'negative'],
               'pollution': ['disgust', 'negative'], 'polygamy': ['disgust', 'negative'], 'pomp': ['negative'],
               'pompous': ['disgust', 'negative'], 'ponderous': ['negative'], 'pontiff': ['trust'],
               'pool': ['positive'], 'poorly': ['negative'], 'pop': ['negative', 'surprise'], 'pope': ['positive'],
               'popularity': ['positive'], 'popularized': ['positive'], 'population': ['positive'],
               'porcupine': ['negative'], 'porn': ['disgust', 'negative'], 'porno': ['negative'],
               'pornographic': ['negative'], 'pornography': ['disgust', 'negative'], 'portable': ['positive'],
               'pose': ['negative'], 'posse': ['fear'], 'possess': ['anticipation', 'joy', 'positive', 'trust'],
               'possessed': ['anger', 'disgust', 'fear', 'negative'],
               'possession': ['anger', 'disgust', 'fear', 'negative'], 'possibility': ['anticipation'],
               'posthumous': ['negative', 'sadness'], 'postponement': ['negative', 'surprise'], 'potable': ['positive'],
               'potency': ['positive'], 'pound': ['anger', 'negative'],
               'poverty': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'pow': ['anger'],
               'powerful': ['anger', 'anticipation', 'disgust', 'fear', 'joy', 'positive', 'trust'],
               'powerfully': ['fear', 'positive'], 'powerless': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'practice': ['positive'], 'practiced': ['joy', 'positive', 'surprise', 'trust'],
               'practise': ['anticipation', 'positive'], 'praise': ['joy', 'positive', 'trust'],
               'praised': ['joy', 'positive'], 'praiseworthy': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'prank': ['negative', 'surprise'],
               'pray': ['anticipation', 'fear', 'joy', 'positive', 'surprise', 'trust'],
               'precarious': ['anticipation', 'fear', 'negative', 'sadness'], 'precaution': ['positive'],
               'precede': ['positive'], 'precedence': ['positive', 'trust'], 'precinct': ['negative'],
               'precious': ['anticipation', 'joy', 'positive', 'surprise'], 'precipice': ['fear'],
               'precise': ['positive'], 'precision': ['positive'], 'preclude': ['anger'], 'precursor': ['anticipation'],
               'predatory': ['negative'], 'predicament': ['fear', 'negative'], 'predict': ['anticipation'],
               'prediction': ['anticipation'], 'predilection': ['anticipation', 'positive'],
               'predispose': ['anticipation'], 'predominant': ['positive', 'trust'], 'preeminent': ['positive'],
               'prefer': ['positive', 'trust'], 'pregnancy': ['disgust', 'negative'],
               'prejudice': ['anger', 'negative'], 'prejudiced': ['disgust', 'fear', 'negative'],
               'prejudicial': ['anger', 'negative'], 'preliminary': ['anticipation'], 'premature': ['surprise'],
               'premeditated': ['fear', 'negative'], 'premier': ['positive'], 'premises': ['positive'],
               'preparation': ['anticipation'], 'preparatory': ['anticipation'],
               'prepare': ['anticipation', 'positive'], 'prepared': ['anticipation', 'positive', 'trust'],
               'preparedness': ['anticipation'], 'preponderance': ['trust'], 'prerequisite': ['anticipation'],
               'prescient': ['anticipation', 'positive'], 'presence': ['positive'],
               'present': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'presentable': ['positive'],
               'presentment': ['negative', 'positive'],
               'preservative': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'preserve': ['positive'],
               'president': ['positive', 'trust'], 'pressure': ['negative'], 'prestige': ['joy', 'positive', 'trust'],
               'presto': ['joy', 'positive', 'surprise'], 'presumption': ['trust'],
               'presumptuous': ['anger', 'disgust', 'negative'], 'pretend': ['negative'], 'pretended': ['negative'],
               'pretending': ['anger', 'negative'], 'pretense': ['negative'], 'pretensions': ['negative'],
               'pretty': ['anticipation', 'joy', 'positive', 'trust'], 'prevail': ['anticipation', 'joy', 'positive'],
               'prevalent': ['trust'], 'prevent': ['fear'], 'prevention': ['anticipation', 'positive'],
               'preventive': ['negative'], 'prey': ['fear', 'negative'], 'priceless': ['positive'],
               'prick': ['anger', 'disgust', 'fear', 'negative', 'surprise'], 'prickly': ['negative'],
               'pride': ['joy', 'positive'], 'priest': ['positive', 'trust'],
               'priesthood': ['anticipation', 'joy', 'positive', 'sadness', 'trust'], 'priestly': ['positive'],
               'primacy': ['positive'], 'primary': ['positive'], 'prime': ['positive'], 'primer': ['positive', 'trust'],
               'prince': ['positive'], 'princely': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'princess': ['positive'], 'principal': ['positive', 'trust'],
               'prison': ['anger', 'fear', 'negative', 'sadness'],
               'prisoner': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'pristine': ['positive'],
               'privileged': ['joy', 'positive', 'trust'], 'privy': ['negative', 'trust'],
               'probability': ['anticipation'], 'probation': ['anticipation', 'fear', 'sadness'],
               'probity': ['positive', 'trust'], 'problem': ['fear', 'negative', 'sadness'],
               'procedure': ['fear', 'positive'], 'proceeding': ['anticipation'],
               'procession': ['joy', 'positive', 'sadness', 'surprise'], 'procrastinate': ['negative'],
               'procrastination': ['negative'], 'proctor': ['positive', 'trust'], 'procure': ['positive'],
               'prodigal': ['negative', 'positive'], 'prodigious': ['positive'], 'prodigy': ['positive'],
               'producer': ['positive'], 'producing': ['positive'], 'production': ['anticipation', 'positive'],
               'productive': ['positive'], 'productivity': ['positive'], 'profane': ['anger', 'negative'],
               'profanity': ['anger', 'negative'], 'profession': ['positive'], 'professional': ['positive', 'trust'],
               'professor': ['positive', 'trust'], 'professorship': ['trust'],
               'proficiency': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'proficient': ['positive', 'trust'], 'profuse': ['positive'], 'profusion': ['negative'],
               'prognosis': ['anticipation', 'fear'], 'prognostic': ['anticipation'],
               'progress': ['anticipation', 'joy', 'positive'],
               'progression': ['anticipation', 'joy', 'positive', 'sadness', 'trust'], 'progressive': ['positive'],
               'prohibited': ['anger', 'disgust', 'fear', 'negative'], 'prohibition': ['negative'],
               'projectile': ['fear'], 'projectiles': ['fear'], 'prolific': ['positive'], 'prologue': ['anticipation'],
               'prolong': ['disgust', 'negative'], 'prominence': ['positive'], 'prominently': ['positive'],
               'promiscuous': ['negative'], 'promise': ['joy', 'positive', 'trust'], 'promising': ['positive'],
               'promotion': ['positive'], 'proof': ['trust'], 'prop': ['positive'], 'propaganda': ['negative'],
               'proper': ['positive'], 'prophecy': ['anticipation'], 'prophet': ['anticipation', 'positive', 'trust'],
               'prophetic': ['anticipation'], 'prophylactic': ['anticipation', 'positive', 'trust'],
               'proposition': ['positive'], 'prosecute': ['anger', 'fear', 'negative', 'sadness'],
               'prosecution': ['disgust', 'negative'], 'prospect': ['positive'], 'prospectively': ['anticipation'],
               'prosper': ['anticipation', 'joy', 'positive'], 'prosperity': ['positive'],
               'prosperous': ['joy', 'positive'], 'prostitute': ['disgust', 'negative'],
               'prostitution': ['disgust', 'negative', 'sadness'], 'protect': ['positive'], 'protected': ['trust'],
               'protecting': ['positive', 'trust'], 'protective': ['positive'], 'protector': ['positive', 'trust'],
               'protestant': ['fear'], 'proud': ['anticipation', 'joy', 'positive', 'trust'], 'prove': ['positive'],
               'proven': ['trust'], 'proverbial': ['positive'], 'provide': ['positive', 'trust'],
               'providing': ['anticipation', 'joy', 'positive', 'trust'], 'proviso': ['trust'],
               'provocation': ['anger', 'negative'], 'provoking': ['anger', 'disgust', 'negative'],
               'prowl': ['fear', 'surprise'], 'proxy': ['trust'], 'prudence': ['positive'],
               'prudent': ['positive', 'trust'], 'pry': ['anger', 'anticipation', 'negative', 'trust'],
               'prying': ['negative'], 'psalm': ['positive'], 'psychosis': ['anger', 'fear', 'negative', 'sadness'],
               'public': ['anticipation', 'positive'], 'publicist': ['negative'], 'puffy': ['negative'],
               'puke': ['disgust'], 'pull': ['positive'], 'pulpit': ['positive'], 'puma': ['fear', 'surprise'],
               'punch': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'punctual': ['anticipation', 'positive', 'trust'], 'punctuality': ['positive'],
               'pungent': ['disgust', 'negative'], 'punish': ['fear', 'negative'],
               'punished': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'punishing': ['anger', 'fear', 'negative', 'sadness'],
               'punishment': ['anger', 'disgust', 'fear', 'negative'],
               'punitive': ['anger', 'fear', 'negative', 'sadness'], 'punt': ['anticipation'],
               'puppy': ['anticipation', 'positive', 'trust'], 'purely': ['positive', 'trust'],
               'purgatory': ['disgust', 'fear', 'negative'], 'purge': ['fear', 'negative'],
               'purification': ['positive', 'trust'], 'purify': ['joy', 'positive', 'trust'], 'purist': ['positive'],
               'purity': ['positive', 'surprise'], 'purr': ['joy', 'positive', 'trust'], 'pus': ['disgust', 'negative'],
               'putative': ['trust'], 'quack': ['disgust', 'negative'], 'quagmire': ['disgust', 'negative'],
               'quail': ['fear', 'negative'], 'quaint': ['joy', 'positive', 'trust'], 'quake': ['fear'],
               'qualified': ['positive', 'trust'], 'qualifying': ['positive'],
               'quandary': ['anger', 'fear', 'negative'], 'quarantine': ['fear'], 'quarrel': ['anger', 'negative'],
               'quash': ['fear', 'negative'], 'quell': ['negative'], 'quest': ['anticipation', 'positive'],
               'question': ['positive'], 'questionable': ['disgust', 'negative'], 'quicken': ['anticipation'],
               'quickness': ['positive', 'surprise'], 'quicksilver': ['negative', 'surprise'],
               'quiescent': ['positive'], 'quiet': ['positive', 'sadness'], 'quinine': ['positive'],
               'quit': ['negative'], 'quiver': ['fear', 'negative'], 'quivering': ['fear', 'negative'],
               'quiz': ['negative'], 'quote': ['anticipation', 'negative', 'positive', 'surprise'],
               'rabble': ['anger', 'fear', 'negative'],
               'rabid': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'], 'rabies': ['negative'],
               'rack': ['negative', 'sadness'], 'racket': ['negative'], 'radar': ['trust'],
               'radiance': ['anticipation', 'joy', 'positive', 'trust'], 'radiant': ['joy', 'positive'],
               'radiate': ['positive'], 'radiation': ['fear', 'negative'], 'radio': ['positive'],
               'radioactive': ['fear', 'negative'], 'radon': ['fear', 'negative'],
               'raffle': ['anticipation', 'surprise'], 'rage': ['anger', 'negative'],
               'raging': ['anger', 'disgust', 'fear', 'negative'], 'rags': ['disgust', 'negative'],
               'raid': ['anger', 'fear', 'negative', 'surprise'], 'rail': ['anger', 'anticipation', 'negative'],
               'rainy': ['sadness'], 'ram': ['anger', 'anticipation', 'negative'], 'rambling': ['negative'],
               'rampage': ['anger', 'fear', 'negative'], 'rancid': ['disgust', 'negative'], 'randomly': ['surprise'],
               'ranger': ['trust'], 'ransom': ['anger', 'fear', 'negative'],
               'rape': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'rapid': ['surprise'], 'rapping': ['anger'],
               'rapt': ['joy', 'positive', 'surprise', 'trust'], 'raptors': ['fear', 'negative'],
               'rapture': ['anticipation', 'joy', 'positive'], 'rarity': ['surprise'],
               'rascal': ['anger', 'disgust', 'fear', 'negative'], 'rash': ['disgust', 'negative'],
               'rat': ['disgust', 'fear', 'negative'], 'ratify': ['trust'],
               'rating': ['anger', 'fear', 'negative', 'sadness'], 'rational': ['positive', 'trust'],
               'rattlesnake': ['fear'], 'raucous': ['negative'],
               'rave': ['anger', 'disgust', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'ravenous': ['anger', 'fear', 'negative', 'sadness'], 'ravine': ['fear'],
               'raving': ['anger', 'anticipation', 'fear', 'joy', 'negative', 'surprise'], 'razor': ['fear'],
               'react': ['anger', 'fear'], 'reactionary': ['negative'], 'reader': ['positive'], 'readily': ['positive'],
               'readiness': ['anticipation', 'joy', 'positive', 'trust'], 'reading': ['positive'],
               'ready': ['anticipation'], 'reaffirm': ['positive'], 'real': ['positive', 'trust'],
               'reappear': ['surprise'], 'rear': ['negative'], 'reason': ['positive'],
               'reassurance': ['positive', 'trust'], 'reassure': ['positive', 'trust'], 'rebate': ['positive'],
               'rebel': ['anger', 'fear', 'negative'], 'rebellion': ['anger', 'disgust', 'fear'],
               'rebuke': ['negative'], 'rebut': ['negative'], 'recalcitrant': ['anger', 'disgust', 'negative'],
               'recast': ['positive'], 'received': ['positive'],
               'receiving': ['anticipation', 'joy', 'positive', 'surprise'], 'recesses': ['fear'],
               'recession': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'recherche': ['positive'],
               'recidivism': ['anger', 'disgust', 'negative', 'sadness'], 'recipient': ['anticipation'],
               'reciprocate': ['positive'], 'reckless': ['anger', 'fear', 'negative'],
               'recklessness': ['anger', 'disgust', 'fear', 'negative', 'surprise'], 'reclamation': ['positive'],
               'recline': ['positive', 'trust'], 'recognizable': ['anticipation', 'positive'],
               'recombination': ['anticipation'], 'recommend': ['positive', 'trust'],
               'reconciliation': ['anticipation', 'joy', 'positive', 'trust'], 'reconsideration': ['positive', 'trust'],
               'reconstruct': ['anticipation', 'positive'], 'reconstruction': ['anticipation', 'positive'],
               'recorder': ['positive'], 'recoup': ['positive'], 'recovery': ['positive'],
               'recreation': ['anticipation', 'joy', 'positive'], 'recreational': ['anticipation', 'joy', 'positive'],
               'recruits': ['trust'], 'rectify': ['positive'], 'recurrent': ['anticipation'],
               'redemption': ['positive'], 'redress': ['positive'], 'redundant': ['negative'], 'referee': ['trust'],
               'refine': ['positive'], 'refinement': ['positive'], 'reflex': ['surprise'],
               'reflux': ['disgust', 'negative'], 'reform': ['positive'], 'reformation': ['positive'],
               'reformer': ['positive'], 'refractory': ['negative'], 'refreshing': ['positive'], 'refugee': ['sadness'],
               'refurbish': ['positive'], 'refusal': ['negative'], 'refuse': ['negative'],
               'refused': ['negative', 'sadness'], 'refutation': ['fear'], 'regal': ['positive', 'trust'],
               'regatta': ['anticipation'], 'regent': ['positive', 'trust'], 'regiment': ['fear'],
               'registry': ['trust'], 'regress': ['negative'], 'regression': ['negative'],
               'regressive': ['negative', 'positive'], 'regret': ['negative', 'sadness'],
               'regrettable': ['negative', 'sadness'], 'regretted': ['negative', 'sadness'],
               'regretting': ['negative', 'sadness'], 'regularity': ['anticipation', 'positive', 'trust'],
               'regulate': ['positive'], 'regulatory': ['fear', 'negative'], 'regurgitation': ['disgust'],
               'rehabilitate': ['positive'], 'rehabilitation': ['anticipation', 'positive'], 'reimburse': ['positive'],
               'reimbursement': ['positive', 'trust'], 'rein': ['negative'], 'reinforcement': ['positive', 'trust'],
               'reinforcements': ['trust'], 'reinstate': ['positive'],
               'reject': ['anger', 'fear', 'negative', 'sadness'], 'rejected': ['negative'],
               'rejection': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'rejects': ['anger', 'fear', 'negative', 'sadness'],
               'rejoice': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'rejoicing': ['anticipation', 'joy', 'positive', 'surprise'], 'rejuvenate': ['positive'],
               'rejuvenated': ['positive'],
               'rekindle': ['anticipation', 'fear', 'joy', 'negative', 'positive', 'surprise'],
               'relapse': ['fear', 'negative', 'sadness'], 'related': ['trust'], 'relative': ['trust'],
               'relaxation': ['positive'], 'relegation': ['negative'], 'relevant': ['positive', 'trust'],
               'reliability': ['positive', 'trust'], 'reliable': ['positive', 'trust'],
               'reliance': ['positive', 'trust'], 'relics': ['sadness'], 'relief': ['positive'],
               'relieving': ['positive'], 'religion': ['trust'], 'relinquish': ['negative'],
               'reluctant': ['fear', 'negative'], 'remains': ['disgust', 'fear', 'negative', 'positive', 'trust'],
               'remake': ['positive'], 'remand': ['anger', 'negative'],
               'remarkable': ['joy', 'positive', 'surprise', 'trust'], 'remarkably': ['positive'],
               'remedial': ['negative'], 'remedy': ['anticipation', 'joy', 'positive', 'trust'],
               'remiss': ['anger', 'disgust', 'negative', 'sadness'], 'remission': ['positive'],
               'remodel': ['positive'], 'remorse': ['negative', 'sadness'], 'removal': ['negative'],
               'remove': ['anger', 'fear', 'negative', 'sadness'], 'renaissance': ['positive'],
               'rencontre': ['negative'], 'rend': ['negative'], 'render': ['positive'],
               'renegade': ['anger', 'negative'], 'renewal': ['positive'], 'renounce': ['anger', 'negative'],
               'renovate': ['anticipation', 'positive'], 'renovation': ['anticipation', 'joy', 'positive'],
               'renown': ['positive'], 'renowned': ['positive'], 'renunciation': ['negative'],
               'reorganize': ['positive'], 'reparation': ['positive', 'trust'],
               'repay': ['anger', 'anticipation', 'joy', 'positive', 'trust'],
               'repellant': ['disgust', 'fear', 'negative'], 'repellent': ['anger', 'disgust', 'fear', 'negative'],
               'repelling': ['disgust', 'negative'], 'repent': ['fear', 'positive'], 'replenish': ['positive'],
               'replete': ['positive'], 'reporter': ['positive', 'trust'], 'repose': ['positive'],
               'represented': ['positive'], 'representing': ['anticipation'], 'repress': ['negative', 'sadness'],
               'repression': ['fear', 'negative'], 'reprimand': ['anger', 'negative'], 'reprint': ['negative'],
               'reprisal': ['anger', 'fear', 'negative', 'sadness'],
               'reproach': ['anger', 'disgust', 'negative', 'sadness'], 'reproductive': ['joy', 'positive'],
               'republic': ['negative'], 'repudiation': ['anger', 'disgust', 'negative'],
               'repulsion': ['disgust', 'fear', 'negative'], 'reputable': ['positive', 'trust'], 'requiem': ['sadness'],
               'rescind': ['negative'], 'rescission': ['negative'],
               'rescue': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'resection': ['fear'],
               'resent': ['anger', 'negative'], 'resentful': ['anger', 'negative'],
               'resentment': ['anger', 'disgust', 'negative', 'sadness'], 'reserve': ['positive'],
               'resident': ['positive'], 'resign': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'resignation': ['negative', 'sadness', 'surprise'], 'resigned': ['negative', 'sadness'],
               'resilient': ['positive'], 'resist': ['negative'], 'resistance': ['anger', 'negative'],
               'resistant': ['fear', 'negative'], 'resisting': ['anger', 'fear', 'negative', 'sadness'],
               'resistive': ['positive'], 'resolutely': ['positive'], 'resources': ['joy', 'positive', 'trust'],
               'respect': ['anticipation', 'joy', 'positive', 'trust'], 'respectability': ['positive'],
               'respectable': ['positive', 'trust'], 'respectful': ['positive', 'trust'], 'respecting': ['positive'],
               'respects': ['positive', 'trust'], 'respite': ['joy', 'positive', 'trust'],
               'resplendent': ['joy', 'positive'], 'responsible': ['positive', 'trust'],
               'responsive': ['anticipation', 'positive', 'trust'], 'rest': ['positive'], 'restful': ['positive'],
               'restitution': ['anger', 'positive'], 'restlessness': ['anticipation'],
               'restorative': ['anticipation', 'joy', 'positive', 'trust'], 'restoring': ['positive'],
               'restrain': ['anger', 'fear', 'negative'], 'restrained': ['fear'], 'restraint': ['positive'],
               'restrict': ['negative', 'sadness'], 'restriction': ['anger', 'fear', 'negative', 'sadness'],
               'restrictive': ['negative'], 'result': ['anticipation'], 'resultant': ['anticipation'],
               'resumption': ['positive'], 'retain': ['trust'], 'retaliate': ['anger', 'negative'],
               'retaliation': ['anger', 'fear', 'negative'], 'retaliatory': ['anger', 'negative'],
               'retard': ['disgust', 'fear', 'negative', 'sadness'], 'retardation': ['negative'],
               'retention': ['positive'], 'reticent': ['fear', 'negative'],
               'retirement': ['anticipation', 'fear', 'joy', 'negative', 'positive', 'sadness', 'trust'],
               'retort': ['negative'], 'retract': ['anger', 'negative'], 'retraction': ['negative'],
               'retrenchment': ['fear', 'negative'], 'retribution': ['anger', 'fear', 'negative', 'sadness'],
               'retrograde': ['negative'], 'reunion': ['anticipation', 'positive', 'trust'],
               'revel': ['joy', 'positive'], 'revels': ['joy', 'positive'],
               'revenge': ['anger', 'anticipation', 'fear', 'negative', 'surprise'],
               'revere': ['anticipation', 'joy', 'positive', 'trust'], 'reverence': ['joy', 'positive', 'trust'],
               'reverend': ['joy', 'positive'], 'reverie': ['joy', 'positive', 'trust'],
               'reversal': ['anger', 'disgust', 'negative', 'surprise'], 'revise': ['positive'],
               'revival': ['anticipation', 'joy', 'positive', 'trust'],
               'revive': ['anticipation', 'negative', 'positive'], 'revocation': ['negative'],
               'revoke': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'revolt': ['anger', 'negative', 'surprise'], 'revolting': ['anger', 'disgust', 'fear', 'negative'],
               'revolution': ['anger', 'anticipation', 'fear', 'negative', 'positive', 'sadness', 'surprise'],
               'revolutionary': ['positive'], 'revolver': ['anger', 'fear', 'negative', 'sadness'],
               'revulsion': ['anger', 'disgust', 'fear', 'negative'],
               'reward': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'rheumatism': ['anger', 'fear', 'negative', 'sadness'], 'rhythm': ['positive'],
               'rhythmical': ['joy', 'positive', 'surprise'], 'ribbon': ['anger', 'anticipation', 'joy', 'positive'],
               'richness': ['positive'], 'rickety': ['negative'], 'riddle': ['surprise'], 'riddled': ['negative'],
               'rider': ['positive'], 'ridicule': ['anger', 'disgust', 'negative', 'sadness'],
               'ridiculous': ['anger', 'disgust', 'negative'], 'rife': ['negative'],
               'rifle': ['anger', 'fear', 'negative'], 'rift': ['negative'], 'righteous': ['positive'],
               'rightful': ['positive'], 'rightly': ['positive'], 'rigid': ['negative'], 'rigidity': ['negative'],
               'rigor': ['disgust', 'fear', 'negative'], 'rigorous': ['negative'], 'ringer': ['anger', 'negative'],
               'riot': ['anger', 'fear', 'negative'], 'riotous': ['anger', 'fear', 'negative', 'surprise'],
               'ripe': ['positive'], 'ripen': ['anticipation', 'positive'],
               'rising': ['anticipation', 'joy', 'positive'], 'risk': ['anticipation', 'fear', 'negative'],
               'risky': ['anticipation', 'fear', 'negative'], 'rivalry': ['anger', 'negative'],
               'riveting': ['positive'], 'roadster': ['joy', 'positive', 'trust'],
               'rob': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'robber': ['disgust', 'fear', 'negative'],
               'robbery': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'robust': ['positive'],
               'rock': ['positive'], 'rocket': ['anger'], 'rod': ['fear', 'positive', 'trust'],
               'rogue': ['disgust', 'negative'], 'rollicking': ['joy', 'positive'],
               'romance': ['anticipation', 'fear', 'joy', 'positive', 'sadness', 'surprise', 'trust'],
               'romantic': ['anticipation', 'joy', 'positive', 'trust'], 'romanticism': ['joy', 'positive'],
               'romp': ['joy', 'positive'], 'rook': ['anger', 'disgust', 'negative'], 'rooted': ['positive', 'trust'],
               'rosy': ['positive'], 'rot': ['disgust', 'fear', 'negative', 'sadness'], 'rota': ['positive', 'trust'],
               'rotting': ['disgust', 'negative'], 'roughness': ['negative'], 'roulette': ['anticipation'],
               'rout': ['negative'], 'routine': ['positive', 'trust'], 'row': ['anger', 'negative'],
               'rowdy': ['negative'], 'royalty': ['positive'], 'rubbish': ['disgust', 'negative'],
               'rubble': ['fear', 'negative', 'sadness'], 'rubric': ['positive'], 'rue': ['negative', 'sadness'],
               'ruffle': ['negative'], 'rugged': ['negative'], 'ruin': ['fear', 'negative', 'sadness'],
               'ruined': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'ruinous': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'rule': ['fear', 'trust'],
               'rumor': ['negative', 'sadness'], 'runaway': ['negative', 'sadness'],
               'rupture': ['fear', 'negative', 'sadness', 'surprise'], 'ruse': ['negative'], 'rust': ['negative'],
               'rusty': ['negative'], 'ruth': ['positive'], 'ruthless': ['anger', 'disgust', 'fear', 'negative'],
               'saber': ['anger', 'fear', 'negative'],
               'sabotage': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'sacrifices': ['disgust', 'fear', 'negative', 'sadness'], 'sadly': ['negative', 'sadness'],
               'sadness': ['negative', 'sadness', 'trust'], 'safe': ['joy', 'positive', 'trust'],
               'safeguard': ['positive', 'trust'], 'safekeeping': ['trust'], 'sag': ['fear', 'negative'],
               'sage': ['positive', 'trust'], 'saint': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'saintly': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'salary': ['anticipation', 'joy', 'positive', 'trust'], 'salient': ['positive'],
               'saliva': ['anticipation'], 'sally': ['surprise'], 'saloon': ['anger'],
               'salutary': ['joy', 'positive', 'trust'], 'salute': ['joy', 'positive'],
               'salvation': ['anticipation', 'joy', 'positive', 'trust'], 'salve': ['positive'],
               'samurai': ['fear', 'positive'], 'sanctification': ['joy', 'positive', 'trust'],
               'sanctify': ['anticipation', 'joy', 'positive', 'sadness', 'surprise', 'trust'],
               'sanctuary': ['anticipation', 'joy', 'positive', 'trust'], 'sanguine': ['positive'],
               'sanitary': ['positive'], 'sap': ['negative', 'sadness'], 'sappy': ['trust'],
               'sarcasm': ['anger', 'disgust', 'negative', 'sadness'], 'sarcoma': ['fear', 'negative', 'sadness'],
               'sardonic': ['negative'], 'satanic': ['anger', 'negative'], 'satin': ['positive'],
               'satisfactorily': ['positive'], 'satisfied': ['joy', 'positive'], 'saturated': ['disgust', 'negative'],
               'savage': ['anger', 'fear', 'negative'], 'savagery': ['anger', 'fear', 'negative'],
               'save': ['joy', 'positive', 'trust'], 'savings': ['positive'],
               'savor': ['anticipation', 'disgust', 'joy', 'positive', 'sadness', 'trust'], 'savory': ['positive'],
               'savvy': ['positive'], 'scab': ['negative'], 'scaffold': ['fear', 'negative'],
               'scalpel': ['fear', 'negative'], 'scaly': ['negative'], 'scandal': ['fear', 'negative'],
               'scandalous': ['anger', 'negative'], 'scanty': ['negative'], 'scapegoat': ['anger', 'fear', 'negative'],
               'scar': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'scarce': ['fear', 'negative', 'sadness'],
               'scarcely': ['negative', 'sadness'], 'scarcity': ['anger', 'fear', 'negative', 'sadness'],
               'scare': ['anger', 'anticipation', 'fear', 'negative', 'surprise'],
               'scarecrow': ['fear', 'negative', 'positive'], 'scavenger': ['negative'], 'sceptical': ['trust'],
               'scheme': ['negative'], 'schism': ['anger', 'negative'],
               'schizophrenia': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'scholar': ['positive'],
               'scholarship': ['joy', 'positive'], 'school': ['trust'], 'sciatica': ['negative'],
               'scientific': ['positive', 'trust'], 'scientist': ['anticipation', 'positive', 'trust'],
               'scintilla': ['positive'], 'scoff': ['anger', 'disgust', 'negative'],
               'scold': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'scolding': ['anger', 'negative'],
               'scorching': ['anger', 'negative'], 'score': ['anticipation', 'joy', 'positive', 'surprise'],
               'scorn': ['anger', 'negative'], 'scorpion': ['anger', 'disgust', 'fear', 'negative', 'surprise'],
               'scotch': ['negative'], 'scoundrel': ['anger', 'disgust', 'fear', 'negative', 'trust'],
               'scourge': ['anger', 'fear', 'negative', 'sadness'], 'scrambling': ['negative'],
               'scrapie': ['anger', 'fear', 'negative', 'sadness'],
               'scream': ['anger', 'disgust', 'fear', 'negative', 'surprise'],
               'screaming': ['anger', 'disgust', 'fear', 'negative'], 'screech': ['fear', 'negative', 'surprise'],
               'screwed': ['anger', 'negative'], 'scribe': ['positive'], 'scrimmage': ['negative', 'surprise'],
               'script': ['positive'], 'scripture': ['trust'], 'scrub': ['disgust', 'negative'],
               'scrumptious': ['positive'], 'scrutinize': ['anticipation', 'negative'], 'scrutiny': ['negative'],
               'sculpture': ['positive'], 'scum': ['disgust', 'negative'], 'sea': ['positive'],
               'seal': ['positive', 'trust'], 'seals': ['trust'], 'sear': ['negative'], 'seasoned': ['positive'],
               'secession': ['negative'], 'secluded': ['negative', 'sadness'],
               'seclusion': ['fear', 'negative', 'positive'], 'secondhand': ['negative'],
               'secrecy': ['surprise', 'trust'], 'secret': ['trust'], 'secretariat': ['positive'],
               'secrete': ['disgust'], 'secretion': ['disgust', 'negative'], 'secretive': ['negative'],
               'sectarian': ['anger', 'fear', 'negative'], 'secular': ['anticipation'], 'securities': ['trust'],
               'sedition': ['anger', 'negative', 'sadness'], 'seduce': ['negative'], 'seduction': ['negative'],
               'seductive': ['anticipation'], 'seek': ['anticipation'],
               'segregate': ['anger', 'disgust', 'negative', 'sadness'], 'segregated': ['negative'],
               'seize': ['fear', 'negative'], 'seizure': ['fear'], 'selfish': ['anger', 'disgust', 'negative'],
               'selfishness': ['negative'], 'senate': ['trust'], 'senile': ['fear', 'negative', 'sadness'],
               'seniority': ['positive', 'trust'], 'sensational': ['joy', 'positive'], 'sense': ['positive'],
               'senseless': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'sensibility': ['positive'], 'sensibly': ['positive'],
               'sensual': ['anticipation', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'sensuality': ['anticipation', 'joy', 'positive'], 'sensuous': ['joy', 'positive'],
               'sentence': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'sentimental': ['positive'], 'sentimentality': ['positive'], 'sentinel': ['positive', 'trust'],
               'sentry': ['trust'], 'separatist': ['anger', 'disgust', 'negative'],
               'sepsis': ['fear', 'negative', 'sadness'], 'septic': ['disgust', 'negative'], 'sequel': ['anticipation'],
               'sequestration': ['negative', 'sadness'], 'serene': ['negative', 'trust'],
               'serenity': ['anticipation', 'joy', 'positive', 'trust'], 'serial': ['anticipation'],
               'series': ['trust'], 'seriousness': ['fear', 'sadness'], 'sermon': ['positive', 'trust'],
               'serpent': ['disgust', 'fear', 'negative'], 'serum': ['positive'], 'servant': ['negative', 'trust'],
               'serve': ['negative', 'trust'], 'servile': ['disgust', 'fear', 'negative', 'sadness'],
               'servitude': ['negative'], 'setback': ['negative', 'sadness'], 'settlor': ['fear', 'positive'],
               'sever': ['negative'], 'severance': ['sadness'], 'sewage': ['disgust', 'negative'], 'sewer': ['disgust'],
               'sewerage': ['disgust', 'negative'], 'sex': ['anticipation', 'joy', 'positive', 'trust'],
               'shabby': ['disgust', 'negative'], 'shack': ['disgust', 'negative', 'sadness'],
               'shackle': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'shady': ['fear', 'negative'], 'shaking': ['fear', 'negative'],
               'shaky': ['anger', 'anticipation', 'fear', 'negative'], 'sham': ['anger', 'disgust', 'negative'],
               'shambles': ['negative'], 'shame': ['disgust', 'fear', 'negative', 'sadness'],
               'shameful': ['negative', 'sadness'], 'shameless': ['disgust', 'negative'],
               'shanghai': ['disgust', 'fear', 'negative'], 'shank': ['fear'], 'shape': ['positive'],
               'shapely': ['positive'], 'share': ['anticipation', 'joy', 'positive', 'trust'], 'shark': ['negative'],
               'sharpen': ['anger', 'anticipation'], 'shatter': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'shattered': ['negative', 'sadness'], 'shed': ['negative'],
               'shell': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'shelter': ['positive', 'trust'],
               'shelved': ['negative'], 'shepherd': ['positive', 'trust'], 'sheriff': ['trust'], 'shine': ['positive'],
               'shining': ['anticipation', 'joy', 'positive'], 'ship': ['anticipation'],
               'shipwreck': ['fear', 'negative', 'sadness'], 'shit': ['anger', 'disgust', 'negative'],
               'shiver': ['anger', 'anticipation', 'fear', 'negative', 'sadness'],
               'shock': ['anger', 'fear', 'negative', 'surprise'], 'shockingly': ['surprise'],
               'shoddy': ['anger', 'disgust', 'negative'], 'shoot': ['anger', 'fear', 'negative'], 'shooter': ['fear'],
               'shooting': ['anger', 'fear', 'negative'], 'shopkeeper': ['trust'],
               'shoplifting': ['anger', 'disgust', 'negative'],
               'shopping': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'shortage': ['anger', 'fear', 'negative', 'sadness'], 'shortcoming': ['negative'],
               'shortly': ['anticipation'], 'shot': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'shoulder': ['positive', 'trust'], 'shout': ['anger', 'surprise'], 'shove': ['anger', 'negative'],
               'show': ['trust'], 'showy': ['negative'], 'shrapnel': ['fear'], 'shrewd': ['positive'],
               'shriek': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'shrill': ['anger', 'fear', 'negative', 'surprise'], 'shrink': ['fear', 'negative', 'sadness'],
               'shroud': ['sadness'], 'shrunk': ['negative'], 'shudder': ['fear', 'negative'],
               'shun': ['anger', 'disgust', 'negative', 'sadness'], 'sib': ['trust'],
               'sick': ['disgust', 'negative', 'sadness'],
               'sickening': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'sickly': ['disgust', 'negative', 'sadness'], 'sickness': ['disgust', 'fear', 'negative', 'sadness'],
               'signature': ['trust'], 'signify': ['anticipation'], 'silk': ['positive'], 'silly': ['joy', 'negative'],
               'simmer': ['anger', 'anticipation'], 'simmering': ['anticipation'],
               'simplify': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'sin': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'sincere': ['positive', 'trust'],
               'sincerity': ['positive'], 'sinful': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'sing': ['anticipation', 'joy', 'positive', 'sadness', 'trust'], 'singly': ['positive'],
               'singularly': ['surprise'], 'sinister': ['anger', 'disgust', 'fear', 'negative'],
               'sinner': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'sinning': ['disgust', 'negative'],
               'sir': ['positive', 'trust'], 'siren': ['fear', 'negative'], 'sissy': ['negative'],
               'sisterhood': ['anger', 'positive', 'sadness', 'surprise', 'trust'], 'sizzle': ['anger'],
               'skeptical': ['negative'], 'sketchy': ['negative'], 'skewed': ['anger', 'anticipation', 'negative'],
               'skid': ['anger', 'fear', 'negative', 'sadness'], 'skilled': ['positive'],
               'skillful': ['positive', 'trust'], 'skip': ['negative'], 'skirmish': ['anger', 'negative'],
               'sky': ['positive'], 'slack': ['negative'], 'slag': ['negative'],
               'slam': ['anger', 'fear', 'negative', 'surprise'], 'slander': ['anger', 'disgust', 'negative'],
               'slanderous': ['negative'], 'slap': ['anger', 'negative', 'surprise'], 'slash': ['anger'],
               'slate': ['positive'], 'slaughter': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'slaughterhouse': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'slaughtering': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'slave': ['anger', 'fear', 'negative', 'sadness'],
               'slavery': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'slay': ['anger', 'negative'],
               'slayer': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'sleek': ['positive'],
               'sleet': ['negative'], 'slender': ['positive'], 'slim': ['positive'], 'slime': ['disgust'],
               'slimy': ['disgust', 'negative'], 'slink': ['negative'], 'slip': ['negative', 'surprise'],
               'slop': ['disgust', 'negative'], 'sloppy': ['disgust', 'negative'], 'sloth': ['disgust', 'negative'],
               'slouch': ['negative'], 'slough': ['negative'], 'slowness': ['negative'],
               'sludge': ['disgust', 'negative'], 'slug': ['disgust', 'negative'], 'sluggish': ['negative', 'sadness'],
               'slum': ['disgust', 'negative'], 'slump': ['negative', 'sadness'],
               'slur': ['anger', 'disgust', 'negative', 'sadness'], 'slush': ['disgust', 'negative', 'surprise'],
               'slut': ['anger', 'disgust', 'negative'], 'sly': ['anger', 'disgust', 'fear', 'negative'],
               'smack': ['anger', 'negative'], 'small': ['negative'], 'smash': ['anger', 'fear', 'negative'],
               'smashed': ['negative'], 'smattering': ['negative'], 'smell': ['anger', 'disgust', 'negative'],
               'smelling': ['disgust', 'negative'], 'smile': ['joy', 'positive', 'surprise', 'trust'],
               'smiling': ['joy', 'positive'], 'smirk': ['negative'], 'smite': ['anger', 'fear', 'negative', 'sadness'],
               'smith': ['trust'], 'smitten': ['positive'], 'smoker': ['negative'], 'smoothness': ['positive'],
               'smother': ['anger', 'negative'], 'smudge': ['negative'], 'smug': ['negative'],
               'smuggle': ['fear', 'negative'], 'smuggler': ['anger', 'disgust', 'fear', 'negative'],
               'smuggling': ['negative'], 'smut': ['disgust', 'fear', 'negative'], 'snag': ['negative', 'surprise'],
               'snags': ['negative'], 'snake': ['disgust', 'fear', 'negative'], 'snare': ['fear', 'negative'],
               'snarl': ['anger', 'disgust', 'negative'], 'snarling': ['anger', 'negative'],
               'sneak': ['anger', 'fear', 'negative', 'surprise'],
               'sneaking': ['anticipation', 'fear', 'negative', 'trust'], 'sneer': ['anger', 'disgust', 'negative'],
               'sneeze': ['disgust', 'negative', 'surprise'], 'snicker': ['positive'], 'snide': ['negative'],
               'snob': ['negative'], 'snort': ['sadness'], 'soak': ['negative'], 'sob': ['negative', 'sadness'],
               'sobriety': ['positive', 'trust'], 'sociable': ['positive'], 'socialism': ['disgust', 'fear'],
               'socialist': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'soil': ['disgust', 'negative'],
               'soiled': ['disgust', 'negative'], 'solace': ['positive'], 'soldier': ['anger', 'positive', 'sadness'],
               'solid': ['positive'], 'solidarity': ['trust'], 'solidity': ['positive', 'trust'],
               'solution': ['positive'], 'solvency': ['positive'], 'somatic': ['negative', 'surprise'],
               'somber': ['negative', 'sadness'], 'sonar': ['anticipation', 'positive'], 'sonata': ['positive'],
               'sonnet': ['joy', 'positive', 'sadness'], 'sonorous': ['joy', 'positive'],
               'soot': ['disgust', 'negative'], 'soothe': ['positive'], 'soothing': ['joy', 'positive', 'trust'],
               'sorcery': ['anticipation', 'fear', 'negative', 'surprise'],
               'sordid': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'sore': ['anger', 'negative', 'sadness'],
               'sorely': ['negative', 'sadness'], 'soreness': ['disgust', 'negative', 'sadness'],
               'sorrow': ['fear', 'negative', 'sadness'], 'sorrowful': ['negative', 'sadness'], 'sorter': ['positive'],
               'sortie': ['fear', 'negative'], 'soulless': ['disgust', 'fear', 'negative', 'sadness'],
               'soulmate': ['fear', 'negative'], 'soundness': ['anticipation', 'joy', 'positive', 'trust'],
               'soup': ['positive'], 'sour': ['disgust', 'negative'], 'sovereign': ['trust'],
               'spa': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'spacious': ['positive'],
               'spaniel': ['joy', 'positive', 'trust'], 'spank': ['anger', 'fear', 'negative', 'sadness'],
               'spanking': ['anger'], 'sparkle': ['anticipation', 'joy', 'positive', 'surprise'],
               'spasm': ['fear', 'negative'], 'spat': ['anger', 'negative'],
               'spear': ['anger', 'anticipation', 'fear', 'negative'], 'special': ['joy', 'positive'],
               'specialist': ['trust'], 'specialize': ['trust'], 'specie': ['positive'],
               'speck': ['disgust', 'negative'], 'spectacle': ['negative', 'positive'],
               'spectacular': ['anticipation', 'surprise'], 'specter': ['fear', 'negative', 'sadness'],
               'spectral': ['negative'], 'speculation': ['fear', 'negative', 'sadness'],
               'speculative': ['anticipation'], 'speech': ['positive'], 'speedy': ['positive'],
               'spelling': ['positive'], 'spent': ['negative'], 'spew': ['disgust'], 'spice': ['positive'],
               'spider': ['disgust', 'fear'], 'spike': ['fear'], 'spine': ['anger', 'negative', 'positive'],
               'spinster': ['fear', 'negative', 'sadness'], 'spirit': ['positive'],
               'spirits': ['anticipation', 'joy', 'positive', 'surprise'], 'spit': ['disgust'],
               'spite': ['anger', 'negative'], 'spiteful': ['anger', 'negative'], 'splash': ['surprise'],
               'splendid': ['joy', 'positive', 'surprise'], 'splendor': ['anticipation', 'joy', 'positive', 'surprise'],
               'splinter': ['negative'], 'split': ['negative'], 'splitting': ['negative', 'sadness'],
               'spoil': ['disgust', 'negative'], 'spoiler': ['negative', 'sadness'], 'spoke': ['negative'],
               'spokesman': ['trust'], 'sponge': ['negative'], 'sponsor': ['positive', 'trust'],
               'spook': ['fear', 'negative'], 'spotless': ['positive', 'trust'], 'spouse': ['joy', 'positive', 'trust'],
               'sprain': ['negative', 'sadness'], 'spree': ['negative'], 'sprite': ['fear', 'negative'],
               'spruce': ['positive'], 'spur': ['fear'], 'spurious': ['disgust', 'negative'],
               'squall': ['fear', 'negative', 'sadness'], 'squatter': ['negative'],
               'squeamish': ['disgust', 'fear', 'negative'], 'squelch': ['anger', 'disgust', 'negative'],
               'squirm': ['disgust', 'negative'], 'stab': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'stable': ['positive', 'trust'], 'staccato': ['positive'], 'stagger': ['surprise'],
               'staggering': ['negative'], 'stagnant': ['negative', 'sadness'], 'stain': ['disgust', 'negative'],
               'stainless': ['positive'], 'stale': ['negative'], 'stalemate': ['anger', 'disgust'],
               'stalk': ['fear', 'negative'], 'stall': ['disgust'], 'stallion': ['positive'], 'stalwart': ['positive'],
               'stamina': ['positive', 'trust'], 'standing': ['positive'], 'standoff': ['anger', 'fear', 'negative'],
               'standstill': ['anger', 'negative'], 'star': ['anticipation', 'joy', 'positive', 'trust'],
               'staring': ['negative'], 'stark': ['negative', 'trust'], 'starlight': ['positive'],
               'starry': ['anticipation', 'joy', 'positive'], 'start': ['anticipation'],
               'startle': ['fear', 'negative', 'surprise'], 'startling': ['surprise'],
               'starvation': ['fear', 'negative', 'sadness'], 'starved': ['negative'], 'starving': ['negative'],
               'stately': ['positive'], 'statement': ['positive', 'trust'], 'stationary': ['negative'],
               'statistical': ['trust'], 'statue': ['positive'], 'status': ['positive'], 'staunch': ['positive'],
               'stave': ['negative'], 'steadfast': ['positive', 'trust'], 'steady': ['surprise', 'trust'],
               'steal': ['anger', 'fear', 'negative', 'sadness'], 'stealing': ['disgust', 'fear', 'negative'],
               'stealth': ['surprise'], 'stealthily': ['surprise'],
               'stealthy': ['anticipation', 'fear', 'negative', 'surprise'], 'stellar': ['positive'],
               'stereotype': ['negative'], 'stereotyped': ['negative'], 'sterile': ['negative', 'sadness'],
               'sterility': ['negative'], 'sterling': ['anger', 'anticipation', 'joy', 'negative', 'positive', 'trust'],
               'stern': ['negative'], 'steward': ['positive', 'trust'], 'sticky': ['disgust'], 'stiff': ['negative'],
               'stiffness': ['negative'], 'stifle': ['negative'], 'stifled': ['anger', 'fear', 'negative', 'sadness'],
               'stigma': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'stillborn': ['negative', 'sadness'],
               'stillness': ['fear', 'positive', 'sadness'], 'sting': ['anger', 'fear', 'negative'],
               'stinging': ['negative'], 'stingy': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'stink': ['disgust', 'negative'], 'stinking': ['disgust', 'negative'],
               'stint': ['fear', 'negative', 'sadness'], 'stocks': ['negative'], 'stolen': ['anger', 'negative'],
               'stomach': ['disgust'], 'stone': ['anger', 'negative'], 'stoned': ['negative'],
               'stools': ['disgust', 'negative'], 'stoppage': ['negative'], 'store': ['anticipation', 'positive'],
               'storm': ['anger', 'negative'], 'storming': ['anger'], 'stormy': ['fear', 'negative'],
               'straightforward': ['positive', 'trust'], 'strained': ['anger', 'negative'],
               'straits': ['fear', 'negative'], 'stranded': ['negative'], 'stranger': ['fear', 'negative'],
               'strangle': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'strategic': ['positive'],
               'strategist': ['anticipation', 'positive', 'trust'], 'stray': ['negative'],
               'strength': ['positive', 'trust'], 'strengthen': ['positive'],
               'strengthening': ['joy', 'positive', 'trust'], 'stress': ['negative'], 'stretcher': ['fear', 'sadness'],
               'stricken': ['sadness'], 'strife': ['anger', 'negative'], 'strike': ['anger', 'negative'],
               'striking': ['positive'], 'strikingly': ['positive'], 'strip': ['negative', 'sadness'],
               'stripe': ['negative'], 'stripped': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'],
               'strive': ['anticipation'], 'stroke': ['fear', 'negative', 'sadness'], 'strongly': ['positive'],
               'structural': ['trust'], 'structure': ['positive', 'trust'],
               'struggle': ['anger', 'fear', 'negative', 'sadness'], 'strut': ['negative'], 'stud': ['positive'],
               'study': ['positive'], 'stuffy': ['negative'], 'stumble': ['negative'],
               'stunned': ['fear', 'negative', 'surprise'], 'stunted': ['negative'], 'stupid': ['negative'],
               'stupidity': ['negative'], 'stupor': ['negative'], 'sturdy': ['positive'],
               'sty': ['disgust', 'negative'], 'subdue': ['negative'], 'subito': ['surprise'], 'subject': ['negative'],
               'subjected': ['negative', 'sadness'], 'subjection': ['negative'],
               'subjugation': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'sublimation': ['joy', 'positive'],
               'submit': ['anticipation'], 'subordinate': ['fear', 'negative'], 'subpoena': ['negative'],
               'subscribe': ['anticipation'], 'subsidence': ['negative', 'sadness'],
               'subsidy': ['anger', 'disgust', 'negative'], 'subsist': ['negative'], 'substance': ['positive'],
               'substantiate': ['trust'], 'substantive': ['positive'], 'subtract': ['negative'],
               'subversion': ['anger', 'fear', 'negative'], 'subversive': ['anger', 'negative', 'surprise'],
               'subvert': ['disgust', 'fear', 'negative', 'sadness'],
               'succeed': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'succeeding': ['anticipation', 'joy', 'positive', 'trust'],
               'success': ['anticipation', 'joy', 'positive'],
               'successful': ['anticipation', 'joy', 'positive', 'trust'], 'succinct': ['positive'],
               'succulent': ['negative', 'positive'], 'succumb': ['negative'], 'suck': ['negative'],
               'sucker': ['anger', 'negative'], 'sudden': ['surprise'], 'suddenly': ['surprise'],
               'sue': ['anger', 'negative', 'sadness'], 'suffer': ['negative'],
               'sufferer': ['fear', 'negative', 'sadness'], 'suffering': ['disgust', 'fear', 'negative', 'sadness'],
               'sufficiency': ['positive'], 'suffocating': ['disgust', 'fear', 'negative', 'sadness'],
               'suffocation': ['anger', 'fear', 'negative'], 'sugar': ['positive'], 'suggest': ['trust'],
               'suicidal': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'suicide': ['anger', 'fear', 'negative', 'sadness'], 'suitable': ['positive'],
               'sullen': ['anger', 'negative', 'sadness'], 'sultan': ['fear'], 'sultry': ['positive'],
               'summons': ['negative'], 'sump': ['disgust'],
               'sun': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'sundial': ['anticipation', 'trust'],
               'sunk': ['disgust', 'fear', 'negative', 'sadness'],
               'sunny': ['anticipation', 'joy', 'positive', 'surprise'], 'sunset': ['anticipation', 'positive'],
               'sunshine': ['joy', 'positive'], 'superb': ['positive'], 'superficial': ['negative'],
               'superfluous': ['negative'], 'superhuman': ['positive'], 'superior': ['positive'],
               'superiority': ['positive'], 'superman': ['joy', 'positive', 'trust'],
               'superstar': ['joy', 'positive', 'trust'], 'superstition': ['fear', 'negative', 'positive'],
               'superstitious': ['anticipation', 'fear', 'negative'], 'supplication': ['positive', 'trust'],
               'supplies': ['positive'], 'supply': ['positive'], 'supported': ['positive'],
               'supporter': ['joy', 'positive', 'trust'], 'supporting': ['positive', 'trust'],
               'suppress': ['anger', 'fear', 'negative', 'sadness'],
               'suppression': ['anger', 'disgust', 'fear', 'negative'],
               'supremacy': ['anger', 'anticipation', 'fear', 'joy', 'negative', 'positive', 'surprise', 'trust'],
               'supreme': ['positive'], 'supremely': ['positive'], 'surcharge': ['anger', 'negative'],
               'surety': ['positive', 'trust'], 'surge': ['surprise'], 'surgery': ['fear', 'sadness'],
               'surly': ['anger', 'disgust', 'negative'], 'surmise': ['positive'], 'surpassing': ['positive'],
               'surprise': ['fear', 'joy', 'positive', 'surprise'], 'surprised': ['surprise'],
               'surprising': ['surprise'], 'surprisingly': ['anticipation', 'surprise'],
               'surrender': ['fear', 'negative', 'sadness'], 'surrendering': ['negative', 'sadness'],
               'surrogate': ['trust'], 'surround': ['anticipation', 'negative', 'positive'], 'surveillance': ['fear'],
               'surveying': ['positive'], 'survive': ['positive'], 'susceptible': ['negative'],
               'suspect': ['fear', 'negative'], 'suspense': ['anticipation', 'fear', 'surprise'],
               'suspension': ['fear', 'negative'], 'suspicion': ['fear', 'negative'],
               'suspicious': ['anger', 'anticipation', 'negative'], 'swab': ['negative'],
               'swamp': ['disgust', 'fear', 'negative'], 'swampy': ['disgust', 'fear', 'negative'],
               'swarm': ['disgust'], 'swastika': ['anger', 'fear', 'negative'], 'swear': ['positive', 'trust'],
               'sweat': ['fear'], 'sweet': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'sweetheart': ['anticipation', 'joy', 'positive', 'sadness', 'trust'], 'sweetie': ['positive'],
               'sweetness': ['positive'], 'sweets': ['anticipation', 'joy', 'positive'],
               'swelling': ['fear', 'negative'], 'swerve': ['fear', 'surprise'], 'swift': ['positive'],
               'swig': ['disgust', 'negative'], 'swim': ['anticipation', 'fear', 'joy', 'positive'],
               'swine': ['disgust', 'negative'], 'swollen': ['negative'], 'symbolic': ['positive'],
               'symmetrical': ['positive'], 'symmetry': ['joy', 'positive', 'trust'],
               'sympathetic': ['fear', 'joy', 'positive', 'sadness', 'trust'], 'sympathize': ['sadness'],
               'sympathy': ['positive', 'sadness'], 'symphony': ['anticipation', 'joy', 'positive'],
               'symptom': ['negative'], 'synchronize': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'syncope': ['fear', 'negative', 'sadness', 'surprise'], 'synergistic': ['positive', 'trust'],
               'synod': ['positive', 'trust'], 'synonymous': ['fear', 'negative', 'positive', 'trust'],
               'syringe': ['fear'], 'system': ['trust'], 'taboo': ['disgust', 'fear', 'negative'],
               'tabulate': ['anticipation'], 'tackle': ['anger', 'surprise'], 'tact': ['positive'],
               'tactics': ['fear', 'trust'], 'taint': ['negative', 'sadness'], 'tale': ['positive'],
               'talent': ['positive'], 'talisman': ['positive'], 'talk': ['positive'],
               'talons': ['anger', 'fear', 'negative'], 'tandem': ['trust'], 'tangled': ['negative'],
               'tanned': ['positive'], 'tantalizing': ['anticipation', 'joy', 'negative', 'positive', 'surprise'],
               'tantamount': ['trust'], 'tardiness': ['negative'], 'tardy': ['negative'],
               'tariff': ['anger', 'disgust', 'negative'], 'tarnish': ['disgust', 'negative', 'sadness'],
               'tarry': ['negative'], 'task': ['positive'], 'tasteful': ['positive'],
               'tasteless': ['disgust', 'negative'], 'tasty': ['positive'], 'taught': ['trust'],
               'taunt': ['anger', 'fear', 'negative', 'sadness'], 'tawny': ['disgust'], 'tax': ['negative', 'sadness'],
               'teach': ['joy', 'positive', 'surprise', 'trust'], 'teacher': ['positive', 'trust'], 'team': ['trust'],
               'tearful': ['disgust', 'fear', 'sadness'], 'tease': ['anger', 'anticipation', 'negative', 'sadness'],
               'teasing': ['anger', 'fear', 'negative'], 'technology': ['positive'], 'tedious': ['negative'],
               'tedium': ['negative'], 'teeming': ['disgust'], 'teens': ['negative', 'positive'],
               'temperance': ['positive'], 'temperate': ['trust'], 'tempered': ['positive'],
               'tempest': ['anger', 'anticipation', 'fear', 'negative', 'sadness', 'surprise'],
               'temptation': ['negative'], 'tenable': ['positive'], 'tenacious': ['positive'], 'tenacity': ['positive'],
               'tenancy': ['positive'], 'tenant': ['positive'], 'tender': ['joy', 'positive', 'trust'],
               'tenderness': ['joy', 'positive'], 'tenement': ['negative'], 'tension': ['anger'],
               'terminal': ['fear', 'negative', 'sadness'], 'terminate': ['sadness'],
               'termination': ['negative', 'sadness'], 'termite': ['disgust', 'negative'],
               'terrible': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'terribly': ['sadness'],
               'terrific': ['sadness'], 'terror': ['fear', 'negative'],
               'terrorism': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'terrorist': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'terrorize': ['anger', 'fear', 'negative', 'sadness'], 'testament': ['anticipation', 'trust'],
               'testimony': ['trust'], 'tetanus': ['disgust', 'negative'], 'tether': ['negative'],
               'thankful': ['joy', 'positive'], 'thanksgiving': ['joy', 'positive'],
               'theft': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'theism': ['disgust', 'negative'],
               'theocratic': ['anger', 'fear', 'negative', 'sadness', 'trust'], 'theological': ['trust'],
               'theology': ['anticipation'], 'theorem': ['trust'], 'theoretical': ['positive'],
               'theory': ['anticipation', 'trust'], 'therapeutic': ['joy', 'positive', 'trust'],
               'therapeutics': ['positive'], 'thermocouple': ['anticipation'], 'thermometer': ['trust'],
               'thief': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'thinker': ['positive'],
               'thirst': ['anticipation', 'sadness', 'surprise'], 'thirsty': ['negative'], 'thirteenth': ['fear'],
               'thorn': ['negative'], 'thorny': ['fear', 'negative'], 'thoroughbred': ['positive'],
               'thought': ['anticipation'], 'thoughtful': ['positive', 'trust'], 'thoughtfulness': ['positive'],
               'thoughtless': ['anger', 'disgust', 'negative'],
               'thrash': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'threat': ['anger', 'fear', 'negative'],
               'threaten': ['anger', 'anticipation', 'fear', 'negative'],
               'threatening': ['anger', 'disgust', 'fear', 'negative'],
               'thresh': ['anger', 'fear', 'negative', 'sadness'], 'thrift': ['disgust', 'positive', 'trust'],
               'thrill': ['anticipation', 'fear', 'joy', 'positive', 'surprise'],
               'thrilling': ['anticipation', 'joy', 'positive', 'surprise'],
               'thriving': ['anticipation', 'joy', 'positive'], 'throb': ['fear', 'negative', 'sadness'],
               'throne': ['positive', 'trust'], 'throttle': ['anger', 'negative'],
               'thug': ['anger', 'disgust', 'fear', 'negative'], 'thump': ['anger', 'negative'], 'thumping': ['fear'],
               'thundering': ['anger', 'fear', 'negative'], 'thwart': ['negative', 'surprise'],
               'tickle': ['anticipation', 'joy', 'positive', 'surprise', 'trust'], 'tiff': ['anger', 'negative'],
               'tighten': ['anger'], 'tiling': ['positive'], 'time': ['anticipation'], 'timely': ['positive'],
               'timid': ['fear', 'negative', 'sadness'], 'timidity': ['anticipation', 'fear', 'negative'],
               'tinsel': ['joy', 'positive'], 'tipsy': ['negative'], 'tirade': ['anger', 'disgust', 'negative'],
               'tired': ['negative'], 'tiredness': ['negative'], 'tiresome': ['negative'], 'tit': ['negative'],
               'title': ['positive', 'trust'], 'toad': ['disgust', 'negative'], 'toast': ['joy', 'positive'],
               'tobacco': ['negative'], 'toilet': ['disgust', 'negative'], 'toils': ['negative'],
               'tolerant': ['positive'], 'tolerate': ['anger', 'negative', 'sadness'], 'toleration': ['positive'],
               'tomb': ['sadness'], 'tomorrow': ['anticipation'], 'toothache': ['fear', 'negative'],
               'top': ['anticipation', 'positive', 'trust'], 'topple': ['surprise'],
               'torment': ['anger', 'fear', 'negative', 'sadness'], 'torn': ['negative'], 'tornado': ['fear'],
               'torpedo': ['anger', 'negative'], 'torrent': ['fear'], 'torrid': ['negative'], 'tort': ['negative'],
               'tortious': ['anger', 'disgust', 'negative'],
               'torture': ['anger', 'anticipation', 'disgust', 'fear', 'negative', 'sadness'], 'touched': ['negative'],
               'touchy': ['anger', 'negative', 'sadness'], 'tough': ['negative', 'sadness'],
               'toughness': ['anger', 'fear', 'positive', 'trust'], 'tower': ['positive'],
               'towering': ['anticipation', 'fear', 'positive'], 'toxic': ['disgust', 'negative'],
               'toxin': ['fear', 'negative'], 'track': ['anticipation'], 'tract': ['fear'], 'trade': ['trust'],
               'traditional': ['positive'], 'tragedy': ['fear', 'negative', 'sadness'], 'tragic': ['negative'],
               'trainer': ['trust'], 'traitor': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'tramp': ['disgust', 'fear', 'negative', 'sadness'], 'trance': ['negative'],
               'tranquil': ['joy', 'positive'], 'tranquility': ['joy', 'positive', 'trust'], 'transaction': ['trust'],
               'transcendence': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'transcendental': ['positive'], 'transcript': ['trust'], 'transgression': ['negative'],
               'transitional': ['anticipation'], 'translation': ['trust'], 'trappings': ['negative'],
               'traps': ['negative'], 'trash': ['disgust', 'negative', 'sadness'], 'trashy': ['disgust', 'negative'],
               'traumatic': ['anger', 'fear', 'negative', 'sadness'], 'travail': ['negative'],
               'traveling': ['positive'], 'travesty': ['disgust', 'fear', 'negative', 'sadness'],
               'treacherous': ['anger', 'disgust', 'fear', 'negative'],
               'treachery': ['anger', 'fear', 'negative', 'sadness', 'surprise'], 'treadmill': ['anticipation'],
               'treason': ['anger', 'disgust', 'fear', 'negative', 'surprise'],
               'treasure': ['anticipation', 'joy', 'positive', 'trust'], 'treasurer': ['trust'],
               'treat': ['anger', 'anticipation', 'disgust', 'fear', 'joy', 'negative', 'positive', 'sadness',
                         'surprise', 'trust'],
               'tree': ['anger', 'anticipation', 'disgust', 'joy', 'positive', 'surprise', 'trust'],
               'trembling': ['fear', 'negative'], 'tremendously': ['positive'],
               'tremor': ['anger', 'anticipation', 'fear', 'negative', 'sadness'], 'trend': ['positive'],
               'trendy': ['positive'], 'trepidation': ['anticipation', 'fear', 'negative', 'surprise'],
               'trespass': ['anger', 'negative'], 'tribe': ['trust'], 'tribulation': ['fear', 'negative', 'sadness'],
               'tribunal': ['anticipation', 'disgust', 'fear', 'negative', 'trust'], 'tribune': ['trust'],
               'tributary': ['anticipation', 'positive'], 'tribute': ['positive'], 'trick': ['negative', 'surprise'],
               'trickery': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'], 'trifle': ['negative'],
               'trig': ['positive'], 'trip': ['surprise'], 'tripping': ['anger', 'negative', 'sadness'],
               'triumph': ['anticipation', 'joy', 'positive'],
               'triumphant': ['anticipation', 'joy', 'positive', 'trust'], 'troll': ['anger', 'fear', 'negative'],
               'trophy': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'troublesome': ['anger', 'fear', 'negative'], 'truce': ['joy', 'positive', 'trust'], 'truck': ['trust'],
               'true': ['joy', 'positive', 'trust'], 'trump': ['surprise'], 'trumpet': ['negative'], 'truss': ['trust'],
               'trust': ['trust'], 'trustee': ['trust'], 'trusty': ['positive'], 'truth': ['positive', 'trust'],
               'truthful': ['trust'], 'truthfulness': ['positive', 'trust'], 'tumble': ['negative'],
               'tumor': ['fear', 'negative'], 'tumour': ['fear', 'negative', 'sadness'],
               'tumult': ['anger', 'fear', 'negative', 'surprise'], 'tumultuous': ['anger', 'fear', 'negative'],
               'turbulence': ['anger', 'fear', 'negative'], 'turbulent': ['fear', 'negative'],
               'turmoil': ['anger', 'fear', 'negative', 'sadness'], 'tussle': ['anger'],
               'tutelage': ['positive', 'trust'], 'tutor': ['positive'], 'twin': ['positive'],
               'twinkle': ['anticipation', 'joy', 'positive'], 'twitch': ['negative'], 'typhoon': ['fear', 'negative'],
               'tyrannical': ['anger', 'disgust', 'fear', 'negative'], 'tyranny': ['fear', 'negative', 'sadness'],
               'tyrant': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'ugliness': ['disgust', 'fear', 'negative', 'sadness'], 'ugly': ['disgust', 'negative'],
               'ulcer': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'ulterior': ['negative'],
               'ultimate': ['anticipation', 'sadness'], 'ultimately': ['anticipation', 'positive'],
               'ultimatum': ['anger', 'fear', 'negative'], 'umpire': ['positive', 'trust'],
               'unable': ['negative', 'sadness'], 'unacceptable': ['negative', 'sadness'],
               'unaccountable': ['anticipation', 'disgust', 'negative', 'sadness', 'trust'],
               'unacknowledged': ['sadness'], 'unanimity': ['positive'], 'unanimous': ['positive'],
               'unanticipated': ['surprise'], 'unapproved': ['negative'], 'unassuming': ['positive'],
               'unattached': ['negative'], 'unattainable': ['anger', 'negative', 'sadness'],
               'unattractive': ['disgust', 'negative', 'sadness'], 'unauthorized': ['negative'],
               'unavoidable': ['negative'], 'unaware': ['negative'], 'unbearable': ['disgust', 'negative', 'sadness'],
               'unbeaten': ['anticipation', 'joy', 'negative', 'positive', 'sadness', 'surprise'],
               'unbelief': ['negative'], 'unbelievable': ['negative'], 'unbiased': ['positive'], 'unborn': ['negative'],
               'unbreakable': ['positive'],
               'unbridled': ['anger', 'anticipation', 'fear', 'negative', 'positive', 'surprise'],
               'unbroken': ['positive', 'trust'], 'uncanny': ['fear', 'negative', 'surprise'],
               'uncaring': ['anger', 'disgust', 'negative', 'sadness'],
               'uncertain': ['anger', 'disgust', 'fear', 'negative', 'surprise'], 'unchangeable': ['negative'],
               'unclean': ['disgust', 'negative'], 'uncomfortable': ['negative'],
               'unconscionable': ['disgust', 'negative'], 'unconscious': ['negative'], 'unconstitutional': ['negative'],
               'unconstrained': ['joy', 'positive'],
               'uncontrollable': ['anger', 'anticipation', 'negative', 'surprise'], 'uncontrolled': ['negative'],
               'uncover': ['surprise'], 'undecided': ['anticipation', 'fear', 'negative'],
               'underestimate': ['surprise'], 'underline': ['positive'], 'undermined': ['negative'],
               'underpaid': ['anger', 'negative', 'sadness'], 'undersized': ['negative'],
               'understanding': ['positive', 'trust'], 'undertaker': ['sadness'], 'undertaking': ['anticipation'],
               'underwrite': ['positive', 'trust'], 'undesirable': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'undesired': ['negative', 'sadness'], 'undisclosed': ['anticipation'], 'undiscovered': ['surprise'],
               'undivided': ['positive'], 'undo': ['negative'], 'undoubted': ['anticipation', 'disgust'],
               'undying': ['anticipation', 'joy', 'positive', 'sadness', 'trust'],
               'uneasiness': ['anticipation', 'negative', 'sadness'], 'uneasy': ['disgust', 'fear', 'negative'],
               'uneducated': ['negative', 'sadness'], 'unemployed': ['fear', 'negative', 'sadness'],
               'unequal': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'unequivocal': ['trust'],
               'unequivocally': ['positive'], 'uneven': ['negative'],
               'unexpected': ['anticipation', 'fear', 'joy', 'negative', 'positive', 'surprise'],
               'unexpectedly': ['surprise'], 'unexplained': ['anticipation', 'negative', 'sadness'],
               'unfair': ['anger', 'disgust', 'negative', 'sadness'], 'unfairness': ['anger', 'negative', 'sadness'],
               'unfaithful': ['disgust', 'negative'], 'unfavorable': ['disgust', 'negative', 'sadness'],
               'unfinished': ['negative'], 'unfold': ['anticipation', 'positive'], 'unforeseen': ['surprise'],
               'unforgiving': ['anger', 'negative', 'sadness'], 'unfortunate': ['negative', 'sadness'],
               'unfriendly': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'unfulfilled': ['anger', 'anticipation', 'negative', 'sadness', 'surprise'], 'unfurnished': ['negative'],
               'ungodly': ['negative', 'sadness'], 'ungrateful': ['anger', 'disgust', 'negative'],
               'unguarded': ['surprise'], 'unhappiness': ['negative', 'sadness'],
               'unhappy': ['anger', 'disgust', 'negative', 'sadness'],
               'unhealthy': ['disgust', 'fear', 'negative', 'sadness'], 'unholy': ['fear', 'negative'],
               'unification': ['anticipation', 'joy', 'positive', 'trust'], 'uniformly': ['positive'],
               'unimaginable': ['negative', 'positive', 'surprise'], 'unimportant': ['negative', 'sadness'],
               'unimpressed': ['negative'], 'unimproved': ['negative'], 'uninfected': ['positive'],
               'uninformed': ['negative'], 'uninitiated': ['negative'], 'uninspired': ['negative', 'sadness'],
               'unintelligible': ['negative'], 'unintended': ['surprise'], 'unintentional': ['surprise'],
               'unintentionally': ['negative', 'surprise'], 'uninterested': ['negative', 'sadness'],
               'uninteresting': ['negative', 'sadness'], 'uninvited': ['sadness'], 'unique': ['positive', 'surprise'],
               'unison': ['positive'], 'unitary': ['positive'], 'united': ['positive', 'trust'],
               'unity': ['positive', 'trust'], 'university': ['anticipation', 'positive'],
               'unjust': ['anger', 'negative'], 'unjustifiable': ['anger', 'disgust', 'fear', 'negative'],
               'unjustified': ['negative'], 'unkind': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'unknown': ['anticipation', 'fear', 'negative'],
               'unlawful': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'unlicensed': ['negative'],
               'unlimited': ['positive'], 'unlucky': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'unmanageable': ['disgust', 'negative'], 'unnatural': ['disgust', 'fear', 'negative'],
               'unofficial': ['negative'], 'unpaid': ['anger', 'negative', 'sadness'],
               'unpleasant': ['disgust', 'negative', 'sadness'], 'unpopular': ['disgust', 'negative', 'sadness'],
               'unprecedented': ['surprise'], 'unpredictable': ['negative', 'surprise'], 'unprepared': ['negative'],
               'unpretentious': ['positive'], 'unproductive': ['negative'], 'unprofitable': ['negative'],
               'unprotected': ['negative'], 'unpublished': ['anticipation', 'negative', 'sadness'],
               'unquestionable': ['positive', 'trust'], 'unquestionably': ['positive', 'trust'],
               'unquestioned': ['positive', 'trust'], 'unreliable': ['negative', 'trust'],
               'unrequited': ['negative', 'sadness'], 'unresolved': ['anticipation'], 'unrest': ['fear', 'sadness'],
               'unruly': ['anger', 'disgust', 'fear', 'negative'], 'unsafe': ['fear', 'negative'],
               'unsatisfactory': ['disgust', 'negative'], 'unsatisfied': ['disgust', 'negative', 'sadness'],
               'unsavory': ['negative'], 'unscathed': ['positive'], 'unscrupulous': ['negative'], 'unseat': ['sadness'],
               'unselfish': ['positive'], 'unsettled': ['anger', 'disgust', 'fear', 'negative'],
               'unsightly': ['disgust', 'negative'], 'unsophisticated': ['negative'],
               'unspeakable': ['fear', 'negative'], 'unstable': ['fear', 'negative', 'surprise'], 'unsteady': ['fear'],
               'unsuccessful': ['negative', 'sadness'], 'unsuitable': ['negative'], 'unsung': ['negative'],
               'unsupported': ['negative'], 'unsurpassed': ['anticipation', 'fear', 'joy', 'positive', 'trust'],
               'unsuspecting': ['surprise'], 'unsustainable': ['negative'], 'unsympathetic': ['anger', 'negative'],
               'untamed': ['negative'], 'untenable': ['negative'],
               'unthinkable': ['anger', 'disgust', 'fear', 'negative'], 'untidy': ['disgust', 'negative'],
               'untie': ['joy', 'negative', 'positive'], 'untimely': ['negative'], 'untitled': ['negative', 'sadness'],
               'untold': ['anticipation', 'negative'], 'untoward': ['anger', 'disgust', 'negative'],
               'untrained': ['negative'], 'untrue': ['negative'], 'untrustworthy': ['anger', 'negative'],
               'unverified': ['anticipation'], 'unwarranted': ['negative'], 'unwashed': ['disgust', 'negative'],
               'unwavering': ['positive', 'trust'], 'unwelcome': ['negative', 'sadness'],
               'unwell': ['negative', 'sadness'], 'unwillingness': ['negative'], 'unwise': ['negative'],
               'unwitting': ['negative'], 'unworthy': ['disgust', 'negative'], 'unyielding': ['negative'],
               'upheaval': ['anger', 'fear', 'negative', 'sadness'],
               'uphill': ['anticipation', 'fear', 'negative', 'positive'],
               'uplift': ['anticipation', 'joy', 'positive', 'trust'], 'upright': ['positive', 'trust'],
               'uprising': ['anger', 'anticipation', 'fear', 'negative'], 'uproar': ['negative'],
               'upset': ['anger', 'negative', 'sadness'], 'urchin': ['negative'],
               'urgency': ['anticipation', 'fear', 'surprise'],
               'urgent': ['anticipation', 'fear', 'negative', 'surprise'], 'urn': ['sadness'],
               'usefulness': ['positive'], 'useless': ['negative'], 'usher': ['positive', 'trust'],
               'usual': ['positive', 'trust'], 'usurp': ['anger', 'negative'], 'usurped': ['anger', 'fear', 'negative'],
               'usury': ['negative'], 'utility': ['positive'], 'utopian': ['anticipation', 'joy', 'positive', 'trust'],
               'vacancy': ['negative'], 'vacation': ['anticipation', 'joy', 'positive'], 'vaccine': ['positive'],
               'vacuous': ['disgust', 'negative'], 'vague': ['negative'], 'vagueness': ['negative'],
               'vainly': ['disgust', 'negative', 'sadness'], 'valiant': ['positive'], 'validity': ['fear'],
               'valor': ['positive', 'trust'], 'valuable': ['positive'],
               'vampire': ['anger', 'disgust', 'fear', 'negative'], 'vanguard': ['positive'], 'vanish': ['surprise'],
               'vanished': ['fear', 'negative', 'sadness', 'surprise'], 'vanity': ['negative'],
               'vanquish': ['positive'], 'variable': ['surprise'],
               'varicella': ['disgust', 'fear', 'negative', 'sadness'], 'varicose': ['negative'],
               'veal': ['sadness', 'trust'], 'veer': ['fear', 'surprise'],
               'vegetative': ['disgust', 'negative', 'sadness'], 'vehement': ['anger', 'fear', 'negative'],
               'velvet': ['positive'], 'velvety': ['positive'], 'vendetta': ['anger', 'fear', 'negative', 'sadness'],
               'venerable': ['anticipation', 'joy', 'positive', 'trust'], 'veneration': ['positive'],
               'vengeance': ['anger', 'negative'], 'vengeful': ['anger', 'fear', 'negative'],
               'venom': ['anger', 'disgust', 'fear', 'negative'], 'venomous': ['anger', 'disgust', 'fear', 'negative'],
               'vent': ['anger'], 'veracity': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'verbosity': ['negative'], 'verdant': ['positive'], 'verdict': ['fear'],
               'verge': ['anticipation', 'fear', 'negative'], 'verification': ['positive', 'trust'],
               'verified': ['positive', 'trust'], 'verily': ['positive', 'trust'], 'veritable': ['positive'],
               'vermin': ['anger', 'disgust', 'fear', 'negative'], 'vernal': ['joy', 'positive'],
               'versus': ['anger', 'negative'], 'vertigo': ['fear', 'negative'], 'verve': ['positive'],
               'vesicular': ['disgust'], 'veteran': ['positive', 'trust'], 'veto': ['anger', 'negative'],
               'vicar': ['positive', 'trust'], 'vice': ['negative'], 'vicious': ['anger', 'disgust', 'negative'],
               'victim': ['anger', 'fear', 'negative', 'sadness'],
               'victimized': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'victor': ['joy', 'positive'], 'victorious': ['joy', 'positive'],
               'victory': ['anticipation', 'joy', 'positive', 'trust'], 'vigil': ['anticipation'],
               'vigilance': ['anticipation', 'positive', 'trust'], 'vigilant': ['fear', 'positive', 'trust'],
               'vigor': ['positive'], 'vigorous': ['positive', 'trust'], 'villager': ['positive', 'trust'],
               'villain': ['fear', 'negative'], 'villainous': ['anger', 'disgust', 'fear', 'negative'],
               'vindicate': ['anger'], 'vindicated': ['positive'],
               'vindication': ['anticipation', 'joy', 'positive', 'trust'],
               'vindictive': ['anger', 'disgust', 'negative'],
               'violation': ['anger', 'fear', 'negative', 'sadness', 'surprise'],
               'violence': ['anger', 'fear', 'negative', 'sadness'],
               'violent': ['anger', 'disgust', 'fear', 'negative', 'surprise'],
               'violently': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'viper': ['fear', 'negative'],
               'virgin': ['positive', 'trust'], 'virginity': ['anticipation', 'positive'],
               'virtue': ['positive', 'trust'], 'virtuous': ['joy', 'positive', 'trust'],
               'virulence': ['anger', 'fear', 'negative'], 'virus': ['negative'],
               'vision': ['anticipation', 'positive'], 'visionary': ['anticipation', 'joy', 'positive', 'trust'],
               'visit': ['positive'], 'visitation': ['negative'], 'visitor': ['anticipation', 'joy', 'positive'],
               'visor': ['anticipation', 'surprise'], 'vital': ['positive'], 'vitality': ['joy', 'positive', 'trust'],
               'vivacious': ['joy', 'positive'], 'vivid': ['joy', 'positive'], 'vixen': ['negative'],
               'vocabulary': ['positive'], 'volatility': ['anger', 'anticipation', 'fear', 'negative', 'surprise'],
               'volcano': ['fear', 'negative', 'surprise'],
               'volunteer': ['anticipation', 'fear', 'joy', 'positive', 'trust'], 'volunteers': ['trust'],
               'voluptuous': ['anticipation', 'joy', 'positive'], 'vomit': ['disgust'], 'vomiting': ['negative'],
               'voodoo': ['negative'],
               'vote': ['anger', 'anticipation', 'joy', 'negative', 'positive', 'sadness', 'surprise', 'trust'],
               'votive': ['trust'], 'vouch': ['positive', 'trust'], 'voucher': ['trust'],
               'vow': ['anticipation', 'joy', 'positive', 'trust'], 'voyage': ['anticipation'],
               'vulgar': ['disgust', 'negative'], 'vulgarity': ['anger', 'disgust', 'negative', 'sadness'],
               'vulnerability': ['fear', 'negative', 'sadness'], 'vulture': ['disgust', 'fear', 'negative'],
               'waffle': ['anger', 'negative', 'sadness'], 'wages': ['joy', 'positive'],
               'wail': ['fear', 'negative', 'sadness'], 'wait': ['anticipation', 'negative'],
               'wallow': ['disgust', 'negative', 'sadness'], 'wan': ['fear', 'negative', 'sadness'],
               'wane': ['negative', 'sadness'], 'wanting': ['negative', 'sadness'], 'war': ['fear', 'negative'],
               'warden': ['anger', 'fear', 'negative', 'trust'], 'ware': ['fear', 'negative'],
               'warfare': ['anger', 'fear', 'negative', 'sadness'], 'warlike': ['anger', 'fear', 'negative'],
               'warlock': ['fear'], 'warn': ['anticipation', 'fear', 'negative', 'surprise', 'trust'],
               'warned': ['anticipation', 'fear', 'surprise'], 'warning': ['fear'],
               'warp': ['anger', 'negative', 'sadness'], 'warped': ['negative'], 'warranty': ['positive', 'trust'],
               'warrior': ['anger', 'fear', 'positive'], 'wart': ['disgust', 'negative'], 'wary': ['fear'],
               'waste': ['disgust', 'negative'], 'wasted': ['anger', 'disgust', 'negative'],
               'wasteful': ['anger', 'disgust', 'negative', 'sadness'],
               'wasting': ['disgust', 'fear', 'negative', 'sadness'], 'watch': ['anticipation', 'fear'],
               'watchdog': ['positive', 'trust'], 'watchful': ['positive', 'trust'], 'watchman': ['positive', 'trust'],
               'waterproof': ['positive'], 'watery': ['negative'], 'waver': ['fear', 'negative'],
               'weakened': ['negative'], 'weakly': ['fear', 'negative', 'sadness'], 'weakness': ['negative'],
               'wealth': ['joy', 'positive', 'trust'], 'wear': ['negative', 'trust'],
               'wearily': ['negative', 'sadness'], 'weariness': ['negative', 'sadness'],
               'weary': ['negative', 'sadness'], 'weatherproof': ['positive'], 'weeds': ['negative', 'sadness'],
               'weep': ['negative', 'sadness'], 'weeping': ['sadness'], 'weigh': ['anticipation', 'trust'],
               'weight': ['anticipation', 'disgust', 'fear', 'joy', 'negative', 'positive', 'sadness', 'surprise',
                          'trust'], 'weighty': ['fear'], 'weird': ['disgust', 'negative'],
               'weirdo': ['fear', 'negative'], 'welcomed': ['joy', 'positive'], 'wen': ['negative'],
               'wench': ['anger', 'disgust', 'negative'], 'whack': ['negative'],
               'whim': ['anticipation', 'joy', 'negative', 'surprise'], 'whimper': ['fear', 'sadness'],
               'whimsical': ['joy'], 'whine': ['disgust', 'negative', 'sadness'], 'whip': ['anger', 'negative'],
               'whirlpool': ['fear'], 'whirlwind': ['fear', 'negative'], 'whisky': ['negative'],
               'white': ['anticipation', 'joy', 'positive', 'trust'], 'whiteness': ['joy', 'positive'],
               'wholesome': ['positive', 'trust'], 'whore': ['disgust', 'negative'], 'wicked': ['fear', 'negative'],
               'wickedness': ['disgust', 'negative'], 'wicket': ['positive'], 'widespread': ['positive'],
               'widow': ['sadness'], 'widower': ['sadness'], 'wild': ['negative', 'surprise'], 'wildcat': ['negative'],
               'wilderness': ['anticipation', 'fear', 'sadness'],
               'wildfire': ['fear', 'negative', 'sadness', 'surprise'], 'willful': ['anger', 'negative', 'sadness'],
               'willingly': ['positive'], 'willingness': ['positive'], 'wimp': ['disgust', 'fear', 'negative'],
               'wimpy': ['anger', 'disgust', 'fear', 'negative', 'sadness'],
               'wince': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'windfall': ['positive'],
               'winner': ['anticipation', 'joy', 'positive', 'surprise'],
               'winning': ['anticipation', 'disgust', 'joy', 'positive', 'sadness', 'surprise', 'trust'],
               'winnings': ['anticipation', 'joy', 'positive'],
               'wireless': ['anger', 'anticipation', 'positive', 'surprise'], 'wis': ['positive'],
               'wisdom': ['positive', 'trust'], 'wise': ['positive'], 'wishful': ['anticipation'], 'wit': ['positive'],
               'witch': ['anger', 'disgust', 'fear', 'negative'],
               'witchcraft': ['anger', 'fear', 'negative', 'sadness'], 'withdraw': ['negative', 'sadness'],
               'wither': ['negative', 'sadness'], 'withered': ['disgust', 'negative'],
               'withstand': ['anticipation', 'fear', 'positive'], 'witness': ['trust'], 'wits': ['positive'],
               'witty': ['joy', 'positive'], 'wizard': ['anticipation', 'positive', 'surprise'],
               'woe': ['disgust', 'fear', 'negative', 'sadness'], 'woeful': ['negative', 'sadness'],
               'woefully': ['disgust', 'negative', 'sadness'], 'womb': ['positive'],
               'wonderful': ['joy', 'positive', 'surprise', 'trust'], 'wonderfully': ['joy', 'positive', 'surprise'],
               'wondrous': ['positive'], 'wont': ['anticipation'], 'wop': ['anger'], 'word': ['positive', 'trust'],
               'words': ['anger', 'negative'], 'working': ['positive'],
               'worm': ['anticipation', 'negative', 'surprise'], 'worn': ['negative', 'sadness'],
               'worried': ['negative', 'sadness'], 'worry': ['anticipation', 'fear', 'negative', 'sadness'],
               'worrying': ['anticipation', 'fear', 'negative', 'sadness'], 'worse': ['fear', 'negative', 'sadness'],
               'worsening': ['disgust', 'negative', 'sadness'],
               'worship': ['anticipation', 'fear', 'joy', 'positive', 'trust'], 'worth': ['positive'],
               'worthless': ['anger', 'disgust', 'negative', 'sadness'], 'worthy': ['positive', 'trust'],
               'wot': ['positive', 'trust'], 'wound': ['anger', 'fear', 'negative', 'sadness'],
               'wrangling': ['anger', 'disgust', 'fear', 'negative', 'sadness'], 'wrath': ['anger', 'fear', 'negative'],
               'wreak': ['anger', 'negative'], 'wreck': ['anger', 'disgust', 'fear', 'negative', 'sadness', 'surprise'],
               'wrecked': ['anger', 'fear', 'negative', 'sadness'], 'wrench': ['negative'], 'wrestling': ['negative'],
               'wretch': ['anger', 'disgust', 'negative', 'sadness'], 'wretched': ['disgust', 'negative', 'sadness'],
               'wring': ['anger'], 'wrinkled': ['sadness'], 'writer': ['positive'], 'wrong': ['negative'],
               'wrongdoing': ['anger', 'disgust', 'negative', 'sadness'],
               'wrongful': ['anger', 'disgust', 'negative', 'sadness'],
               'wrongly': ['anger', 'fear', 'negative', 'sadness'], 'wrought': ['negative'], 'wry': ['negative'],
               'xenophobia': ['fear', 'negative'], 'yawn': ['negative'], 'yawning': ['negative'],
               'yearning': ['anticipation', 'joy', 'negative', 'positive', 'trust'],
               'yell': ['anger', 'fear', 'negative', 'surprise'], 'yellows': ['negative'],
               'yelp': ['anger', 'fear', 'negative', 'surprise'],
               'young': ['anticipation', 'joy', 'positive', 'surprise'], 'younger': ['positive'],
               'youth': ['anger', 'anticipation', 'fear', 'joy', 'positive', 'surprise'], 'zany': ['surprise'],
               'zeal': ['anticipation', 'joy', 'positive', 'surprise', 'trust'],
               'zealous': ['joy', 'positive', 'trust'], 'zest': ['anticipation', 'joy', 'positive', 'trust'],
               'zip': ['negative']}

    def __init__(self, text):
        self.text = text
        blob = TextBlob(text)
        self.words = list(blob.words)
        self.sentences = list(blob.sentences)
        build_word_affect(self)
        top_emotions(self)

    def append_text(self, text_add):
        self.text = self.text + ' ' + text_add
        blob = TextBlob(self.text)
        self.words = list(blob.words)
        self.sentences = list(blob.sentences)
        build_word_affect(self)
