# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/01a_sparsifier.ipynb (unless otherwise specified).

__all__ = ['Sparsifier']

# Cell
import numpy as np
import torch
import torch.nn as nn
from fastcore.basics import store_attr
from .criteria import *

# Cell
class Sparsifier():

    def __init__(self, model, granularity, method, criteria, layer_type=nn.Conv2d):
        store_attr()
        self._save_weights() # Save the original weights

    def prune_layer(self, m, sparsity, round_to=None):
        weight = self.criteria(m, self.granularity)
        mask = self._compute_mask(weight, sparsity, round_to)
        m.register_buffer("_mask", mask) # Put the mask into a buffer
        self._apply(m)

    def prune_model(self, sparsity, round_to=None):
        self.threshold=None
        for m in self.model.modules():
            if isinstance(m, self.layer_type): self.prune_layer(m, sparsity, round_to)

    def _apply(self, m):
        mask = getattr(m, "_mask", None)
        if mask is not None: m.weight.data.mul_(mask)
        if self.granularity == 'filter' and m.bias is not None:
            if mask is not None: m.bias.data.mul_(mask.squeeze()) # We want to prune the bias when pruning filters

    def _mask_grad(self):
        for m in self.model.modules():
            if isinstance(m, self.layer_type) and hasattr(m, '_mask'):
                mask = getattr(m, "_mask")
                if m.weight.grad is not None: m.weight.grad.mul_(mask)
                if self.granularity == 'filter' and m.bias is not None:
                    if m.bias.grad is not None: m.bias.grad.mul_(mask.squeeze())

    def _reset_weights(self): # Reset non-pruned weights
        for m in self.model.modules():
            if hasattr(m, 'weight'):
                init_weights = getattr(m, "_init_weights", m.weight)
                init_biases = getattr(m, "_init_biases", m.bias)
                with torch.no_grad():
                    if m.weight is not None: m.weight.copy_(init_weights)
                    if hasattr(m, 'bias'): m.bias.copy_(init_biases)
                self._apply(m)
            if isinstance(m, nn.modules.batchnorm._BatchNorm): m.reset_parameters()

    def _save_weights(self):
        for m in self.model.modules():
            if hasattr(m, 'weight'):
                m.register_buffer("_init_weights", m.weight.clone())
                b = getattr(m, 'bias', None)
                if b is not None: m.register_buffer("_init_biases", b.clone())

    def _clean_buffers(self):
        for m in self.model.modules():
            if hasattr(m, 'weight'):
                if hasattr(m, '_mask'): del m._buffers["_mask"]
                if hasattr(m, '_init_weights'): del m._buffers["_init_weights"]
                if hasattr(m, '_init_biases'): del m._buffers["_init_biases"]

    def _compute_threshold(self, weight, sparsity):
        if self.method == 'global':
            global_weight = torch.cat([self.criteria(m, self.granularity).view(-1) for m in self.model.modules() if isinstance(m, self.layer_type)])
            if self.threshold is None: self.threshold = torch.quantile(global_weight, sparsity/100) # Compute the threshold globally (only once per model pruning)
            return self.threshold
        elif self.method == 'local':
            return torch.quantile(weight.view(-1), sparsity/100) # Compute the threshold locally
        else: raise NameError('Invalid Method')

    def _rounded_sparsity(self, n_to_prune, round_to):
        return max(round_to*torch.ceil(n_to_prune/round_to), round_to)

    def _compute_mask(self, weight, sparsity, round_to):
        threshold = self._compute_threshold(weight, sparsity)
        if round_to:
            n_to_keep = sum(weight.ge(threshold)).squeeze()
            threshold = torch.topk(weight.squeeze(), int(self._rounded_sparsity(n_to_keep, round_to)))[0].min()
        if threshold > weight.max(): threshold = weight.max() # Make sure we don't remove every weight of a given layer
        return weight.ge(threshold).to(dtype=weight.dtype)

    def print_sparsity(self):
        for k,m in enumerate(self.model.modules()):
            if isinstance(m, self.layer_type):
                print(f"Sparsity in {m.__class__.__name__} {k}: {100. * float(torch.sum(m.weight == 0))/ float(m.weight.nelement()):.2f}%")