"""
Generates input data for `pywapor.et_look`.
"""
from pywapor.collect import downloader
from pywapor.general.logger import log, adjust_logger
from pywapor.general import compositer
import pywapor.general.levels as levels
import datetime
import numpy as np
import os
import pandas as pd
import xarray as xr
from functools import partial
import pywapor.general.pre_defaults as defaults
from pywapor.general.variables import fill_attrs
from pywapor.enhancers.temperature import lapse_rate as _lapse_rate
from pywapor.general.processing_functions import remove_temp_files

def rename_vars(ds, *args):
    """Rename some variables in a dataset.

    Parameters
    ----------
    ds : xr.Dataset
        Input dataset whose variables will be renamed.

    Returns
    -------
    xr.Dataset
        Dataset with renamed variables.
    """
    varis = ["p", "ra", "t_air", "t_air_min", "t_air_max", "u", "vp",
            "u2m", "v2m", "qv", "p_air", "p_air_0", "wv", "t_dew"]
    present_vars = [x for x in varis if x in ds.variables]
    ds = ds.rename({k: k + "_24" for k in present_vars})
    return ds

def lapse_rate(ds, *args):
    """Applies lapse rate correction to variables whose name contains `"t_air"`.

    Parameters
    ----------
    ds : xr.Dataset
        Dataset on whose variables containing `"t_air"` a lapse rate correction will be applied.

    Returns
    -------
    xr.Dataset
        Dataset on whose variables containing `"t_air"` a lapse rate correction has been applied.
    """
    present_vars = [x for x in ds.variables if "t_air" in x]
    for var in present_vars:
        ds = _lapse_rate(ds, var)
    return ds

def calc_doys(ds, *args, bins = None):
    """Calculate the day-of-the-year (doy) in the middle of a timebin and assign the results to a new
    variable `doy`.

    Parameters
    ----------
    ds : xr.Dataset
        Input dataset.
    bins : list, optional
        List of boundaries of the timebins, by default None

    Returns
    -------
    xr.Dataset
        Dataset with a new variable containing the `doy` per timebin.
    """
    bin_doys = [int(pd.Timestamp(x).strftime("%j")) for x in bins]
    doy = np.mean([bin_doys[:-1], bin_doys[1:]], axis=0, dtype = int)
    if "time_bins" in list(ds.variables):
        ds["doy"] = xr.DataArray(doy, coords = ds["time_bins"].coords).chunk("auto")
    return ds

def add_constants(ds, *args):
    """Adds default dimensionless constants to a dataset.

    Parameters
    ----------
    ds : xr.Dataset
        Dataset to which the constants will be added.

    Returns
    -------
    xr.Dataset
        Dataset with extra variables.
    """
    ds = ds.assign(defaults.constants_defaults())
    return ds

def main(folder, latlim, lonlim, timelim, sources = "level_1", bin_length = "DEKAD", enhancers = [lapse_rate]):
    """Prepare input data for `et_look`.

    Parameters
    ----------
    folder : str
        Path to folder in which to store (intermediate) data.
    latlim : list
        Latitude limits of area of interest.
    lonlim : list
        Longitude limits of area of interest.
    timelim : list
        Period for which to prepare data.
    sources : "level_1" | "level_2" | "level_2_v3" | dict
        Configuration for each variable and source.
    bin_length : int | "DEKAD"
        Composite length.
    enhancers : list, optional
        Functions to apply to the xr.Dataset before creating the final
        output, by default `[lapse_rate]`.

    Returns
    -------
    xr.Dataset
        Dataset with input data for `pywapor.et_look`.
    """
    if not os.path.exists(folder):
        os.makedirs(folder)

    _ = adjust_logger(True, folder, "INFO")

    t1 = datetime.datetime.now()
    log.info("> PRE_ET_LOOK").add()

    if isinstance(sources, str):
        sources = levels.pre_et_look_levels(sources, bin_length = bin_length)

    bins = compositer.time_bins(timelim, bin_length)

    general_enhancers = enhancers + [rename_vars, fill_attrs, partial(calc_doys, bins = bins), add_constants]

    adjusted_timelim = [bins[0], bins[-1]]
    buffered_timelim = [adjusted_timelim[0] - np.timedelta64(3, "D"), 
                        adjusted_timelim[1] + np.timedelta64(3, "D")]

    dss, sources = downloader.collect_sources(folder, sources, latlim, lonlim, buffered_timelim)

    ds = compositer.main(dss, sources, folder, general_enhancers, bins)

    t2 = datetime.datetime.now()
    log.sub().info(f"< PRE_ET_LOOK ({str(t2 - t1)})")

    files = remove_temp_files(folder)

    return ds

if __name__ == "__main__":

#     enhancers = "default"
#     diagnostics = None
#     example_source = None
#     bin_length = "DEKAD"
    enhancers = [lapse_rate]

#     import pywapor

#     folder = r"/Users/hmcoerver/pywapor_notebooks_b"
#     latlim = [28.9, 29.7]
#     lonlim = [30.2, 31.2]
#     timelim = ["2021-07-01", "2021-07-11"]
#     composite_length = "DEKAD"

#     et_look_version = "v2"
#     export_vars = "default"

#     level = "level_1"

#     et_look_sources = pywapor.general.levels.pre_et_look_levels(level)

#     et_look_sources = {k: v for k, v in et_look_sources.items() if k in ["ndvi", "z"]}

    # et_look_sources["ndvi"]["products"] = [
    #     {'source': 'MODIS',
    #         'product_name': 'MOD13Q1.061',
    #         'enhancers': 'default'},
    #     {'source': 'MODIS', 
    #         'product_name': 'MYD13Q1.061', 
    #         'enhancers': 'default'},
    #     {'source': 'PROBAV',
    #         'product_name': 'S5_TOC_100_m_C1',
    #         'enhancers': 'default',
    #         'is_example': True}
    # ]

    # et_look_sources["r0"]["products"] = [
    #     {'source': 'MODIS',
    #         'product_name': 'MCD43A3.061',
    #         'enhancers': 'default'},
    #     {'source': 'PROBAV',
    #         'product_name': 'S5_TOC_100_m_C1',
    #         'enhancers': 'default'}
    # ]

    # se_root_sources = pywapor.general.levels.pre_se_root_levels(level)
    # se_root_sources["ndvi"]["products"] = et_look_sources["ndvi"]["products"]

    # from functools import partial
    # et_look_sources["se_root"]["products"] = [
    #     {'source': partial(pywapor.se_root.se_root, sources = se_root_sources),
    #         'product_name': 'v2',
    #         'enhancers': 'default'},]

    # ds = pywapor.pre_et_look.main(folder, latlim, lonlim, timelim, 
    #                                 sources = et_look_sources,
    #                                 bin_length = composite_length)



