from gaurabda.GCSunData import SUNDATA
from gaurabda.GCMoonData import MOONDATA
from gaurabda.GCGregorianDate import GCGregorianDate
import gaurabda.GCDayData as GCDayData
import gaurabda.GCEarthData as GCEarthData
import gaurabda.GCAyanamsha as GCAyanamsha
import gaurabda.GCRasi as GCRasi
import gaurabda.GCMath as GCMath
import gaurabda.GCConjunction as GCConjunction
import gaurabda.GCUT as GCUT
from math import floor

class GCDayData:
    def __init__(self):
        self.jdate = 0.0
        self.sun = SUNDATA()
        self.moon = MOONDATA()
        self.nGaurabdaYear = 0
        self.msAyanamsa = 0
        self.nSunRasi = 0
        self.nMoonRasi = 0
        self.nTithi = 0
        # tithi at arunodaya
        self.nTithiArunodaya = 0
        # tithi at sunset
        self.nTithiSunset = 0
        # tithi elaps.
        self.nTithiElapse = 0.0
        # yoga
        self.nYoga = 0
        # yoga elaps.
        self.nYogaElapse = 0.0
        # naksatra
        self.nNaksatra = 0
        # naksatra elaps.
        self.nNaksatraElapse = 0.0
        # masa
        self.nMasa = 0
        # distance of moon and sun in degrees
        self.msDistance = 0.0

    @property
    def nPaksa(self):
        return 1 if self.nTithi >= 15 else 0

    #********************************************************************/
    #                                                                   */
    # Calculation of tithi, paksa, naksatra, yoga for given             */
    #    Gregorian date                                                 */
    #                                                                   */
    #                                                                   */
    #********************************************************************/

    def DayCalc(self,date,earth):
        # sun position on sunrise on that day
        self.sun.SunCalc( date, earth)
        date.shour = self.sun.sunrise_deg/360.0

        # date.shour is [0..1] time of sunrise in local timezone time
        self.jdate = jdate = date.GetJulianDetailed()

        # moon position at sunrise on that day
        self.moon.Calculate(date.GetJulianDetailed(), earth)

        self.msDistance = GCMath.putIn360( self.moon.longitude_deg - self.sun.longitude_deg - 180.0)
        self.msAyanamsa = GCAyanamsha.GetAyanamsa( jdate )

        # tithi
        d = self.msDistance / 12.0
        self.nTithi = int(floor(d))
        self.nTithiElapse = (d - floor(d)) * 100.0


        # naksatra
        d = GCMath.putIn360( self.moon.longitude_deg - self.msAyanamsa )
        d = ( d * 3.0) / 40.0
        self.nNaksatra = int(floor(d))
        self.nNaksatraElapse = (d - floor(d)) * 100.0

        # yoga
        d = GCMath.putIn360( self.moon.longitude_deg + self.sun.longitude_deg - 2*self.msAyanamsa)
        d = (d * 3.0) / 40.0
        self.nYoga = int(floor(d))
        self.nYogaElapse = (d - floor(d)) * 100.0

        # masa
        self.nMasa = -1

        # rasi
        self.nSunRasi = GCRasi.GetRasi(self.sun.longitude_deg, self.msAyanamsa)
        self.nMoonRasi = GCRasi.GetRasi(self.moon.longitude_deg, self.msAyanamsa)

        moon = MOONDATA()
        date.shour = self.sun.sunset_deg/360.0
        moon.Calculate(date.GetJulianDetailed(), earth)
        d = GCMath.putIn360(moon.longitude_deg - self.sun.longitude_set_deg - 180) / 12.0
        self.nTithiSunset = int(floor(d))

        date.shour = self.sun.arunodaya_deg/360.0
        moon.Calculate(date.GetJulianDetailed(), earth)
        d = GCMath.putIn360(moon.longitude_deg - self.sun.longitude_arun_deg - 180) / 12.0
        self.nTithiArunodaya = int(floor(d))

        return 1

    def MasaCalc(self, date, earth):
        PREV_MONTHS = 6
        L = [0] * 8
        C = [GCGregorianDate() for i in range(8)]
        R = [0] * 8

        ksaya_from = -1
        ksaya_to = -1

        date.shour = self.sun.sunrise_deg / 360.0 + earth.tzone / 24.0

        # STEP 1: calculate position of the sun and moon
        # it is done by previous call of DayCalc
        # and results are in argument GCDayData day
        # *DayCalc(date, earth, day, moon, sun);*

        L[1] = GCConjunction.GetNextConjunctionEx(date, C[1], False, earth)
        L[0] = GCConjunction.GetNextConjunctionEx(C[1], C[0], True, earth)

        # on Pratipat (nTithi == 15) we need to look for previous conjunction
        # but this conjunction can occur on this date before sunrise
        # so we need to include this very date into looking for conjunction
        # on other days we cannot include it
        # and exclude it from looking for next because otherwise that will cause
        # incorrect detection of Purusottama adhika masa
        L[2] = GCConjunction.GetPrevConjunctionEx(date, C[2], False, earth)

        for n in range(3,PREV_MONTHS):
            L[n] = GCConjunction.GetPrevConjunctionEx(C[n-1], C[n], True, earth)

        for n in range(0,PREV_MONTHS):
            R[n] = GCRasi.GetRasi(L[n], GCAyanamsha.GetAyanamsa(C[n].GetJulian()))

    #    TRACE("TEST Date: %d %d %d\n", date.day, date.month, date.year)
    #    TRACE("FOUND CONJ Date: %d %d %d rasi: %d\n", C[1].day, C[1].month, C[1].year, R[1])
    #    TRACE("FOUND CONJ Date: %d %d %d rasi: %d\n", C[2].day, C[2].month, C[2].year, R[2])
    #    TRACE("FOUND CONJ Date: %d %d %d rasi: %d\n", C[3].day, C[3].month, C[3].year, R[3])
    #    TRACE("FOUND CONJ Date: %d %d %d rasi: %d\n", C[4].day, C[4].month, C[4].year, R[4])
    #    TRACE("---\n")

        # look for ksaya month
        ksaya_from = -1
        for n in range(PREV_MONTHS - 2,0,-1):
            if (R[n+1] + 2) % 12 == R[n]:
                ksaya_from = n
                break

        if ksaya_from >= 0:
            for n in range(ksaya_from,0,-1):
                if R[n] == R[n-1]:
                    ksaya_to = n
                    break

            if ksaya_to >= 0:
                pass
                # adhika masa found
                # now correct succession of rasis
            else:
                # adhika masa not found
                # there will be some break in masa queue
                ksaya_to = 0

            current_rasi = R[ksaya_from + 1] + 1
            for n in range(ksaya_from,ksaya_to-1,-1):
                R[n] = current_rasi
                current_rasi = (current_rasi + 1) % 12

        # STEP 3: test for adhika masa
        # test for adhika masa
        if R[1] == R[2]:
            # it is adhika masa
            masa = 12
            rasi = R[1]
        else:
            # STEP 2. select nearest Conjunction
            if self.nPaksa == 0:
                masa = R[1]
            elif self.nPaksa == 1:
                masa = R[2]
            rasi = masa

        # calculation of Gaurabda year
        self.nGaurabdaYear = date.year - 1486

        if (rasi > 7) and (rasi < 11): # Visnu
            if date.month < 6:
                self.nGaurabdaYear-=1

        self.nMasa = masa

        return masa



def GetFirstDayOfYear(earth, nYear):

    a = [2, 15, 3, 1, 3, 15, 4, 1, 4, 15]
    d = GCGregorianDate()
    day = GCDayData()

    if nYear >= 1950 and nYear < 2058:
        tmp = gGaurBeg[(nYear - 1950)*26 + 22]
        d.month = (tmp & 0x3e0) >> 5
        d.day   = (tmp & 0x1f)
        d.year  = nYear
        d.NextDay()
        a[0] = d.month
        a[1] = d.day

    for i in range(0,10,2):
        d.year = nYear
        d.month = a[i]
        d.day = a[i+1]

        day.DayCalc(d, earth)
        masa = day.MasaCalc(d, earth)
        gy = day.nGaurabdaYear

        if masa == 11: # visnu masa
            while True:
                # shifts date
                step = max(int(day.nTithi / 2), 1)
                for j in range(step):
                    d.PreviousDay()
                # try new time
                day.DayCalc(d, earth)
                if day.nTithi>=28: break
            d.NextDay()
            d.tzone = earth.tzone
            d.shour = day.sun.sunrise_deg / 360.0
            return d

    d.year = -1
    d.month = -1
    d.day = -1
    d.tzone = earth.tzone
    d.shour = day.sun.sunrise_deg / 360.0

    return d


gGaurBeg = [
#Gaurabda Year = 464
1996931, 1996946, 1996963, 1996977, 1996993, 1997040, 1997054, 1997070, 1997084, 1997100, 1997114, 1997132, 1997146, 1997162,
    1997177, 1997194, 1997209, 1997864, 1997879, 1997895, 1997910, 1997928, 1996901, 1996915, 1997022, 1997008,
#Gaurabda Year = 465
1997974, 1997990, 1998006, 1998021, 1998036, 1998053, 1998067, 1998083, 1998097, 1998114, 1998128, 1998145, 1998159, 1998175,
    1998190, 1998205, 1998222, 1998237, 1998892, 1998907, 1998923, 1998938, 1997944, 1997959, 0, 0,
#Gaurabda Year = 466
1998987, 1999001, 1999018, 1999032, 1999049, 1999063, 1999080, 1999094, 1999110, 1999125, 1999140, 1999156, 1999172, 1999187,
    1999202, 1999218, 1999234, 1999249, 1999263, 1999920, 1999934, 1999950, 1998956, 1998970, 0, 0,
#Gaurabda Year = 467
1999999, 2000046, 2000061, 2000076, 2000092, 2000107, 2000123, 2000138, 2000153, 2000169, 2000183, 2000200, 2000215, 2000231,
    2000245, 2000263, 2000277, 2000933, 2000947, 2000964, 2000978, 2000997, 1999969, 1999984, 2000030, 2000014,
#Gaurabda Year = 468
2001043, 2001059, 2001074, 2001090, 2001105, 2001121, 2001136, 2001150, 2001167, 2001181, 2001197, 2001211, 2001229, 2001243,
    2001259, 2001274, 2001290, 2001306, 2001961, 2001976, 2001991, 2002007, 2001012, 2001028, 0, 0,
#Gaurabda Year = 469
2002056, 2002071, 2002087, 2002102, 2002118, 2002133, 2002150, 2002164, 2002180, 2002225, 2002242, 2002256, 2002273, 2002287,
    2002302, 2002319, 2002333, 2002989, 2003004, 2003020, 2003034, 2003053, 2002025, 2002040, 2002211, 2002194,
#Gaurabda Year = 470
2003097, 2003115, 2003129, 2003145, 2003160, 2003177, 2003191, 2003207, 2003222, 2003237, 2003252, 2003268, 2003284, 2003299,
    2003315, 2003331, 2003346, 2004001, 2004016, 2004031, 2004047, 2004066, 2003067, 2003083, 0, 0,
#Gaurabda Year = 471
2004111, 2004126, 2004142, 2004158, 2004173, 2004188, 2004204, 2004220, 2004235, 2004250, 2004266, 2004280, 2004297, 2004311,
    2004328, 2004342, 2004359, 2004373, 2005030, 2005044, 2005061, 2005075, 2004080, 2004097, 0, 0,
#Gaurabda Year = 472
2005124, 2005139, 2005156, 2005171, 2005186, 2005202, 2005218, 2005263, 2005278, 2005294, 2005308, 2005325, 2005340, 2005356,
    2005371, 2005387, 2005402, 2006057, 2006073, 2006088, 2006104, 2006122, 2005094, 2005109, 2005247, 2005233,
#Gaurabda Year = 473
2006168, 2006184, 2006199, 2006215, 2006229, 2006246, 2006261, 2006277, 2006291, 2006307, 2006321, 2006339, 2006353, 2006369,
    2006384, 2006401, 2006415, 2006430, 2007086, 2007100, 2007117, 2007131, 2006137, 2006152, 0, 0,
#Gaurabda Year = 474
2007180, 2007194, 2007212, 2007226, 2007242, 2007257, 2007273, 2007288, 2007303, 2007319, 2007334, 2007349, 2007365, 2007381,
    2007396, 2007411, 2007427, 2007443, 2008098, 2008113, 2008129, 2008144, 2007150, 2007164, 0, 0,
#Gaurabda Year = 475
2008194, 2008208, 2008225, 2008270, 2008285, 2008301, 2008316, 2008332, 2008346, 2008362, 2008377, 2008394, 2008408, 2008425,
    2008439, 2008456, 2008470, 2009127, 2009141, 2009157, 2009172, 2009191, 2008163, 2008177, 2008255, 2008239,
#Gaurabda Year = 476
2009236, 2009253, 2009268, 2009283, 2009298, 2009314, 2009330, 2009345, 2009360, 2009374, 2009390, 2009405, 2009422, 2009437,
    2009452, 2009468, 2009484, 2009499, 2010154, 2010170, 2010185, 2010200, 2009206, 2009221, 0, 0,
#Gaurabda Year = 477
2010249, 2010264, 2010281, 2010296, 2010312, 2010326, 2010343, 2010357, 2010374, 2010388, 2010404, 2010418, 2010435, 2010481,
    2010497, 2010512, 2010527, 2011183, 2011197, 2011214, 2011228, 0, 2010219, 2010234, 2010466, 2010450,
#Gaurabda Year = 478
2011291, 2011308, 2011323, 2011339, 2011353, 2011370, 2011385, 2011400, 2011416, 2011431, 2011446, 2011462, 2011477, 2011493,
    2011508, 2011524, 2011539, 2012195, 2012210, 2012226, 2012240, 2012260, 0, 2011277, 2011260, 2011246,
#Gaurabda Year = 479
2012304, 2012322, 2012336, 2012351, 2012366, 2012382, 2012398, 2012413, 2012429, 2012443, 2012459, 2012473, 2012491, 2012505,
    2012521, 2012535, 2012553, 2012567, 2013223, 2013238, 2013254, 2013269, 2012274, 2012290, 0, 0,
#Gaurabda Year = 480
2013318, 2013333, 2013349, 2013365, 2013380, 2013395, 2013411, 2013457, 2013471, 2013487, 2013502, 2013518, 2013534, 2013549,
    2013564, 2013580, 2013596, 2014251, 2014267, 2014282, 2014297, 2014315, 2013287, 2013302, 2013442, 2013427,
#Gaurabda Year = 481
2014361, 2014378, 2014392, 2014409, 2014423, 2014440, 2014454, 2014470, 2014484, 2014501, 2014515, 2014532, 2014547, 2014563,
    2014577, 2014594, 2014609, 2014623, 2015280, 2015294, 2015311, 2015325, 2014330, 2014346, 0, 0,
#Gaurabda Year = 482
2015374, 2015388, 2015405, 2015420, 2015435, 2015450, 2015467, 2015482, 2015497, 2015512, 2015527, 2015543, 2015559, 2015574,
    2015589, 2015605, 2015621, 2015636, 2016292, 2016306, 2016323, 2016337, 2015343, 2015357, 0, 0,
#Gaurabda Year = 483
2016387, 2016401, 2016419, 2016433, 2016449, 2016495, 2016509, 2016526, 2016540, 2016556, 2016570, 2016588, 2016602, 2016618,
    2016632, 2016650, 2016664, 2017320, 2017335, 2017351, 2017366, 2017384, 2016356, 2016370, 2016478, 2016463,
#Gaurabda Year = 484
2017430, 2017446, 2017462, 2017476, 2017492, 2017508, 2017523, 2017539, 2017553, 2017569, 2017584, 2017601, 2017615, 2017631,
    2017646, 2017661, 2017677, 2017693, 2018348, 2018363, 2018379, 2018394, 2017399, 2017414, 0, 0,
#Gaurabda Year = 485
2018443, 2018458, 2018475, 2018489, 2018505, 2018519, 2018537, 2018551, 2018567, 2018581, 2018597, 2018612, 2018629, 2018644,
    2018659, 2018674, 2018691, 2018706, 2019361, 2019377, 2019391, 2019407, 2018412, 2018427, 0, 0,
#Gaurabda Year = 486
2019454, 2019502, 2019517, 2019532, 2019547, 2019563, 2019579, 2019594, 2019609, 2019624, 2019639, 2019656, 2019671, 2019686,
    2019701, 2019718, 2019733, 2020389, 2020403, 2020420, 2020434, 2020453, 2019421, 2019440, 2019485, 2019470,
#Gaurabda Year = 487
2020498, 2020515, 2020530, 2020546, 2020560, 2020577, 2020592, 2020606, 2020622, 2020636, 2020653, 2020667, 2020684, 2020698,
    2020715, 2020729, 2020746, 2020761, 2021417, 2021432, 2021447, 2021462, 2020467, 2020484, 0, 0,
#Gaurabda Year = 488
2021511, 2021527, 2021543, 2021558, 2021573, 2021589, 2021605, 2021620, 2021636, 2021680, 2021698, 2021712, 2021727, 2021742,
    2021758, 2021774, 2021789, 2022445, 2022460, 2022475, 2022490, 2022509, 2021481, 2021496, 2021666, 2021650,
#Gaurabda Year = 489
2022554, 2022572, 2022586, 2022602, 2022616, 2022634, 2022648, 2022664, 2022678, 2022694, 2022709, 2022725, 2022740, 2022756,
    2022771, 2022787, 2022803, 2023458, 2023473, 2023487, 2023504, 2023521, 2022524, 2022540, 0, 0,
#Gaurabda Year = 490
2023567, 2023582, 2023598, 2023613, 2023629, 2023644, 2023660, 2023675, 2023690, 2023706, 2023721, 2023736, 2023752, 2023767,
    2023783, 2023798, 2023815, 2023829, 2024486, 2024500, 2024516, 2024530, 2023536, 2023551, 0, 0,
#Gaurabda Year = 491
2024580, 2024595, 2024612, 2024626, 2024642, 2024657, 2024674, 2024719, 2024733, 2024750, 2024764, 2024781, 2024795, 2024812,
    2024826, 2024843, 2024858, 2025513, 2025529, 2025544, 2025559, 2025577, 2024550, 2024564, 2024703, 2024689,
#Gaurabda Year = 492
2025624, 2025640, 2025655, 2025670, 2025685, 2025702, 2025716, 2025732, 2025747, 2025763, 2025777, 2025795, 2025809, 2025825,
    2025839, 2025857, 2025871, 2025886, 2026542, 2026556, 2026572, 2026587, 2025593, 2025608, 0, 0,
#Gaurabda Year = 493
2026637, 2026651, 2026668, 2026682, 2026699, 2026713, 2026730, 2026744, 2026760, 2026775, 2026791, 2026806, 2026822, 2026837,
    2026853, 2026868, 2026884, 2026900, 2027555, 2027570, 2027585, 2027601, 2026606, 2026620, 0, 0,
#Gaurabda Year = 494
2027649, 2027663, 2027681, 2027725, 2027741, 2027757, 2027772, 2027787, 2027802, 2027818, 2027833, 2027849, 2027864, 2027880,
    2027895, 2027912, 2027926, 2028583, 2028597, 2028613, 2028627, 2028647, 2027618, 2027633, 2027710, 2027695,
#Gaurabda Year = 495
2028692, 2028709, 2028723, 2028739, 2028754, 2028770, 2028786, 2028801, 2028816, 2028830, 2028846, 2028860, 2028878, 2028892,
    2028908, 2028923, 2028940, 2028955, 2029610, 2029625, 2029641, 2029656, 2028661, 2028677, 0, 0,
#Gaurabda Year = 496
2029705, 2029720, 2029736, 2029752, 2029767, 2029782, 2029799, 2029813, 2029829, 2029843, 2029860, 2029905, 2029922, 2029936,
    2029953, 2029968, 2029983, 2030638, 2030653, 2030669, 2030684, 2030703, 2029674, 2029690, 2029891, 2029874,
#Gaurabda Year = 497
2030748, 2030765, 2030779, 2030796, 2030810, 2030827, 2030841, 2030857, 2030872, 2030887, 2030903, 2030919, 2030934, 2030949,
    2030965, 2030981, 2030996, 2031651, 2031667, 2031682, 2031697, 2031715, 2030717, 2030734, 0, 0,
#Gaurabda Year = 498
2031760, 2031778, 2031792, 2031807, 2031822, 2031838, 2031853, 2031869, 2031884, 2031899, 2031915, 2031929, 2031946, 2031961,
    2031977, 2031991, 2032009, 2032023, 2032679, 2032693, 2032710, 2032724, 2031730, 2031746, 0, 0,
#Gaurabda Year = 499
2032774, 2032789, 2032805, 2032820, 2032836, 2032851, 2032867, 2032913, 2032927, 2032943, 2032957, 2032974, 2032989, 2033005,
    2033020, 2033036, 2033052, 2033707, 2033722, 2033737, 2033753, 2033771, 2032743, 2032758, 2032897, 2032882,
#Gaurabda Year = 500
2033817, 2033833, 2033848, 2033864, 2033879, 2033896, 2033910, 2033926, 2033940, 2033957, 2033971, 2033988, 2034002, 2034019,
    2034033, 2034050, 2034065, 2034079, 2034735, 2034750, 2034766, 2034780, 2033786, 2033802, 0, 0,
#Gaurabda Year = 501
2034830, 2034844, 2034862, 2034876, 2034892, 2034907, 2034924, 2034938, 2034954, 2034969, 2034984, 2034999, 2035015, 2035031,
    2035046, 2035062, 2035078, 2035093, 2035748, 2035763, 2035779, 2035794, 2034800, 2034814, 0, 0,
#Gaurabda Year = 502
2035843, 2035857, 2035874, 2035919, 2035934, 2035950, 2035966, 2035981, 2035996, 2036012, 2036026, 2036043, 2036057, 2036074,
    2036088, 2036105, 2036119, 2036776, 2036790, 2036807, 2036821, 2036840, 2035812, 2035826, 2035905, 2035888,
#Gaurabda Year = 503
2036885, 2036902, 2036917, 2036932, 2036948, 2036964, 2036979, 2036994, 2037009, 2037025, 2037040, 2037054, 2037071, 2037086,
    2037101, 2037117, 2037133, 2037148, 2037803, 2037819, 2037834, 2037850, 2036855, 2036870, 0, 0,
#Gaurabda Year = 504
2037898, 2037914, 2037930, 2037945, 2037961, 2037975, 2037992, 2038006, 2038023, 2038037, 2038053, 2038067, 2038085, 2038099,
    2038115, 2038130, 2038147, 2038161, 2038817, 2038832, 2038846, 2038863, 2037868, 2037883, 0, 0,
#Gaurabda Year = 505
2038911, 2038959, 2038973, 2038989, 2039004, 2039020, 2039035, 2039050, 2039066, 2039081, 2039096, 2039112, 2039128, 2039143,
    2039158, 2039174, 2039190, 2039845, 2039860, 2039876, 2039891, 2039909, 2038881, 2038897, 2038941, 2038927,
#Gaurabda Year = 506
2039954, 2039971, 2039985, 2040002, 2040016, 2040033, 2040047, 2040062, 2040078, 2040092, 2040108, 2040123, 2040140, 2040154,
    2040171, 2040185, 2040202, 2040216, 2040873, 2040887, 2040903, 2040918, 2039923, 2039940, 0, 0,
#Gaurabda Year = 507
2040967, 2040982, 2040999, 2041014, 2041029, 2041044, 2041060, 2041076, 2041091, 2041136, 2041153, 2041168, 2041183, 2041198,
    2041214, 2041230, 2041245, 2041900, 2041916, 2041931, 2041946, 2041965, 2040937, 2040952, 2041121, 2041106,
#Gaurabda Year = 508
2042010, 2042027, 2042042, 2042058, 2042072, 2042089, 2042103, 2042120, 2042134, 2042150, 2042164, 2042181, 2042196, 2042212,
    2042227, 2042243, 2042258, 2042914, 2042929, 2042943, 2042960, 2042978, 2041980, 2041995, 0, 0,
#Gaurabda Year = 509
2043024, 2043038, 2043055, 2043070, 2043086, 2043100, 2043117, 2043132, 2043147, 2043163, 2043177, 2043193, 2043209, 2043224,
    2043240, 2043255, 2043271, 2043286, 2043942, 2043957, 2043973, 2043987, 2042993, 2043007, 0, 0,
#Gaurabda Year = 510
2044036, 2044050, 2044068, 2044082, 2044098, 2044144, 2044159, 2044175, 2044189, 2044205, 2044219, 2044237, 2044251, 2044267,
    2044281, 2044299, 2044313, 2044969, 2044984, 2045000, 2045015, 2045033, 2044006, 2044020, 2044130, 2044112,
#Gaurabda Year = 511
2045079, 2045095, 2045111, 2045126, 2045141, 2045157, 2045173, 2045188, 2045203, 2045218, 2045233, 2045250, 2045264, 2045281,
    2045295, 2045310, 2045326, 2045342, 2045997, 2046012, 2046028, 2046043, 2045048, 2045064, 0, 0,
#Gaurabda Year = 512
2046092, 2046107, 2046124, 2046138, 2046155, 2046169, 2046186, 2046200, 2046216, 2046230, 2046247, 2046261, 2046278, 2046293,
    2046308, 2046323, 2046340, 2046355, 2047010, 2047026, 2047041, 2047057, 2046061, 2046076, 0, 0,
#Gaurabda Year = 513
2047105, 2047121, 2047137, 2047182, 2047197, 2047213, 2047229, 2047244, 2047259, 2047274, 2047290, 2047306, 2047321, 2047336,
    2047352, 2047368, 2047383, 2048039, 2048053, 2048070, 2048084, 2048102, 2047075, 2047090, 2047167, 2047152,
#Gaurabda Year = 514
2048147, 2048165, 2048179, 2048195, 2048209, 2048226, 2048241, 2048255, 2048272, 2048286, 2048302, 2048316, 2048334, 2048348,
    2048364, 2048378, 2048396, 2048410, 2049066, 2049081, 2049097, 2049112, 2048116, 2048133, 0, 0,
#Gaurabda Year = 515
2049160, 2049176, 2049192, 2049207, 2049222, 2049238, 2049254, 2049269, 2049285, 2049299, 2049315, 2049361, 2049378, 2049392,
    2049409, 2049423, 2049439, 2050094, 2050109, 2050125, 2050140, 2050158, 2049130, 2049145, 2049347, 2049330,
#Gaurabda Year = 516
2050203, 2050221, 2050235, 2050251, 2050265, 2050283, 2050297, 2050313, 2050327, 2050343, 2050358, 2050375, 2050390, 2050405,
    2050420, 2050437, 2050452, 2051107, 2051123, 2051138, 2051153, 2051171, 2050173, 2050189, 0, 0,
#Gaurabda Year = 517
2051217, 2051234, 2051249, 2051265, 2051279, 2051294, 2051310, 2051326, 2051341, 2051356, 2051371, 2051386, 2051403, 2051418,
    2051433, 2051448, 2051465, 2051480, 2052136, 2052150, 2052167, 2052181, 2051187, 2051202, 0, 0,
#Gaurabda Year = 518
2052230, 2052244, 2052261, 2052276, 2052292, 2052306, 2052323, 2052368, 2052382, 2052399, 2052413, 2052430, 2052444, 2052461,
    2052475, 2052492, 2052507, 2053163, 2053178, 2053193, 2053208, 2053227, 2052199, 2052213, 2052353, 2052338,
#Gaurabda Year = 519
2053273, 2053289, 2053304, 2053319, 2053335, 2053351, 2053366, 2053381, 2053396, 2053412, 2053426, 2053444, 2053458, 2053474,
    2053488, 2053506, 2053520, 2053535, 2054191, 2054206, 2054221, 2054236, 2053242, 2053257, 0, 0,
#Gaurabda Year = 520
2054286, 2054300, 2054318, 2054332, 2054348, 2054362, 2054380, 2054394, 2054410, 2054424, 2054440, 2054455, 2054471, 2054486,
    2054502, 2054517, 2054533, 2054549, 2055204, 2055219, 2055234, 2055250, 2054255, 2054270, 0, 0,
#Gaurabda Year = 521
2055299, 2055314, 2055331, 2055376, 2055393, 2055407, 2055422, 2055437, 2055453, 2055468, 2055483, 2055499, 2055514, 2055530,
    2055545, 2055562, 2055576, 2056233, 2056247, 2056263, 2056277, 2056296, 2055268, 2055283, 2055361, 2055345,
#Gaurabda Year = 522
2056341, 2056358, 2056372, 2056388, 2056403, 2056419, 2056435, 2056450, 2056465, 2056479, 2056496, 2056510, 2056527, 2056541,
    2056558, 2056572, 2056589, 2056604, 2057259, 2057275, 2057290, 2057305, 2056310, 2056326, 0, 0,
#Gaurabda Year = 523
2057354, 2057369, 2057386, 2057401, 2057416, 2057431, 2057448, 2057462, 2057478, 2057493, 2057509, 2057523, 2057541, 2057555,
    2057571, 2057585, 2057603, 2057617, 2058273, 2058288, 2058302, 2058318, 2057323, 2057339, 0, 0,
#Gaurabda Year = 524
2058366, 2058414, 2058428, 2058445, 2058459, 2058476, 2058490, 2058506, 2058521, 2058537, 2058552, 2058568, 2058583, 2058598,
    2058614, 2058630, 2058646, 2059301, 2059316, 2059331, 2059347, 2059365, 2058337, 2058352, 2058397, 2058383,
#Gaurabda Year = 525
2059410, 2059428, 2059442, 2059458, 2059472, 2059490, 2059504, 2059519, 2059534, 2059549, 2059565, 2059580, 2059596, 2059611,
    2059627, 2059641, 2059659, 2059673, 2060330, 2060344, 2060360, 2060374, 2059380, 2059396, 0, 0,
#Gaurabda Year = 526
2060423, 2060438, 2060455, 2060469, 2060485, 2060500, 2060516, 2060532, 2060546, 2060592, 2060606, 2060624, 2060638, 2060654,
    2060669, 2060686, 2060701, 2061356, 2061371, 2061387, 2061402, 2061420, 2060393, 2060407, 2060577, 2060562,
#Gaurabda Year = 527
2061466, 2061482, 2061498, 2061513, 2061528, 2061545, 2061559, 2061575, 2061589, 2061606, 2061620, 2061637, 2061651, 2061668,
    2061682, 2061699, 2061714, 2062370, 2062384, 2062399, 2062415, 2062434, 2061436, 2061451, 0, 0,
#Gaurabda Year = 528
2062480, 2062494, 2062511, 2062525, 2062542, 2062556, 2062573, 2062587, 2062603, 2062618, 2062633, 2062649, 2062665, 2062680,
    2062695, 2062711, 2062727, 2062742, 2063397, 2063413, 2063428, 2063443, 2062449, 2062463, 0, 0,
#Gaurabda Year = 529
2063493, 2063507, 2063525, 2063539, 2063555, 2063600, 2063617, 2063631, 2063646, 2063662, 2063676, 2063693, 2063708, 2063724,
    2063738, 2063756, 2063770, 2064426, 2064440, 2064457, 2064471, 2064489, 2063462, 2063477, 2063586, 2063569,
#Gaurabda Year = 530
2064535, 2064551, 2064566, 2064582, 2064597, 2064613, 2064628, 2064643, 2064659, 2064674, 2064689, 2064705, 2064720, 2064735,
    2064751, 2064766, 2064782, 2064798, 2065453, 2065468, 2065483, 2065499, 2064504, 2064520, 0, 0,
#Gaurabda Year = 531
2065548, 2065563, 2065579, 2065594, 2065610, 2065624, 2065642, 2065656, 2065672, 2065686, 2065703, 2065717, 2065734, 2065748,
    2065764, 2065779, 2065796, 2065810, 2066466, 2066481, 2066497, 2066512, 2065517, 2065532, 0, 0,
#Gaurabda Year = 532
2066561, 2066576, 2066590, 2066638, 2066653, 2066669, 2066684, 2066700, 2066715, 2066730, 2066745, 2066761, 2066777, 2066792,
    2066807, 2066824, 2066839, 2067494, 2067509, 2067525, 2067540, 2067559, 2066530, 2066546, 2066622, 2066608,
#Gaurabda Year = 533
2067604, 2067621, 2067635, 2067652, 2067666, 2067683, 2067697, 2067713, 2067728, 2067743, 2067759, 2067773, 2067790, 2067804,
    2067821, 2067835, 2067852, 2067866, 2068523, 2068537, 2068554, 2068568, 2067573, 2067590, 0, 0,
#Gaurabda Year = 534
2068616, 2068631, 2068648, 2068663, 2068678, 2068694, 2068710, 2068725, 2068740, 2068755, 2068771, 2068817, 2068833, 2068847,
    2068865, 2068879, 2068894, 2069549, 2069565, 2069580, 2069596, 2069614, 2068586, 2068601, 2068802, 2068786,
#Gaurabda Year = 535
2069659, 2069676, 2069691, 2069707, 2069721, 2069738, 2069752, 2069769, 2069783, 2069799, 2069813, 2069831, 2069845, 2069861,
    2069876, 2069893, 2069907, 2070563, 2070578, 2070593, 2070609, 2070627, 2069629, 2069644, 0, 0,
#Gaurabda Year = 536
2070673, 2070689, 2070705, 2070719, 2070735, 2070749, 2070766, 2070781, 2070796, 2070812, 2070827, 2070842, 2070858, 2070874,
    2070889, 2070904, 2070920, 2070936, 2071591, 2071606, 2071622, 2071637, 2070643, 2070658, 0, 0,
#Gaurabda Year = 537
2071687, 2071701, 2071718, 2071732, 2071749, 2071763, 2071780, 2071825, 2071839, 2071855, 2071870, 2071887, 2071901, 2071918,
    2071932, 2071949, 2071963, 2072620, 2072634, 2072650, 2072665, 2072683, 2071656, 2071670, 2071810, 2071794,
#Gaurabda Year = 538
2072728, 2072745, 2072760, 2072775, 2072790, 2072806, 2072822, 2072837, 2072852, 2072867, 2072882, 2072899, 2072914, 2072930,
    2072944, 2072962, 2072976, 2072991, 2073646, 2073662, 2073677, 2073692, 2072698, 2072713, 0, 0,
#Gaurabda Year = 539
2073741, 2073756, 2073773, 2073788, 2073804, 2073818, 2073835, 2073849, 2073866, 2073880, 2073896, 2073910, 2073927, 2073942,
    2073958, 2073973, 2073989, 2074004, 2074660, 2074675, 2074690, 2074706, 2073711, 2073726, 0, 0,
#Gaurabda Year = 540
2074754, 2074770, 2074786, 2074832, 2074846, 2074863, 2074878, 2074893, 2074909, 2074923, 2074939, 2074955, 2074970, 2074986,
    2075001, 2075017, 2075032, 2075688, 2075703, 2075719, 2075733, 2075753, 2074724, 2074739, 2074817, 2074801,
#Gaurabda Year = 541
2075797, 2075815, 2075829, 2075845, 2075859, 2075877, 2075891, 2075907, 2075922, 2075937, 2075952, 2075966, 2075984, 2075998,
    2076014, 2076028, 2076046, 2076060, 2076716, 2076731, 2076747, 2076762, 2075767, 2075783, 0, 0,
#Gaurabda Year = 542
2076810, 2076825, 2076841, 2076857, 2076872, 2076887, 2076903, 2076918, 2076934, 2076949, 2076964, 2076979, 2076996, 2077010,
    2077027, 2077041, 2077058, 2077072, 2077729, 2077743, 2077758, 2077774, 2076779, 2076794, 0, 0,
#Gaurabda Year = 543
2077853, 2077870, 2077884, 2077901, 2077915, 2077932, 2077946, 2077962, 2077976, 2077993, 2078007, 2078024, 2078039, 2078054,
    2078069, 2078086, 2078101, 2078756, 2078772, 2078787, 2078803, 2078821, 2077793, 2077838, 2077822, 2077807,
#Gaurabda Year = 544
2078866, 2078883, 2078898, 2078914, 2078928, 2078945, 2078959, 2078975, 2078990, 2079005, 2079020, 2079036, 2079052, 2079067,
    2079082, 2079098, 2079114, 2079129, 2079785, 2079799, 2079816, 2079830, 2078836, 2078851, 0, 0,
#Gaurabda Year = 545
2079880, 2079894, 2079912, 2079926, 2079942, 2079956, 2079973, 2079988, 2080003, 2080049, 2080065, 2080081, 2080095, 2080111,
    2080125, 2080143, 2080157, 2080813, 2080828, 2080844, 2080859, 2080876, 2079849, 2079863, 2080034, 2080019,
#Gaurabda Year = 546
2080922, 2080938, 2080953, 2080968, 2080984, 2081000, 2081015, 2081031, 2081045, 2081061, 2081076, 2081093, 2081107, 2081124,
    2081138, 2081155, 2081169, 2081826, 2081840, 2081855, 2081871, 2081890, 2080891, 2080906, 0, 0,
#Gaurabda Year = 547
2081935, 2081949, 2081967, 2081981, 2081997, 2082011, 2082029, 2082043, 2082059, 2082073, 2082089, 2082104, 2082121, 2082136,
    2082151, 2082166, 2082183, 2082198, 2082853, 2082869, 2082884, 2082899, 2081904, 2081919, 0, 0,
#Gaurabda Year = 548
2082948, 2082963, 2082980, 2082995, 2083011, 2083056, 2083073, 2083087, 2083102, 2083117, 2083132, 2083149, 2083164, 2083179,
    2083194, 2083211, 2083226, 2083882, 2083896, 2083913, 2083927, 2083946, 2082917, 2082933, 2083042, 2083025,
#Gaurabda Year = 549
2083991, 2084008, 2084023, 2084039, 2084053, 2084070, 2084085, 2084100, 2084115, 2084130, 2084146, 2084162, 2084177, 2084191,
    2084208, 2084222, 2084239, 2084254, 2084910, 2084925, 2084940, 2084955, 2083960, 2083977, 0, 0,
#Gaurabda Year = 550
2085003, 2085019, 2085035, 2085050, 2085065, 2085081, 2085097, 2085112, 2085127, 2085142, 2085158, 2085172, 2085190, 2085204,
    2085220, 2085234, 2085252, 2085266, 2085922, 2085937, 2085953, 2085967, 2084973, 2084988, 0, 0,
#Gaurabda Year = 551
2086017, 2086032, 2086046, 2086094, 2086108, 2086125, 2086140, 2086156, 2086170, 2086186, 2086201, 2086217, 2086232, 2086248,
    2086263, 2086279, 2086295, 2086950, 2086965, 2086980, 2086996, 2087014, 2085986, 2086001, 2086078, 2086064,
#Gaurabda Year = 552
2087060, 2087077, 2087091, 2087107, 2087121, 2087139, 2087153, 2087169, 2087183, 2087199, 2087214, 2087229, 2087245, 2087260,
    2087276, 2087291, 2087308, 2087322, 2087979, 2087993, 2088009, 2088023, 2087029, 2087045, 0, 0,
#Gaurabda Year = 553
2088073, 2088088, 2088105, 2088119, 2088135, 2088150, 2088166, 2088182, 2088197, 2088212, 2088227, 2088274, 2088289, 2088304,
    2088321, 2088336, 2088351, 2089006, 2089022, 2089037, 2089052, 2089069, 2088043, 2088057, 2088259, 2088243,
#Gaurabda Year = 554
2089115, 2089132, 2089147, 2089162, 2089177, 2089194, 2089208, 2089224, 2089239, 2089255, 2089269, 2089287, 2089300, 2089317,
    2089331, 2089349, 2089363, 2090019, 2090034, 2090049, 2090064, 2090083, 2089085, 2089100, 0, 0,
#Gaurabda Year = 555
2090129, 2090145, 2090160, 2090174, 2090191, 2090205, 2090222, 2090236, 2090252, 2090267, 2090283, 2090298, 2090314, 2090329,
    2090344, 2090360, 2090376, 2090392, 2091047, 2091062, 2091077, 2091093, 2090098, 2090113, 0, 0,
#Gaurabda Year = 556
2091142, 2091156, 2091174, 2091188, 2091204, 2091218, 2091236, 2091280, 2091295, 2091311, 2091326, 2091342, 2091357, 2091373,
    2091387, 2091405, 2091419, 2092076, 2092090, 2092106, 2092120, 2092140, 2091111, 2091126, 2091266, 2091250,
#Gaurabda Year = 557
2092185, 2092202, 2092216, 2092232, 2092247, 2092263, 2092279, 2092293, 2092309, 2092324, 2092339, 2092355, 2092371, 2092386,
    2092401, 2092418, 2092433, 2093089, 2093103, 2093118, 2093134, 2093149, 2092154, 2092170, 0, 0,
#Gaurabda Year = 558
2093197, 2093212, 2093228, 2093244, 2093259, 2093274, 2093291, 2093305, 2093321, 2093335, 2093352, 2093366, 2093383, 2093397,
    2093414, 2093428, 2093445, 2093460, 2094116, 2094130, 2094146, 2094161, 2093166, 2093182, 0, 0,
#Gaurabda Year = 559
2094210, 2094226, 2094242, 2094288, 2094302, 2094319, 2094333, 2094349, 2094364, 2094379, 2094395, 2094411, 2094426, 2094441,
    2094457, 2094473, 2094488, 2095143, 2095159, 2095174, 2095189, 2095208, 2094180, 2094195, 2094271, 2094257,
#Gaurabda Year = 560
2095253, 2095270, 2095285, 2095301, 2095315, 2095332, 2095346, 2095363, 2095377, 2095393, 2095408, 2095422, 2095439, 2095454,
    2095470, 2095484, 2095502, 2095516, 2096172, 2096186, 2096203, 2096217, 2095223, 2095239, 0, 0,
#Gaurabda Year = 561
2096267, 2096282, 2096298, 2096313, 2096328, 2096344, 2096360, 2096375, 2096390, 2096406, 2096421, 2096436, 2096452, 2096467,
    2096483, 2096498, 2096515, 2096529, 2097186, 2097200, 2097215, 2097230, 2096236, 2096251, 0, 0,
#Gaurabda Year = 562
2097309, 2097325, 2097340, 2097356, 2097370, 2097388, 2097402, 2097418, 2097432, 2097449, 2097463, 2097480, 2097494, 2097510,
    2097525, 2097542, 2097557, 2098212, 2098227, 2098243, 2098258, 2098276, 2097249, 2097294, 2097278, 2097263,
#Gaurabda Year = 563
2098322, 2098338, 2098354, 2098369, 2098384, 2098401, 2098415, 2098430, 2098446, 2098461, 2098476, 2098491, 2098507, 2098523,
    2098538, 2098553, 2098570, 2098585, 2099240, 2099255, 2099271, 2099286, 2098292, 2098307, 0, 0,
#Gaurabda Year = 564
2099336, 2099350, 2099367, 2099381, 2099398, 2099412, 2099429, 2099443, 2099459, 2099504, 2099521, 2099536, 2099550, 2099567,
    2099581, 2099598, 2099612, 2100269, 2100283, 2100300, 2100314, 2100333, 2099305, 2099319, 2099490, 2099474,
#Gaurabda Year = 565
2100378, 2100395, 2100410, 2100425, 2100441, 2100457, 2100472, 2100487, 2100502, 2100518, 2100533, 2100549, 2100564, 2100580,
    2100594, 2100612, 2100626, 2101282, 2101296, 2101313, 2101327, 2101346, 2100348, 2100363, 0, 0,
#Gaurabda Year = 566
2101390, 2101405, 2101422, 2101437, 2101453, 2101467, 2101484, 2101498, 2101515, 2101529, 2101545, 2101559, 2101577, 2101591,
    2101607, 2101622, 2101639, 2101653, 2102309, 2102324, 2102339, 2102355, 2101360, 2101375, 0, 0,
#Gaurabda Year = 567
2102404, 2102419, 2102435, 2102451, 2102466, 2102512, 2102527, 2102542, 2102558, 2102573, 2102588, 2102604, 2102620, 2102635,
    2102650, 2102666, 2102682, 2103337, 2103352, 2103368, 2103383, 2103402, 2102373, 2102389, 2102497, 2102481,
#Gaurabda Year = 568
2103447, 2103464, 2103478, 2103494, 2103509, 2103526, 2103540, 2103556, 2103571, 2103586, 2103601, 2103618, 2103633, 2103647,
    2103664, 2103678, 2103695, 2103709, 2104366, 2104380, 2104396, 2104411, 2103416, 2103433, 0, 0,
#Gaurabda Year = 569
2104460, 2104475, 2104491, 2104507, 2104522, 2104537, 2104553, 2104569, 2104584, 2104599, 2104614, 2104629, 2104646, 2104661,
    2104677, 2104691, 2104708, 2104723, 2105379, 2105393, 2105410, 2105424, 2104430, 2104445, 0, 0,
#Gaurabda Year = 570
2105473, 2105519, 2105533, 2105550, 2105564, 2105581, 2105595, 2105612, 2105626, 2105642, 2105656, 2105673, 2105688, 2105704,
    2105719, 2105735, 2105750, 2106406, 2106421, 2106436, 2106452, 2106470, 2105442, 2105457, 2105502, 2105487,
#Gaurabda Year = 571
2106516, 2106532, 2106547, 2106563, 2106577, 2106594, 2106609, 2106625, 2106639, 2106654, 2106669, 2106685, 2106701, 2106716,
    2106732, 2106747, 2106763, 2106778, 2107434, 2107449, 2107465, 2107479, 2106485, 2106500, 0, 0
]


def unittests():
    GCUT.info('daydata')
    e = GCEarthData.EARTHDATA()
    e.longitude_deg = 27.0
    e.latitude_deg = 45.0
    e.tzone = 1.0
    GCUT.msg('First day in year: ' + str(GetFirstDayOfYear(e,2020)))
