# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['toascii',
 'toascii.cli',
 'toascii.converters',
 'toascii.converters.extensions']

package_data = \
{'': ['*']}

install_requires = \
['colorama>=0.4.5,<0.5.0',
 'numpy>=1.23.1,<2.0.0',
 'opencv-python>=4.6.0,<5.0.0',
 'pydantic>=1.9.1,<2.0.0']

extras_require = \
{'speedups': ['nimporter>=1.1.0,<2.0.0']}

entry_points = \
{'console_scripts': ['toascii = toascii.cli.cli:main']}

setup_kwargs = {
    'name': 'to-ascii',
    'version': '5.0.0',
    'description': 'Convert videos, images, gifs, and even live video to ASCII art!',
    'long_description': '# To-Ascii ![Code Quality](https://www.codefactor.io/repository/github/iapetus-11/to-ascii/badge/master) ![PYPI Version](https://img.shields.io/pypi/v/to-ascii.svg) ![PYPI Downloads](https://img.shields.io/pypi/dw/to-ascii?color=0FAE6E)\n*Converts videos, images, gifs, and even live video into ascii art!*\n\n* Works on most image and video types including GIFs\n* Works on LIVE VIDEO\n\n<img src="https://user-images.githubusercontent.com/38477514/180253533-e0725ba5-6c6d-408d-a643-ff02f021cff8.png" width="360" /> <img src="https://user-images.githubusercontent.com/38477514/180254306-9e8eca93-ea38-47bf-b1c2-72ad75244604.png" width="360" /> <img src="https://user-images.githubusercontent.com/38477514/180251469-8826a23d-a292-42b2-83c6-c9a637214b5e.png" width="360" /> <img src="https://user-images.githubusercontent.com/38477514/180251666-49b07f5f-da3c-4790-85b9-ba72dbca606b.png" width="360" />\n\n[\\[Video Example\\]](https://www.youtube.com/watch?v=S5-_BzdrOkQ) [\\[Video Example 2\\]](https://www.youtube.com/watch?v=eX4pYQjCyYg)\n\n## Installation\nVia pip:\n```\npip install to-ascii\n```\n\n## CLI Usage\n*Note: Required arguments are surrounded in `<>`, optional arguments are surrounded in `[]`.*\n```\ntoascii <media_type> <source> <converter> [--gradient <string>] [--width <int>] [--height <int>] [--xstretch <float>] [--ystretch <float>] [--saturation <float>] [--contrast <float>] [--loop]\n```\n\n### CLI Arguments\n- `media_type` - the type of media, either `image` or `video`\n- `source` - the source for the media, this is the primary argument in constructing an instance of an [OpenCV `VideoCapture`](https://docs.opencv.org/3.4/d8/dfe/classcv_1_1VideoCapture.html)\n- `converter` - the converter used to convert the media, check [this file](toascii/converters/__init__.py) for the available options\n- `gradient` - the characters used when converting an image to ascii\n- `width` - an integer value for the width in characters of the final converted image\n- `height` - an integer value for the height in characters of the final converted image\n- `xstretch` - the amount to stretch the width by\n- `ystretch` - the amount to stretch the height by\n- `saturation` - how much to saturate the image, a value from -1.0 to 1.0.\n- `contrast` - how much to modify the contrast, a value from 0.0 to 1.0\n- `loop` - whether or not to loop the video when it is done playing\n\n### CLI Examples\n```bash\n# live video\ntoascii video 0 grayscaleconverternim --xstretch 3 --height 56 --saturation .25 --contrast .01\n```\n```bash\ntoascii image "C:\\Users\\miloi\\Pictures\\my_image.png" colorconverter\n```\n```bash\ntoascii video "C:\\Users\\miloi\\Videos\\omni_man.mp4" colorconverternim --height 32 --width 48 --saturation 0.5 --contrast 0.01 --loop\n```\n\n## API Reference\n### [Usage Examples Folder](examples)\n#### *class* [`ConverterOptions`](toascii/converters/options.py)(\\*, `gradient`: *`str`*, `width`: *`Optional[int]`*, `height`: *`Optional[int]`*, `x_stretch`: *`float`*, `y_stretch`: *`float`*, `saturation`: *`float`*, `contrast`: *`Optional[float]`*)\n- *pydantic model for converter options*\n- Parameters / Attributes:\n    - `gradient`: *`str`* - *string containing the characters the converter will use when converting the image to ascii*\n        - must be at least one character\n    - `width`: *`Optional[int]`* - *width in characters of the final converted image*\n        - default value is `None`\n        - must be greater than `0`\n    - `height`: *`Optional[int]`* - *height in characters of the final converted image*\n        - default value is `None`\n        - must be greater than `0`\n    - `x_stretch`: *`float`* - *how much to stretch the width by*\n        - default value is `1.0` (which doesn\'t change the width by anything)\n        - must be greater than `0.0`\n    - `y_stretch`: *`float`* - *how much to stretch the height by*\n        - default value is `1.0` (which doesn\'t change the height by anything)\n        - must be greater than `0.0`\n    - `saturation`: *`float`* - *how much to adjust the saturation*\n        - default value is `0.5` (which increases the saturation)\n        - must be between `-1.0` and `1.0`, `0.0` is no change to saturation\n    - `contrast`: *`Optional[float]`* - *how much to increase the contrast by*\n        - default value is `None` (which doesn\'t apply any contrast filter)\n        - must be between `0.0` and `1.0`\n#### *class* [`ConverterBase`](toascii/converters/base_converter.py)(`options`: *`ConverterOptions`*)\n- *base class for implementing converters*\n- Parameters:\n    - `options`: *`ConverterOptions`* - *Options used when converting media*\n- Methods:\n    - *abstract* `asciify_image`(`image`: *`numpy.ndarray`*) -> *`str`*\n    - `calculate_dimensions`(`initial_height`: *`int`*, `initial_width`: *`int`*) -> *`Tuple[int, int]`*\n    - `resize_image`(`image`: *`numpy.ndarray`*) -> *`numpy.ndarray`*\n- Implementations:\n    - [`GrayscaleConverter`](toascii/converters/grayscale_converter.py) - *converts media to grayscale ascii*\n    - [`GrayscaleConverterNim`](toascii/converters/extensions/grayscale_converter_nim.py) - *converters media to grayscale ascii, see the [Extensions](#extensions) section*\n    - [`ColorConverter`](toascii/converters/color_converter.py) - *converts media to colored ascii using [Colorama](https://pypi.org/project/colorama/)*\n    - [`ColorConverterNim`](toascii/converters/extensions/color_converter_nim.py) - *converts media to colored ascii using [Colorama](https://pypi.org/project/colorama/), see the [Extensions](#extensions) section*\n    - [`HtmlColorConverter`](toascii/converters/html_color_converter.py) - *converts media to ascii in colored html spans*\n    - [`HtmlColorConverterNim`](toascii/converters/extensions/html_color_converter_nim.py) - *converts media to ascii in colored html spans, see the [Extensions](#extensions) section*\n#### *class* [`Image`](toascii/image.py)(`source`: *`Union[str, bytes, IOBase]`*, `converter`: *`BaseConverter`*)\n- *class for converting an image to ascii*\n- Parameters:\n    - `source`: *`Union[str, bytes, IOBase]`* - *the source of the image that is to be loaded and converted*\n        - if `source` is a `str`, it\'s assumed that it\'s a path to an image file\n        - if `source` is `bytes` or `IOBase` it\'s assumed to be the data of an image and is decoded in-memory\n    - `converter`: *`ConverterBase`* - *the converter used to convert the image*\n        - takes anything that implements `ConverterBase`\n- Methods:\n    - `to_ascii`() -> *`str`*\n        - *returns the image converted by the converter*\n    - `view`() -> *`None`*\n        - *prints out the converted image to the console*\n#### *class* [`Video`](toascii/video.py)(`source`: *`Union[str, int, bytes, IOBase]`*, `converter`: *`BaseConverter`*, \\*, `fps`: *`Optional[float]`*, `loop`: *`bool`*)\n- *class for converting a video to ascii*\n- Parameters:\n    - `source`: *`Union[str, int bytes, IOBase]`* - *the source of the video that is to be loaded and converted*\n        - if `source` is a `str`, it\'s assumed that it\'s a path to an image file\n        - if `source` is `bytes` or `IOBase` it\'s assumed to be the data of an image and is written to a temporary file before being loaded and decoded by OpenCV\n        - if `source` is an `int`, it\'s assumed to be the index of a camera device\n        - see [OpenCV\'s `VideoCapture`](https://docs.opencv.org/3.4/d8/dfe/classcv_1_1VideoCapture.html) for more information\n    - `converter`: *`ConverterBase`* - *the converter used to convert the image*\n        - takes anything that implements `ConverterBase`\n    - `fps`: *`Optional[float]`* - *the fps to play the video at*\n        - default value is `None`\n        - if `None` then the fps used is fetched from OpenCV\'s `VideoCapture` API\n    - `loop`: *`bool`* - *whether or not to loop the video when it\'s done playing*\n        - default value is `False`\n        - if the video source is live, this parameter is ignored\n- Methods:\n    - `get_ascii_frames`() -> *`Generator[str, None, None]`* - *returns a generator which yields each ascii frame as it is converted*\n    - `view`() -> *`None`* - *prints out each frame of the converted video*\n        - if the video source is not live, this method will first generate all frames and cache them in memory for a smoother playback\n        - if the `loop` parameter was set to `True` earlier, then this will play the video and restart it when it finishes unless the source is live\n\n## Extensions\n- For each converter available, there is a separate implementation written in [Nim](https://nim-lang.org/)\n- These implementations are generally orders of magnitude faster than their Python counterparts\n- To use these extensions you must [install Nim](https://nim-lang.org/install.html) and install the `to-ascii[speedups]` package via pip or your package manager of choice\n',
    'author': 'Milo Weinberg',
    'author_email': 'iapetus011@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Iapetus-11/To-ASCII',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
