class Core:
    """
    Core class for Netbox API
    """

    def __init__(self, netbox: object, endpoint: str):
        """
        Responsible for creating, getting, updating, and deleting resources in Netbox, obs. this class is not meant to be used directly, but to be inherited by other classes.

        Args:
            netbox: (object) The Netbox object generated by the Client class.
            endpoint: (str) The endpoint to interact with in Netbox.
        """
        self._netbox = netbox
        self._endpoint = endpoint

    def create(self, data) -> dict:
        if isinstance(data, dict):
            if not data['name']:
                return {'status': 400, 'data': 'Name is required.'}

            data['slug'] = self._netbox._slug(data['name'])
        elif isinstance(data, list):
            for item in data:
                item['slug'] = self._netbox._slug(item['name'])
        else:
            return {'status': 400, 'data': 'Invalid data type.'}

        response = self._netbox._request('post', f'{self._endpoint}', data)
        return response

    def get(
        self,
        id: int = None,
        name: str = None,
        tags: list = None,
        search: str = None,
        limit: int = 1000,
    ) -> dict:

        result = {'status': None, 'data': None}
        if id:
            response = self._netbox._request(
                'get', f'{self._endpoint}?id={id}'
            )
            return response

        if not id and name:
            id_name = self._netbox._get_id(name, f'{self._endpoint}')

            if id_name is not None:
                response = self._netbox._request(
                    'get', f'{self._endpoint}?id={id_name}'
                )
                return response
            else:
                result['status'] = 404
                return result

        else:
            if search:
                response = self._netbox._request(
                    'get', f'{self._endpoint}?q={search}&?limit={limit}'
                )
            else:
                response = self._netbox._request(
                    'get', f'{self._endpoint}?limit={limit}'
                )

            if tags:
                if response['status'] != 200:
                    return response

                filtered_resources = []
                slug_tags = [self._netbox._slug(tag) for tag in tags]
                for item in response['data']['results']:
                    if item['tags']:
                        for tag in item['tags']:
                            if tag['slug'] in slug_tags:
                                filtered_resources.append(item)

                return response
            else:
                return response

    def update(self, data: list) -> dict:
        return self._netbox._request('put', f'{self._endpoint}', data)

    def delete(self, id: int) -> dict:
        return self._netbox._request('delete', f'{self._endpoint}{id}/')
