from datetime import date
from typing import Dict, List, Union, Generator

from belvo.resources.base import Resource


class Transactions(Resource):
    endpoint = "/api/transactions/"

    def list(self, link, **kwargs) -> Generator:
        return super().list(link=link, **kwargs)

    def create(
        self,
        link: str,
        date_from: str,
        *,
        date_to: str = None,
        account: str = None,
        token: str = None,
        save_data: bool = True,
        raise_exception: bool = False,
        **kwargs: Dict,
    ) -> Union[List[Dict], Dict]:
        """Retrieve transactions for a link

        <div style="background-color:#f4f6f8; border-left: 6px solid
        #0663F9;padding: 12px;margin-left: 25px; border-radius: 4px; margin-right:
        25px"> <strong>Info: </strong> When retrieving transactions, it is important
        to understand that the available transaction data ranges depend on each
        institution. <br><br>

        If you try to access older information than what we can access, we will
        return all the data we can read within that date range. For example, if you
        request transactions for the last year and we can only access the last six
        months, we will return the information corresponding to these six months of
        data. </div>

        Example:
            ```python
            # Fetch transactions for a Link
            transactions = client.Transactions.create(
                "b91835f5-6f83-4d9b-a0ad-a5a249f18b7c",
                "2019-07-01",
                date_to="2019-07-31"
            )

            # Fetch transactions for a Link with timeout after 15 seconds
            transactions = client.Transactions.create(
                "b91835f5-6f83-4d9b-a0ad-a5a249f18b7c",
                "2019-07-01",
                date_to="2019-07-31",    
                timeout=15
            )

            ```

        Args:
            link (str): The `link.id` that you want to get information for (UUID).
            date_from (str): The date from which you want to start getting transactions for, in `YYYY-MM-DD` format. The value of `date_from` cannot be greater than `date_to`.
            date_to (str, optional): The date you want to stop getting transactions for, in `YYYY-MM-DD` format. The value of `date_to` cannot be greater than today's date (in other words, no future dates). Defaults to today.
            account (str, optional): If provided, we return transactions only from this account (UUID). Defaults to None.
            token (str, optional): The MFA token generated by the bank in order to access the institution. Defaults to None.
            save_data (bool, optional): Indicates whether or not to persist the data in Belvo. Defaults to `True`.
            raise_exception (bool, optional): Indicates whether to raise an exception or return the API error. Defaults to `False`.

        Returns:
            Dict: The details of the object. For more information on the response from the API, see our [Transactions API documentation](https://developers.belvo.com/reference/retrievetransactions).
        """

        date_to = date_to or date.today().isoformat()

        data = {"link": link, "date_from": date_from, "date_to": date_to, "save_data": save_data}

        if account:
            data.update(account=account)
        if token:
            data.update(token=token)

        return self.session.post(
            self.endpoint, data=data, raise_exception=raise_exception, **kwargs
        )
