from typing import Dict, List, Union

from belvo.resources.base import Resource


class RiskInsights(Resource):
    endpoint = "/api/risk-insights/"

    def create(
        self,
        link: str,
        *,
        token: str = None,
        save_data: bool = True,
        raise_exception: bool = False,
        **kwargs: Dict,
    ) -> Union[List[Dict], Dict]:
        """Retrieve risk insights for a link

        Example:
            ```python
            # Fetch risk insights for a Link
            risk_insights = client.RiskInsights.create("44d309dc-24c6-4734-99e0-22c595fee2c2")

            # Fetch risk insights for a Link with and timeout after 15 seconds
            risk_insights = client.RiskInsights.create(
                "44d309dc-24c6-4734-99e0-22c595fee2c2",
                timeout=15
            )

            ```

        Args:
            link (str): The `link.id` that you want to get information for (UUID).
            token (str, optional): The MFA token generated by the bank in order to access the institution. Defaults to None.
            save_data (bool, optional): Indicates whether or not to persist the data in Belvo. Defaults to `True`.
            raise_exception (bool, optional): Indicates whether to raise an exception or return the API error. Defaults to `False`.

        Returns:
            Dict: The details of the object. For more information on the response from the API, see our [Risk insights API documentation](https://developers.belvo.com/reference/retrieveriskinsights).
        """

        data = {"link": link, "save_data": save_data}

        if token:
            data.update(token=token)

        return self.session.post(
            self.endpoint, data=data, raise_exception=raise_exception, **kwargs
        )
