"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'peering between two security groups does not recursive infinitely'(test) {
        // GIVEN
        const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '12345678', region: 'dummy' } });
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SG1', { vpc });
        const sg2 = new lib_1.SecurityGroup(stack, 'SG2', { vpc });
        const conn1 = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const conn2 = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg2] }));
        // WHEN
        conn1.connections.allowTo(conn2, lib_1.Port.tcp(80), 'Test');
        // THEN -- it finishes!
        test.done();
    },
    '(imported) SecurityGroup can be used as target of .allowTo()'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        // WHEN
        somethingConnectable.connections.allowTo(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            IpProtocol: 'tcp',
            Description: 'Connect there',
            DestinationSecurityGroupId: 'sg-12345',
            FromPort: 0,
            ToPort: 65535,
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            Description: 'Connect there',
            FromPort: 0,
            GroupId: 'sg-12345',
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            ToPort: 65535,
        }));
        test.done();
    },
    'security groups added to connections after rule still gets rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const connections = new lib_1.Connections({ securityGroups: [sg1] });
        // WHEN
        connections.allowFromAnyIpv4(lib_1.Port.tcp(88));
        connections.addSecurityGroup(sg2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'SecurityGroup1',
            SecurityGroupIngress: [
                {
                    Description: 'from 0.0.0.0/0:88',
                    CidrIp: '0.0.0.0/0',
                    FromPort: 88,
                    ToPort: 88,
                    IpProtocol: 'tcp',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'SecurityGroup2',
            SecurityGroupIngress: [
                {
                    Description: 'from 0.0.0.0/0:88',
                    CidrIp: '0.0.0.0/0',
                    FromPort: 88,
                    ToPort: 88,
                    IpProtocol: 'tcp',
                },
            ],
        }));
        test.done();
    },
    'when security groups are added to target they also get the rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const sg3 = new lib_1.SecurityGroup(stack, 'SecurityGroup3', { vpc, allowAllOutbound: false });
        const connections1 = new lib_1.Connections({ securityGroups: [sg1] });
        const connections2 = new lib_1.Connections({ securityGroups: [sg2] });
        const connectable = new SomethingConnectable(connections2);
        // WHEN
        connections1.allowTo(connectable, lib_1.Port.tcp(88));
        connections2.addSecurityGroup(sg3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroup23BE86BB7', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroup3E5E374B9', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        test.done();
    },
    'multiple security groups allows internally between them'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const connections = new lib_1.Connections({ securityGroups: [sg1] });
        // WHEN
        connections.allowInternally(lib_1.Port.tcp(88));
        connections.addSecurityGroup(sg2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            GroupId: { 'Fn::GetAtt': ['SecurityGroup1F554B36F', 'GroupId'] },
            FromPort: 88,
            ToPort: 88,
        }));
        test.done();
    },
    'can establish cross stack Security Group connections - allowFrom'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack1, 'SecurityGroup', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowFrom(sg1, lib_1.Port.tcp(100));
        // THEN -- both rules are in Stack2
        core_1.ConstructNode.prepare(app.node);
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
            SourceSecurityGroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        test.done();
    },
    'can establish cross stack Security Group connections - allowTo'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack1, 'SecurityGroup', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowTo(sg1, lib_1.Port.tcp(100));
        // THEN -- both rules are in Stack2
        core_1.ConstructNode.prepare(app.node);
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
            SourceSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
            DestinationSecurityGroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09' },
        }));
        test.done();
    },
    'can establish multiple cross-stack SGs'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1a = new lib_1.SecurityGroup(stack1, 'SecurityGroupA', { vpc: vpc1, allowAllOutbound: false });
        const sg1b = new lib_1.SecurityGroup(stack1, 'SecurityGroupB', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowFrom(sg1a, lib_1.Port.tcp(100));
        sg2.connections.allowFrom(sg1b, lib_1.Port.tcp(100));
        // THEN -- both egress rules are in Stack2
        core_1.ConstructNode.prepare(app.node);
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupAED40ADC5GroupId1D10C76A' },
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { 'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttSecurityGroupB04591F90GroupIdFA7208D5' },
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
        }));
        test.done();
    },
    'Imported SecurityGroup does not create egress rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        // WHEN
        somethingConnectable.connections.allowFrom(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            IpProtocol: 'tcp',
            Description: 'Connect there',
            SourceSecurityGroupId: 'sg-12345',
            FromPort: 0,
            ToPort: 65535,
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroupEgress'));
        test.done();
    },
    'Imported SecurityGroup with allowAllOutbound: false DOES create egress rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345', {
            allowAllOutbound: false,
        });
        // WHEN
        somethingConnectable.connections.allowFrom(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            IpProtocol: 'tcp',
            Description: 'Connect there',
            SourceSecurityGroupId: 'sg-12345',
            FromPort: 0,
            ToPort: 65535,
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            IpProtocol: 'tcp',
            Description: 'Connect there',
            FromPort: 0,
            GroupId: 'sg-12345',
            DestinationSecurityGroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
            ToPort: 65535,
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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