"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        const subnets = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        const subnetIds = subnets.subnetIds;
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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