"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
/**
 * Peer object factories (to be used in Security Group management)
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 */
class Peer {
    /**
     * Create an IPv4 peer from a CIDR
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * Any IPv4 address
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * Create an IPv6 peer from a CIDR
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * Any IPv6 address
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * A prefix list
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
    constructor() {
    }
}
exports.Peer = Peer;
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super('0.0.0.0/0');
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super('::/0');
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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