"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ssm = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage {
    constructor(version, props = {}) {
        this.version = version;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const parameterName = this.imageParameterName();
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    imageParameterName() {
        return '/aws/service/ami-windows-latest/' + this.version;
    }
}
exports.WindowsImage = WindowsImage;
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage {
    constructor(props = {}) {
        this.props = props;
        this.generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        this.edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        this.virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        this.storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const parts = [
            this.generation,
            'ami',
            this.edition !== AmazonLinuxEdition.STANDARD ? this.edition : undefined,
            this.virtualization,
            'x86_64',
            this.storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        var _a;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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