"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceType = exports.ManagedRuleIdentifiers = exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = exports.RuleScope = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines an EventBridge event rule which triggers for rule events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines an EventBridge event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines an EventBridge event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule.
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
}
/**
 * Determines which resources trigger an evaluation of an AWS Config rule.
 */
class RuleScope {
    constructor(resourceId, resourceTypes, tagKey, tagValue) {
        this.resourceTypes = resourceTypes;
        this.resourceId = resourceId;
        this.key = tagKey;
        this.value = tagValue;
    }
    /**
     * restricts scope of changes to a specific resource type or resource identifier.
     */
    static fromResource(resourceType, resourceId) {
        return new RuleScope(resourceId, [resourceType]);
    }
    /**
     * restricts scope of changes to specific resource types.
     */
    static fromResources(resourceTypes) {
        return new RuleScope(undefined, resourceTypes);
    }
    /**
     * restricts scope of changes to a specific tag.
     */
    static fromTag(key, value) {
        return new RuleScope(undefined, undefined, key, value);
    }
}
exports.RuleScope = RuleScope;
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        this.ruleScope = props.ruleScope;
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        this.ruleScope = props.ruleScope;
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => renderScope(this.ruleScope) }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
/**
 * Managed rules that are supported by AWS Config.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html
 */
class ManagedRuleIdentifiers {
    // utility class
    constructor() { }
}
exports.ManagedRuleIdentifiers = ManagedRuleIdentifiers;
/**
 * Checks that the inline policies attached to your AWS Identity and Access Management users, roles, and groups do not allow blocked actions on all AWS Key Management Service keys.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-inline-policy-blocked-kms-actions.html
 */
ManagedRuleIdentifiers.IAM_INLINE_POLICY_BLOCKED_KMS_ACTIONS = 'IAM_INLINE_POLICY_BLOCKED_KMS_ACTIONS';
/**
 * Checks that the managed AWS Identity and Access Management policies that you create do not allow blocked actions on all AWS AWS KMS keys.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-customer-policy-blocked-kms-actions.html
 */
ManagedRuleIdentifiers.IAM_CUSTOMER_POLICY_BLOCKED_KMS_ACTIONS = 'IAM_CUSTOMER_POLICY_BLOCKED_KMS_ACTIONS';
/**
 * Checks whether the active access keys are rotated within the number of days specified in maxAccessKeyAge.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 */
ManagedRuleIdentifiers.ACCESS_KEYS_ROTATED = 'ACCESS_KEYS_ROTATED';
/**
 * Checks whether AWS account is part of AWS Organizations.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/account-part-of-organizations.html
 */
ManagedRuleIdentifiers.ACCOUNT_PART_OF_ORGANIZATIONS = 'ACCOUNT_PART_OF_ORGANIZATIONS';
/**
 * Checks whether ACM Certificates in your account are marked for expiration within the specified number of days.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/acm-certificate-expiration-check.html
 */
ManagedRuleIdentifiers.ACM_CERTIFICATE_EXPIRATION_CHECK = 'ACM_CERTIFICATE_EXPIRATION_CHECK';
/**
 * Checks if rule evaluates Application Load Balancers (ALBs) to ensure they are configured to drop http headers.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-http-drop-invalid-header-enabled.html
 */
ManagedRuleIdentifiers.ALB_HTTP_DROP_INVALID_HEADER_ENABLED = 'ALB_HTTP_DROP_INVALID_HEADER_ENABLED';
/**
 * Checks whether HTTP to HTTPS redirection is configured on all HTTP listeners of Application Load Balancer.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-http-to-https-redirection-check.html
 */
ManagedRuleIdentifiers.ALB_HTTP_TO_HTTPS_REDIRECTION_CHECK = 'ALB_HTTP_TO_HTTPS_REDIRECTION_CHECK';
/**
 * Checks if Web Application Firewall (WAF) is enabled on Application Load Balancers (ALBs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/alb-waf-enabled.html
 */
ManagedRuleIdentifiers.ALB_WAF_ENABLED = 'ALB_WAF_ENABLED';
/**
 * Checks that all methods in Amazon API Gateway stages have caching enabled and encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-cache-enabled-and-encrypted.html
 */
ManagedRuleIdentifiers.API_GW_CACHE_ENABLED_AND_ENCRYPTED = 'API_GW_CACHE_ENABLED_AND_ENCRYPTED';
/**
 * Checks that Amazon API Gateway APIs are of the type specified in the rule parameter endpointConfigurationType.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-endpoint-type-check.html
 */
ManagedRuleIdentifiers.API_GW_ENDPOINT_TYPE_CHECK = 'API_GW_ENDPOINT_TYPE_CHECK';
/**
 * Checks that all methods in Amazon API Gateway stage has logging enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/api-gw-execution-logging-enabled.html
 */
ManagedRuleIdentifiers.API_GW_EXECUTION_LOGGING_ENABLED = 'API_GW_EXECUTION_LOGGING_ENABLED';
/**
 * Checks whether running instances are using specified AMIs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/approved-amis-by-id.html
 */
ManagedRuleIdentifiers.APPROVED_AMIS_BY_ID = 'APPROVED_AMIS_BY_ID';
/**
 * Checks whether running instances are using specified AMIs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/approved-amis-by-tag.html
 */
ManagedRuleIdentifiers.APPROVED_AMIS_BY_TAG = 'APPROVED_AMIS_BY_TAG';
/**
 * Checks whether your Auto Scaling groups that are associated with a load balancer are using Elastic Load Balancing health checks.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/autoscaling-group-elb-healthcheck-required.html
 */
ManagedRuleIdentifiers.AUTOSCALING_GROUP_ELB_HEALTHCHECK_REQUIRED = 'AUTOSCALING_GROUP_ELB_HEALTHCHECK_REQUIRED';
/**
 * Checks whether an AWS CloudFormation stack's actual configuration differs, or has drifted, from it's expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 */
ManagedRuleIdentifiers.CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK = 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK';
/**
 * Checks whether your CloudFormation stacks are sending event notifications to an SNS topic.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 */
ManagedRuleIdentifiers.CLOUDFORMATION_STACK_NOTIFICATION_CHECK = 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK';
/**
 * Checks if an Amazon CloudFront distribution is configured to return a specific object that is the default root object.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-default-root-object-configured.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_DEFAULT_ROOT_OBJECT_CONFIGURED = 'CLOUDFRONT_DEFAULT_ROOT_OBJECT_CONFIGURED';
/**
 * Checks that Amazon CloudFront distribution with Amazon S3 Origin type has Origin Access Identity (OAI) configured.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-origin-access-identity-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_ORIGIN_ACCESS_IDENTITY_ENABLED = 'CLOUDFRONT_ORIGIN_ACCESS_IDENTITY_ENABLED';
/**
 * Checks whether an origin group is configured for the distribution of at least 2 origins in the origin group for Amazon CloudFront.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-origin-failover-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_ORIGIN_FAILOVER_ENABLED = 'CLOUDFRONT_ORIGIN_FAILOVER_ENABLED';
/**
 * Checks if Amazon CloudFront distributions are using a custom SSL certificate and are configured to use SNI to serve HTTPS requests.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-sni-enabled.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_SNI_ENABLED = 'CLOUDFRONT_SNI_ENABLED';
/**
 * Checks whether your Amazon CloudFront distributions use HTTPS (directly or via a redirection).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudfront-viewer-policy-https.html
 */
ManagedRuleIdentifiers.CLOUDFRONT_VIEWER_POLICY_HTTPS = 'CLOUDFRONT_VIEWER_POLICY_HTTPS';
/**
 * Checks whether AWS CloudTrail trails are configured to send logs to Amazon CloudWatch Logs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-cloud-watch-logs-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_CLOUD_WATCH_LOGS_ENABLED = 'CLOUD_TRAIL_CLOUD_WATCH_LOGS_ENABLED';
/**
 * Checks whether AWS CloudTrail is enabled in your AWS account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_ENABLED = 'CLOUD_TRAIL_ENABLED';
/**
 * Checks whether AWS CloudTrail is configured to use the server side encryption (SSE) AWS Key Management Service (AWS KMS) customer master key (CMK) encryption.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-encryption-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_ENCRYPTION_ENABLED = 'CLOUD_TRAIL_ENCRYPTION_ENABLED';
/**
 * Checks whether AWS CloudTrail creates a signed digest file with logs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloud-trail-log-file-validation-enabled.html
 */
ManagedRuleIdentifiers.CLOUD_TRAIL_LOG_FILE_VALIDATION_ENABLED = 'CLOUD_TRAIL_LOG_FILE_VALIDATION_ENABLED';
/**
 * Checks whether at least one AWS CloudTrail trail is logging Amazon S3 data events for all S3 buckets.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-s3-dataevents-enabled.html
 */
ManagedRuleIdentifiers.CLOUDTRAIL_S3_DATAEVENTS_ENABLED = 'CLOUDTRAIL_S3_DATAEVENTS_ENABLED';
/**
 * Checks that there is at least one AWS CloudTrail trail defined with security best practices.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudtrail-security-trail-enabled.html
 */
ManagedRuleIdentifiers.CLOUDTRAIL_SECURITY_TRAIL_ENABLED = 'CLOUDTRAIL_SECURITY_TRAIL_ENABLED';
/**
 * Checks whether CloudWatch alarms have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-action-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_ACTION_CHECK = 'CLOUDWATCH_ALARM_ACTION_CHECK';
/**
 * Checks whether the specified resource type has a CloudWatch alarm for the specified metric.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-resource-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_RESOURCE_CHECK = 'CLOUDWATCH_ALARM_RESOURCE_CHECK';
/**
 * Checks whether CloudWatch alarms with the given metric name have the specified settings.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-alarm-settings-check.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_ALARM_SETTINGS_CHECK = 'CLOUDWATCH_ALARM_SETTINGS_CHECK';
/**
 * Checks whether a log group in Amazon CloudWatch Logs is encrypted with a AWS Key Management Service (KMS) managed Customer Master Keys (CMK).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudwatch-log-group-encrypted.html
 */
ManagedRuleIdentifiers.CLOUDWATCH_LOG_GROUP_ENCRYPTED = 'CLOUDWATCH_LOG_GROUP_ENCRYPTED';
/**
 * Checks that key rotation is enabled for each key and matches to the key ID of the customer created customer master key (CMK).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cmk-backing-key-rotation-enabled.html
 */
ManagedRuleIdentifiers.CMK_BACKING_KEY_ROTATION_ENABLED = 'CMK_BACKING_KEY_ROTATION_ENABLED';
/**
 * Checks whether the project contains environment variables AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-envvar-awscred-check.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_ENVVAR_AWSCRED_CHECK = 'CODEBUILD_PROJECT_ENVVAR_AWSCRED_CHECK';
/**
 * Checks whether the GitHub or Bitbucket source repository URL contains either personal access tokens or user name and password.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codebuild-project-source-repo-url-check.html
 */
ManagedRuleIdentifiers.CODEBUILD_PROJECT_SOURCE_REPO_URL_CHECK = 'CODEBUILD_PROJECT_SOURCE_REPO_URL_CHECK';
/**
 * Checks whether the first deployment stage of the AWS CodePipeline performs more than one deployment.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codepipeline-deployment-count-check.html
 */
ManagedRuleIdentifiers.CODEPIPELINE_DEPLOYMENT_COUNT_CHECK = 'CODEPIPELINE_DEPLOYMENT_COUNT_CHECK';
/**
 * Checks whether each stage in the AWS CodePipeline deploys to more than N times the number of the regions the AWS CodePipeline has deployed in all the previous combined stages, where N is the region fanout number.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/codepipeline-region-fanout-check.html
 */
ManagedRuleIdentifiers.CODEPIPELINE_REGION_FANOUT_CHECK = 'CODEPIPELINE_REGION_FANOUT_CHECK';
/**
 * Checks whether Amazon CloudWatch LogGroup retention period is set to specific number of days.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cw-loggroup-retention-period-check.html
 */
ManagedRuleIdentifiers.CW_LOGGROUP_RETENTION_PERIOD_CHECK = 'CW_LOGGROUP_RETENTION_PERIOD_CHECK';
/**
 * Checks that DynamoDB Accelerator (DAX) clusters are encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dax-encryption-enabled.html
 */
ManagedRuleIdentifiers.DAX_ENCRYPTION_ENABLED = 'DAX_ENCRYPTION_ENABLED';
/**
 * Checks whether RDS DB instances have backups enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/db-instance-backup-enabled.html
 */
ManagedRuleIdentifiers.RDS_DB_INSTANCE_BACKUP_ENABLED = 'DB_INSTANCE_BACKUP_ENABLED';
/**
 * Checks instances for specified tenancy.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/desired-instance-tenancy.html
 */
ManagedRuleIdentifiers.EC2_DESIRED_INSTANCE_TENANCY = 'DESIRED_INSTANCE_TENANCY';
/**
 * Checks whether your EC2 instances are of the specified instance types.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/desired-instance-type.html
 */
ManagedRuleIdentifiers.EC2_DESIRED_INSTANCE_TYPE = 'DESIRED_INSTANCE_TYPE';
/**
 * Checks whether AWS Database Migration Service replication instances are public.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dms-replication-not-public.html
 */
ManagedRuleIdentifiers.DMS_REPLICATION_NOT_PUBLIC = 'DMS_REPLICATION_NOT_PUBLIC';
/**
 * Checks whether Auto Scaling or On-Demand is enabled on your DynamoDB tables and/or global secondary indexes.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-autoscaling-enabled.html
 */
ManagedRuleIdentifiers.DYNAMODB_AUTOSCALING_ENABLED = 'DYNAMODB_AUTOSCALING_ENABLED';
/**
 * Checks whether Amazon DynamoDB table is present in AWS Backup plans.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-in-backup-plan.html
 */
ManagedRuleIdentifiers.DYNAMODB_IN_BACKUP_PLAN = 'DYNAMODB_IN_BACKUP_PLAN';
/**
 * Checks that point in time recovery (PITR) is enabled for Amazon DynamoDB tables.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-pitr-enabled.html
 */
ManagedRuleIdentifiers.DYNAMODB_PITR_ENABLED = 'DYNAMODB_PITR_ENABLED';
/**
 * Checks whether Amazon DynamoDB table is encrypted with AWS Key Management Service (KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encrypted-kms.html
 */
ManagedRuleIdentifiers.DYNAMODB_TABLE_ENCRYPTED_KMS = 'DYNAMODB_TABLE_ENCRYPTED_KMS';
/**
 * Checks whether the Amazon DynamoDB tables are encrypted and checks their status.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-table-encryption-enabled.html
 */
ManagedRuleIdentifiers.DYNAMODB_TABLE_ENCRYPTION_ENABLED = 'DYNAMODB_TABLE_ENCRYPTION_ENABLED';
/**
 * Checks whether provisioned DynamoDB throughput is approaching the maximum limit for your account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/dynamodb-throughput-limit-check.html
 */
ManagedRuleIdentifiers.DYNAMODB_THROUGHPUT_LIMIT_CHECK = 'DYNAMODB_THROUGHPUT_LIMIT_CHECK';
/**
 * Checks if Amazon Elastic Block Store (Amazon EBS) volumes are added in backup plans of AWS Backup.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-in-backup-plan.html
 */
ManagedRuleIdentifiers.EBS_IN_BACKUP_PLAN = 'EBS_IN_BACKUP_PLAN';
/**
 * Checks whether Amazon Elastic File System (Amazon EFS) file systems are added in the backup plans of AWS Backup.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-in-backup-plan.html
 */
ManagedRuleIdentifiers.EFS_IN_BACKUP_PLAN = 'EFS_IN_BACKUP_PLAN';
/**
 * Check that Amazon Elastic Block Store (EBS) encryption is enabled by default.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-ebs-encryption-by-default.html
 */
ManagedRuleIdentifiers.EC2_EBS_ENCRYPTION_BY_DEFAULT = 'EC2_EBS_ENCRYPTION_BY_DEFAULT';
/**
 * Checks whether EBS optimization is enabled for your EC2 instances that can be EBS-optimized.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-optimized-instance.html
 */
ManagedRuleIdentifiers.EBS_OPTIMIZED_INSTANCE = 'EBS_OPTIMIZED_INSTANCE';
/**
 * Checks whether Amazon Elastic Block Store snapshots are not publicly restorable.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ebs-snapshot-public-restorable-check.html
 */
ManagedRuleIdentifiers.EBS_SNAPSHOT_PUBLIC_RESTORABLE_CHECK = 'EBS_SNAPSHOT_PUBLIC_RESTORABLE_CHECK';
/**
 * Checks whether detailed monitoring is enabled for EC2 instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-detailed-monitoring-enabled.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_DETAILED_MONITORING_ENABLED = 'EC2_INSTANCE_DETAILED_MONITORING_ENABLED';
/**
 * Checks whether the Amazon EC2 instances in your account are managed by AWS Systems Manager.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-managed-by-systems-manager.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_MANAGED_BY_SSM = 'EC2_INSTANCE_MANAGED_BY_SSM';
/**
 * Checks whether Amazon Elastic Compute Cloud (Amazon EC2) instances have a public IP association.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instance-no-public-ip.html
 */
ManagedRuleIdentifiers.EC2_INSTANCE_NO_PUBLIC_IP = 'EC2_INSTANCE_NO_PUBLIC_IP';
/**
 * Checks whether your EC2 instances belong to a virtual private cloud (VPC).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-instances-in-vpc.html
 */
ManagedRuleIdentifiers.EC2_INSTANCES_IN_VPC = 'INSTANCES_IN_VPC';
/**
 * Checks that none of the specified applications are installed on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-applications-blacklisted.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_APPLICATIONS_BLOCKED = 'EC2_MANAGEDINSTANCE_APPLICATIONS_BLACKLISTED';
/**
 * Checks whether all of the specified applications are installed on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-applications-required.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_APPLICATIONS_REQUIRED = 'EC2_MANAGEDINSTANCE_APPLICATIONS_REQUIRED';
/**
 * Checks whether the compliance status of AWS Systems Manager association compliance is COMPLIANT or NON_COMPLIANT after the association execution on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-association-compliance-status-check.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_ASSOCIATION_COMPLIANCE_STATUS_CHECK = 'EC2_MANAGEDINSTANCE_ASSOCIATION_COMPLIANCE_STATUS_CHECK';
/**
 * Checks whether instances managed by AWS Systems Manager are configured to collect blocked inventory types.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-inventory-blacklisted.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_INVENTORY_BLOCKED = 'EC2_MANAGEDINSTANCE_INVENTORY_BLACKLISTED';
/**
 * Checks whether the compliance status of the Amazon EC2 Systems Manager patch compliance is COMPLIANT or NON_COMPLIANT after the patch installation on the instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-patch-compliance-status-check.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_PATCH_COMPLIANCE_STATUS_CHECK = 'EC2_MANAGEDINSTANCE_PATCH_COMPLIANCE_STATUS_CHECK';
/**
 * Checks whether EC2 managed instances have the desired configurations.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-managedinstance-platform-check.html
 */
ManagedRuleIdentifiers.EC2_MANAGED_INSTANCE_PLATFORM_CHECK = 'EC2_MANAGEDINSTANCE_PLATFORM_CHECK';
/**
 * Checks that security groups are attached to Amazon Elastic Compute Cloud (Amazon EC2) instances or to an elastic network interface.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-security-group-attached-to-eni.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUP_ATTACHED_TO_ENI = 'EC2_SECURITY_GROUP_ATTACHED_TO_ENI';
/**
 * Checks whether there are instances stopped for more than the allowed number of days.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-stopped-instance.html
 */
ManagedRuleIdentifiers.EC2_STOPPED_INSTANCE = 'EC2_STOPPED_INSTANCE';
/**
 * Checks whether EBS volumes are attached to EC2 instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-volume-inuse-check.html
 */
ManagedRuleIdentifiers.EC2_VOLUME_INUSE_CHECK = 'EC2_VOLUME_INUSE_CHECK';
/**
 * hecks whether Amazon Elastic File System (Amazon EFS) is configured to encrypt the file data using AWS Key Management Service (AWS KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/efs-encrypted-check.html
 */
ManagedRuleIdentifiers.EFS_ENCRYPTED_CHECK = 'EFS_ENCRYPTED_CHECK';
/**
 * Checks whether all Elastic IP addresses that are allocated to a VPC are attached to EC2 instances or in-use elastic network interfaces (ENIs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eip-attached.html
 */
ManagedRuleIdentifiers.EIP_ATTACHED = 'EIP_ATTACHED';
/**
 * Checks whether Amazon Elasticsearch Service (Amazon ES) domains have encryption at rest configuration enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-encrypted-at-rest.html
 */
ManagedRuleIdentifiers.ELASTICSEARCH_ENCRYPTED_AT_REST = 'ELASTICSEARCH_ENCRYPTED_AT_REST';
/**
 * Checks whether Amazon Elasticsearch Service (Amazon ES) domains are in Amazon Virtual Private Cloud (Amazon VPC).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-in-vpc-only.html
 */
ManagedRuleIdentifiers.ELASTICSEARCH_IN_VPC_ONLY = 'ELASTICSEARCH_IN_VPC_ONLY';
/**
 * Check if the Amazon ElastiCache Redis clusters have automatic backup turned on.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticache-redis-cluster-automatic-backup-check.html
 */
ManagedRuleIdentifiers.ELASTICACHE_REDIS_CLUSTER_AUTOMATIC_BACKUP_CHECK = 'ELASTICACHE_REDIS_CLUSTER_AUTOMATIC_BACKUP_CHECK';
/**
 * Checks whether your Amazon Elastic Compute Cloud (Amazon EC2) instance metadata version is configured with Instance Metadata Service Version 2 (IMDSv2).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/ec2-imdsv2-check.html
 */
ManagedRuleIdentifiers.EC2_IMDSV2_CHECK = 'EC2_IMDSV2_CHECK';
/**
 * Checks whether Amazon Elastic Kubernetes Service (Amazon EKS) endpoint is not publicly accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-endpoint-no-public-access.html
 */
ManagedRuleIdentifiers.EKS_ENDPOINT_NO_PUBLIC_ACCESS = 'EKS_ENDPOINT_NO_PUBLIC_ACCESS';
/**
 * Checks whether Amazon Elastic Kubernetes Service clusters are configured to have Kubernetes secrets encrypted using AWS Key Management Service (KMS) keys.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/eks-secrets-encrypted.html
 */
ManagedRuleIdentifiers.EKS_SECRETS_ENCRYPTED = 'EKS_SECRETS_ENCRYPTED';
/**
 * Check that Amazon ElasticSearch Service nodes are encrypted end to end.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elasticsearch-node-to-node-encryption-check.html
 */
ManagedRuleIdentifiers.ELASTICSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK = 'ELASTICSEARCH_NODE_TO_NODE_ENCRYPTION_CHECK';
/**
 * Checks if cross-zone load balancing is enabled for the Classic Load Balancers (CLBs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-cross-zone-load-balancing-enabled.html
 */
ManagedRuleIdentifiers.ELB_CROSS_ZONE_LOAD_BALANCING_ENABLED = 'ELB_CROSS_ZONE_LOAD_BALANCING_ENABLED';
/**
 * Checks whether your Classic Load Balancer is configured with SSL or HTTPS listeners.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-tls-https-listeners-only.html
 */
ManagedRuleIdentifiers.ELB_TLS_HTTPS_LISTENERS_ONLY = 'ELB_TLS_HTTPS_LISTENERS_ONLY';
/**
 * Checks whether the Classic Load Balancers use SSL certificates provided by AWS Certificate Manager.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-acm-certificate-required.html
 */
ManagedRuleIdentifiers.ELB_ACM_CERTIFICATE_REQUIRED = 'ELB_ACM_CERTIFICATE_REQUIRED';
/**
 * Checks whether your Classic Load Balancer SSL listeners are using a custom policy.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-custom-security-policy-ssl-check.html
 */
ManagedRuleIdentifiers.ELB_CUSTOM_SECURITY_POLICY_SSL_CHECK = 'ELB_CUSTOM_SECURITY_POLICY_SSL_CHECK';
/**
 * Checks whether Elastic Load Balancing has deletion protection enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-deletion-protection-enabled.html
 */
ManagedRuleIdentifiers.ELB_DELETION_PROTECTION_ENABLED = 'ELB_DELETION_PROTECTION_ENABLED';
/**
 * Checks whether the Application Load Balancer and the Classic Load Balancer have logging enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-logging-enabled.html
 */
ManagedRuleIdentifiers.ELB_LOGGING_ENABLED = 'ELB_LOGGING_ENABLED';
/**
 * Checks whether your Classic Load Balancer SSL listeners are using a predefined policy.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/elb-predefined-security-policy-ssl-check.html
 */
ManagedRuleIdentifiers.ELB_PREDEFINED_SECURITY_POLICY_SSL_CHECK = 'ELB_PREDEFINED_SECURITY_POLICY_SSL_CHECK';
/**
 * Checks that Amazon EMR clusters have Kerberos enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/emr-kerberos-enabled.html
 */
ManagedRuleIdentifiers.EMR_KERBEROS_ENABLED = 'EMR_KERBEROS_ENABLED';
/**
 * Checks whether Amazon Elastic MapReduce (EMR) clusters' master nodes have public IPs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/emr-master-no-public-ip.html
 */
ManagedRuleIdentifiers.EMR_MASTER_NO_PUBLIC_IP = 'EMR_MASTER_NO_PUBLIC_IP';
/**
 * Checks whether the EBS volumes that are in an attached state are encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/encrypted-volumes.html
 */
ManagedRuleIdentifiers.EBS_ENCRYPTED_VOLUMES = 'ENCRYPTED_VOLUMES';
/**
 * Checks whether the security groups associated inScope resources are compliant with the master security groups at each rule level based on allowSecurityGroup and denySecurityGroup flag.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-audit-policy-check.html
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_AUDIT_POLICY_CHECK = 'FMS_SECURITY_GROUP_AUDIT_POLICY_CHECK';
/**
 * Checks whether AWS Firewall Manager created security groups content is the same as the master security groups.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-content-check.html
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_CONTENT_CHECK = 'FMS_SECURITY_GROUP_CONTENT_CHECK';
/**
 * Checks whether Amazon EC2 or an elastic network interface is associated with AWS Firewall Manager security groups.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-security-group-resource-association-check.html
 */
ManagedRuleIdentifiers.FMS_SECURITY_GROUP_RESOURCE_ASSOCIATION_CHECK = 'FMS_SECURITY_GROUP_RESOURCE_ASSOCIATION_CHECK';
/**
 * Checks whether an Application Load Balancer, Amazon CloudFront distributions, Elastic Load Balancer or Elastic IP has AWS Shield protection.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-shield-resource-policy-check.html
 */
ManagedRuleIdentifiers.FMS_SHIELD_RESOURCE_POLICY_CHECK = 'FMS_SHIELD_RESOURCE_POLICY_CHECK';
/**
 * Checks whether the web ACL is associated with an Application Load Balancer, API Gateway stage, or Amazon CloudFront distributions.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-webacl-resource-policy-check.html
 */
ManagedRuleIdentifiers.FMS_WEBACL_RESOURCE_POLICY_CHECK = 'FMS_WEBACL_RESOURCE_POLICY_CHECK';
/**
 * Checks that the rule groups associate with the web ACL at the correct priority.
 *
 * The correct priority is decided by the rank of the rule groups in the ruleGroups parameter.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/fms-webacl-rulegroup-association-check.html
 */
ManagedRuleIdentifiers.FMS_WEBACL_RULEGROUP_ASSOCIATION_CHECK = 'FMS_WEBACL_RULEGROUP_ASSOCIATION_CHECK';
/**
 * Checks whether Amazon GuardDuty is enabled in your AWS account and region.
 *
 * If you provide an AWS account for centralization,
 * the rule evaluates the Amazon GuardDuty results in the centralized account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/guardduty-enabled-centralized.html
 */
ManagedRuleIdentifiers.GUARDDUTY_ENABLED_CENTRALIZED = 'GUARDDUTY_ENABLED_CENTRALIZED';
/**
 * Checks whether the Amazon GuardDuty has findings that are non archived.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/guardduty-non-archived-findings.html
 */
ManagedRuleIdentifiers.GUARDDUTY_NON_ARCHIVED_FINDINGS = 'GUARDDUTY_NON_ARCHIVED_FINDINGS';
/**
 * Checks that inline policy feature is not in use.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-no-inline-policy-check.html
 */
ManagedRuleIdentifiers.IAM_NO_INLINE_POLICY_CHECK = 'IAM_NO_INLINE_POLICY_CHECK';
/**
 * Checks whether IAM groups have at least one IAM user.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-group-has-users-check.html
 */
ManagedRuleIdentifiers.IAM_GROUP_HAS_USERS_CHECK = 'IAM_GROUP_HAS_USERS_CHECK';
/**
 * Checks whether the account password policy for IAM users meets the specified requirements indicated in the parameters.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-password-policy.html
 */
ManagedRuleIdentifiers.IAM_PASSWORD_POLICY = 'IAM_PASSWORD_POLICY';
/**
 * Checks whether for each IAM resource, a policy ARN in the input parameter is attached to the IAM resource.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-blacklisted-check.html
 */
ManagedRuleIdentifiers.IAM_POLICY_BLOCKED_CHECK = 'IAM_POLICY_BLACKLISTED_CHECK';
/**
 * Checks whether the IAM policy ARN is attached to an IAM user, or an IAM group with one or more IAM users, or an IAM role with one or more trusted entity.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-in-use.html
 */
ManagedRuleIdentifiers.IAM_POLICY_IN_USE = 'IAM_POLICY_IN_USE';
/**
 * Checks the IAM policies that you create for Allow statements that grant permissions to all actions on all resources.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-policy-no-statements-with-admin-access.html
 */
ManagedRuleIdentifiers.IAM_POLICY_NO_STATEMENTS_WITH_ADMIN_ACCESS = 'IAM_POLICY_NO_STATEMENTS_WITH_ADMIN_ACCESS';
/**
 * Checks that AWS Identity and Access Management (IAM) policies in a list of policies are attached to all AWS roles.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-role-managed-policy-check.html
 */
ManagedRuleIdentifiers.IAM_ROLE_MANAGED_POLICY_CHECK = 'IAM_ROLE_MANAGED_POLICY_CHECK';
/**
 * Checks whether the root user access key is available.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-root-access-key-check.html
 */
ManagedRuleIdentifiers.IAM_ROOT_ACCESS_KEY_CHECK = 'IAM_ROOT_ACCESS_KEY_CHECK';
/**
 * Checks whether IAM users are members of at least one IAM group.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-group-membership-check.html
 */
ManagedRuleIdentifiers.IAM_USER_GROUP_MEMBERSHIP_CHECK = 'IAM_USER_GROUP_MEMBERSHIP_CHECK';
/**
 * Checks whether the AWS Identity and Access Management users have multi-factor authentication (MFA) enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-mfa-enabled.html
 */
ManagedRuleIdentifiers.IAM_USER_MFA_ENABLED = 'IAM_USER_MFA_ENABLED';
/**
 * Checks that none of your IAM users have policies attached.
 *
 * IAM users must inherit permissions from IAM groups or roles.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-no-policies-check.html
 */
ManagedRuleIdentifiers.IAM_USER_NO_POLICIES_CHECK = 'IAM_USER_NO_POLICIES_CHECK';
/**
 * Checks whether your AWS Identity and Access Management (IAM) users have passwords or active access keys that have not been used within the specified number of days you provided.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/iam-user-unused-credentials-check.html
 */
ManagedRuleIdentifiers.IAM_USER_UNUSED_CREDENTIALS_CHECK = 'IAM_USER_UNUSED_CREDENTIALS_CHECK';
/**
 * Checks that Internet gateways (IGWs) are only attached to an authorized Amazon Virtual Private Cloud (VPCs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/internet-gateway-authorized-vpc-only.html
 */
ManagedRuleIdentifiers.INTERNET_GATEWAY_AUTHORIZED_VPC_ONLY = 'INTERNET_GATEWAY_AUTHORIZED_VPC_ONLY';
/**
 * Checks whether customer master keys (CMKs) are not scheduled for deletion in AWS Key Management Service (KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/kms-cmk-not-scheduled-for-deletion.html
 */
ManagedRuleIdentifiers.KMS_CMK_NOT_SCHEDULED_FOR_DELETION = 'KMS_CMK_NOT_SCHEDULED_FOR_DELETION';
/**
 * Checks whether the AWS Lambda function is configured with function-level concurrent execution limit.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-concurrency-check.html
 */
ManagedRuleIdentifiers.LAMBDA_CONCURRENCY_CHECK = 'LAMBDA_CONCURRENCY_CHECK';
/**
 * Checks whether an AWS Lambda function is configured with a dead-letter queue.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-dlq-check.html
 */
ManagedRuleIdentifiers.LAMBDA_DLQ_CHECK = 'LAMBDA_DLQ_CHECK';
/**
 * Checks whether the AWS Lambda function policy attached to the Lambda resource prohibits public access.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-public-access-prohibited.html
 */
ManagedRuleIdentifiers.LAMBDA_FUNCTION_PUBLIC_ACCESS_PROHIBITED = 'LAMBDA_FUNCTION_PUBLIC_ACCESS_PROHIBITED';
/**
 * Checks that the lambda function settings for runtime, role, timeout, and memory size match the expected values.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-function-settings-check.html
 */
ManagedRuleIdentifiers.LAMBDA_FUNCTION_SETTINGS_CHECK = 'LAMBDA_FUNCTION_SETTINGS_CHECK';
/**
 * Checks whether an AWS Lambda function is in an Amazon Virtual Private Cloud.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/lambda-inside-vpc.html
 */
ManagedRuleIdentifiers.LAMBDA_INSIDE_VPC = 'LAMBDA_INSIDE_VPC';
/**
 * Checks whether AWS Multi-Factor Authentication (MFA) is enabled for all IAM users that use a console password.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/mfa-enabled-for-iam-console-access.html
 */
ManagedRuleIdentifiers.MFA_ENABLED_FOR_IAM_CONSOLE_ACCESS = 'MFA_ENABLED_FOR_IAM_CONSOLE_ACCESS';
/**
 * Checks that there is at least one multi-region AWS CloudTrail.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/multi-region-cloudtrail-enabled.html
 */
ManagedRuleIdentifiers.CLOUDTRAIL_MULTI_REGION_ENABLED = 'MULTI_REGION_CLOUD_TRAIL_ENABLED';
/**
 * Checks if an Amazon Relational Database Service (Amazon RDS) cluster has deletion protection enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-cluster-deletion-protection-enabled.html
 */
ManagedRuleIdentifiers.RDS_CLUSTER_DELETION_PROTECTION_ENABLED = 'RDS_CLUSTER_DELETION_PROTECTION_ENABLED';
/**
 * Checks if an Amazon Relational Database Service (Amazon RDS) instance has deletion protection enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-deletion-protection-enabled.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_DELETION_PROTECTION_ENABLED = 'RDS_INSTANCE_DELETION_PROTECTION_ENABLED';
/**
 * Checks if an Amazon RDS instance has AWS Identity and Access Management (IAM) authentication enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-iam-authentication-enabled.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_IAM_AUTHENTICATION_ENABLED = 'RDS_INSTANCE_IAM_AUTHENTICATION_ENABLED';
/**
 * Checks that respective logs of Amazon Relational Database Service (Amazon RDS) are enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-logging-enabled.html
 */
ManagedRuleIdentifiers.RDS_LOGGING_ENABLED = 'RDS_LOGGING_ENABLED';
/**
 * Checks that Amazon Redshift automated snapshots are enabled for clusters.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-backup-enabled.html
 */
ManagedRuleIdentifiers.REDSHIFT_BACKUP_ENABLED = 'REDSHIFT_BACKUP_ENABLED';
/**
 * Checks whether enhanced monitoring is enabled for Amazon Relational Database Service (Amazon RDS) instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-enhanced-monitoring-enabled.html
 */
ManagedRuleIdentifiers.RDS_ENHANCED_MONITORING_ENABLED = 'RDS_ENHANCED_MONITORING_ENABLED';
/**
 * Checks whether Amazon Relational Database Service (Amazon RDS) DB snapshots are encrypted.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshot-encrypted.html
 */
ManagedRuleIdentifiers.RDS_SNAPSHOT_ENCRYPTED = 'RDS_SNAPSHOT_ENCRYPTED';
/**
 * Checks whether Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-require-tls-ssl.html
 */
ManagedRuleIdentifiers.REDSHIFT_REQUIRE_TLS_SSL = 'REDSHIFT_REQUIRE_TLS_SSL';
/**
 * Checks whether Amazon RDS database is present in back plans of AWS Backup.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-in-backup-plan.html
 */
ManagedRuleIdentifiers.RDS_IN_BACKUP_PLAN = 'RDS_IN_BACKUP_PLAN';
/**
 * Check whether the Amazon Relational Database Service instances are not publicly accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-instance-public-access-check.html
 */
ManagedRuleIdentifiers.RDS_INSTANCE_PUBLIC_ACCESS_CHECK = 'RDS_INSTANCE_PUBLIC_ACCESS_CHECK';
/**
 * Checks whether high availability is enabled for your RDS DB instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-multi-az-support.html
 */
ManagedRuleIdentifiers.RDS_MULTI_AZ_SUPPORT = 'RDS_MULTI_AZ_SUPPORT';
/**
 * Checks if Amazon Relational Database Service (Amazon RDS) snapshots are public.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-snapshots-public-prohibited.html
 */
ManagedRuleIdentifiers.RDS_SNAPSHOTS_PUBLIC_PROHIBITED = 'RDS_SNAPSHOTS_PUBLIC_PROHIBITED';
/**
 * Checks whether storage encryption is enabled for your RDS DB instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/rds-storage-encrypted.html
 */
ManagedRuleIdentifiers.RDS_STORAGE_ENCRYPTED = 'RDS_STORAGE_ENCRYPTED';
/**
 * Checks whether Amazon Redshift clusters have the specified settings.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-configuration-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_CONFIGURATION_CHECK = 'REDSHIFT_CLUSTER_CONFIGURATION_CHECK';
/**
 * Checks whether Amazon Redshift clusters have the specified maintenance settings.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-maintenancesettings-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_MAINTENANCE_SETTINGS_CHECK = 'REDSHIFT_CLUSTER_MAINTENANCESETTINGS_CHECK';
/**
 * Checks whether Amazon Redshift clusters are not publicly accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/redshift-cluster-public-access-check.html
 */
ManagedRuleIdentifiers.REDSHIFT_CLUSTER_PUBLIC_ACCESS_CHECK = 'REDSHIFT_CLUSTER_PUBLIC_ACCESS_CHECK';
/**
 * Checks whether your resources have the tags that you specify.
 *
 * For example, you can check whether your Amazon EC2 instances have the CostCenter tag.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/required-tags.html
 */
ManagedRuleIdentifiers.REQUIRED_TAGS = 'REQUIRED_TAGS';
/**
 * Checks whether the security groups in use do not allow unrestricted incoming TCP traffic to the specified ports.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/restricted-common-ports.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_RESTRICTED_INCOMING_TRAFFIC = 'RESTRICTED_INCOMING_TRAFFIC';
/**
 * Checks whether the incoming SSH traffic for the security groups is accessible.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/restricted-ssh.html
 */
ManagedRuleIdentifiers.EC2_SECURITY_GROUPS_INCOMING_SSH_DISABLED = 'INCOMING_SSH_DISABLED';
/**
 * Checks whether your AWS account is enabled to use multi-factor authentication (MFA) hardware device to sign in with root credentials.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/root-account-hardware-mfa-enabled.html
 */
ManagedRuleIdentifiers.ROOT_ACCOUNT_HARDWARE_MFA_ENABLED = 'ROOT_ACCOUNT_HARDWARE_MFA_ENABLED';
/**
 * Checks whether users of your AWS account require a multi-factor authentication (MFA) device to sign in with root credentials.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/root-account-mfa-enabled.html
 */
ManagedRuleIdentifiers.ROOT_ACCOUNT_MFA_ENABLED = 'ROOT_ACCOUNT_MFA_ENABLED';
/**
 * Checks whether Amazon Simple Storage Service (Amazon S3) bucket has lock enabled, by default.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-default-lock-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_DEFAULT_LOCK_ENABLED = 'S3_BUCKET_DEFAULT_LOCK_ENABLED';
/**
 * Checks whether the Amazon Simple Storage Service (Amazon S3) buckets are encrypted with AWS Key Management Service (AWS KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-default-encryption-kms.html
 */
ManagedRuleIdentifiers.S3_DEFAULT_ENCRYPTION_KMS = 'S3_DEFAULT_ENCRYPTION_KMS';
/**
 * Checks that AWS Security Hub is enabled for an AWS account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/securityhub-enabled.html
 */
ManagedRuleIdentifiers.SECURITYHUB_ENABLED = 'SECURITYHUB_ENABLED';
/**
 * Checks whether Amazon SNS topic is encrypted with AWS Key Management Service (AWS KMS).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sns-encrypted-kms.html
 */
ManagedRuleIdentifiers.SNS_ENCRYPTED_KMS = 'SNS_ENCRYPTED_KMS';
/**
 * Checks whether the required public access block settings are configured from account level.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-account-level-public-access-blocks.html
 */
ManagedRuleIdentifiers.S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS = 'S3_ACCOUNT_LEVEL_PUBLIC_ACCESS_BLOCKS';
/**
 * Checks that the Amazon Simple Storage Service bucket policy does not allow blocked bucket-level and object-level actions on resources in the bucket for principals from other AWS accounts.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-blacklisted-actions-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_BLOCKED_ACTIONS_PROHIBITED = 'S3_BUCKET_BLACKLISTED_ACTIONS_PROHIBITED';
/**
 * Verifies that your Amazon Simple Storage Service bucket policies do not allow other inter-account permissions than the control Amazon S3 bucket policy provided.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-not-more-permissive.html
 */
ManagedRuleIdentifiers.S3_BUCKET_POLICY_NOT_MORE_PERMISSIVE = 'S3_BUCKET_POLICY_NOT_MORE_PERMISSIVE';
/**
 * Checks whether logging is enabled for your S3 buckets.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-logging-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_LOGGING_ENABLED = 'S3_BUCKET_LOGGING_ENABLED';
/**
 * Checks that the access granted by the Amazon S3 bucket is restricted by any of the AWS principals, federated users, service principals, IP addresses, or VPCs that you provide.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-policy-grantee-check.html
 */
ManagedRuleIdentifiers.S3_BUCKET_POLICY_GRANTEE_CHECK = 'S3_BUCKET_POLICY_GRANTEE_CHECK';
/**
 * Checks that your Amazon S3 buckets do not allow public read access.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-read-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_PUBLIC_READ_PROHIBITED = 'S3_BUCKET_PUBLIC_READ_PROHIBITED';
/**
 * Checks that your Amazon S3 buckets do not allow public write access.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-public-write-prohibited.html
 */
ManagedRuleIdentifiers.S3_BUCKET_PUBLIC_WRITE_PROHIBITED = 'S3_BUCKET_PUBLIC_WRITE_PROHIBITED';
/**
 * Checks whether S3 buckets have cross-region replication enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-replication-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_REPLICATION_ENABLED = 'S3_BUCKET_REPLICATION_ENABLED';
/**
 * Checks that your Amazon S3 bucket either has Amazon S3 default encryption enabled or that the S3 bucket policy explicitly denies put-object requests without server side encryption that uses AES-256 or AWS Key Management Service.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-server-side-encryption-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_SERVER_SIDE_ENCRYPTION_ENABLED = 'S3_BUCKET_SERVER_SIDE_ENCRYPTION_ENABLED';
/**
 * Checks whether S3 buckets have policies that require requests to use Secure Socket Layer (SSL).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-ssl-requests-only.html
 */
ManagedRuleIdentifiers.S3_BUCKET_SSL_REQUESTS_ONLY = 'S3_BUCKET_SSL_REQUESTS_ONLY';
/**
 * Checks whether versioning is enabled for your S3 buckets.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/s3-bucket-versioning-enabled.html
 */
ManagedRuleIdentifiers.S3_BUCKET_VERSIONING_ENABLED = 'S3_BUCKET_VERSIONING_ENABLED';
/**
 * Checks whether AWS Key Management Service (KMS) key is configured for an Amazon SageMaker endpoint configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-endpoint-configuration-kms-key-configured.html
 */
ManagedRuleIdentifiers.SAGEMAKER_ENDPOINT_CONFIGURATION_KMS_KEY_CONFIGURED = 'SAGEMAKER_ENDPOINT_CONFIGURATION_KMS_KEY_CONFIGURED';
/**
 * Check whether an AWS Key Management Service (KMS) key is configured for SageMaker notebook instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-instance-kms-key-configured.html
 */
ManagedRuleIdentifiers.SAGEMAKER_NOTEBOOK_INSTANCE_KMS_KEY_CONFIGURED = 'SAGEMAKER_NOTEBOOK_INSTANCE_KMS_KEY_CONFIGURED';
/**
 * Checks whether direct internet access is disabled for an Amazon SageMaker notebook instance.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/sagemaker-notebook-no-direct-internet-access.html
 */
ManagedRuleIdentifiers.SAGEMAKER_NOTEBOOK_NO_DIRECT_INTERNET_ACCESS = 'SAGEMAKER_NOTEBOOK_NO_DIRECT_INTERNET_ACCESS';
/**
 * Checks whether AWS Secrets Manager secret has rotation enabled.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-rotation-enabled-check.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_ROTATION_ENABLED_CHECK = 'SECRETSMANAGER_ROTATION_ENABLED_CHECK';
/**
 * Checks whether AWS Secrets Manager secret rotation has rotated successfully as per the rotation schedule.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/secretsmanager-scheduled-rotation-success-check.html
 */
ManagedRuleIdentifiers.SECRETSMANAGER_SCHEDULED_ROTATION_SUCCESS_CHECK = 'SECRETSMANAGER_SCHEDULED_ROTATION_SUCCESS_CHECK';
/**
 * Checks whether Service Endpoint for the service provided in rule parameter is created for each Amazon VPC.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/service-vpc-endpoint-enabled.html
 */
ManagedRuleIdentifiers.SERVICE_VPC_ENDPOINT_ENABLED = 'SERVICE_VPC_ENDPOINT_ENABLED';
/**
 * Checks whether EBS volumes are attached to EC2 instances.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/shield-advanced-enabled-autorenew.html
 */
ManagedRuleIdentifiers.SHIELD_ADVANCED_ENABLED_AUTO_RENEW = 'SHIELD_ADVANCED_ENABLED_AUTORENEW';
/**
 * Verify that DDoS response team (DRT) can access AWS account.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/shield-drt-access.html
 */
ManagedRuleIdentifiers.SHIELD_DRT_ACCESS = 'SHIELD_DRT_ACCESS';
/**
 * Checks that the default security group of any Amazon Virtual Private Cloud (VPC) does not allow inbound or outbound traffic.
 *
 * The rule returns NOT_APPLICABLE if the security group
 * is not default.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-default-security-group-closed.html
 */
ManagedRuleIdentifiers.VPC_DEFAULT_SECURITY_GROUP_CLOSED = 'VPC_DEFAULT_SECURITY_GROUP_CLOSED';
/**
 * Checks whether Amazon Virtual Private Cloud flow logs are found and enabled for Amazon VPC.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-flow-logs-enabled.html
 */
ManagedRuleIdentifiers.VPC_FLOW_LOGS_ENABLED = 'VPC_FLOW_LOGS_ENABLED';
/**
 * Checks whether the security group with 0.0.0.0/0 of any Amazon Virtual Private Cloud (Amazon VPC) allows only specific inbound TCP or UDP traffic.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-sg-open-only-to-authorized-ports.html
 */
ManagedRuleIdentifiers.VPC_SG_OPEN_ONLY_TO_AUTHORIZED_PORTS = 'VPC_SG_OPEN_ONLY_TO_AUTHORIZED_PORTS';
/**
 * Checks that both AWS Virtual Private Network tunnels provided by AWS Site-to-Site VPN are in UP status.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/vpc-vpn-2-tunnels-up.html
 */
ManagedRuleIdentifiers.VPC_VPN_2_TUNNELS_UP = 'VPC_VPN_2_TUNNELS_UP';
/**
 * Checks if logging is enabled on AWS Web Application Firewall (WAF) classic global web ACLs.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/waf-classic-logging-enabled.html
 */
ManagedRuleIdentifiers.WAF_CLASSIC_LOGGING_ENABLED = 'WAF_CLASSIC_LOGGING_ENABLED';
/**
 * Checks whether logging is enabled on AWS Web Application Firewall (WAFV2) regional and global web access control list (ACLs).
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/wafv2-logging-enabled.html
 */
ManagedRuleIdentifiers.WAFV2_LOGGING_ENABLED = 'WAFV2_LOGGING_ENABLED';
/**
 * Resources types that are supported by AWS Config.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html
 */
class ResourceType {
    constructor(type) {
        this.complianceResourceType = type;
    }
    /**
     * A custom resource type to support future cases.
     */
    static of(type) {
        return new ResourceType(type);
    }
}
exports.ResourceType = ResourceType;
/**
 * API Gateway Stage.
 */
ResourceType.APIGATEWAY_STAGE = new ResourceType('AWS::ApiGateway::Stage');
/**
 * API Gatewayv2 Stage.
 */
ResourceType.APIGATEWAYV2_STAGE = new ResourceType('AWS::ApiGatewayV2::Stage');
/**
 * API Gateway REST API.
 */
ResourceType.APIGATEWAY_REST_API = new ResourceType('AWS::ApiGateway::RestApi');
/**
 * API Gatewayv2 API.
 */
ResourceType.APIGATEWAYV2_API = new ResourceType('AWS::ApiGatewayV2::Api');
/**
 * Amazon CloudFront Distribution.
 */
ResourceType.CLOUDFRONT_DISTRIBUTION = new ResourceType('AWS::CloudFront::Distribution');
/**
 * Amazon CloudFront streaming distribution.
 */
ResourceType.CLOUDFRONT_STREAMING_DISTRIBUTION = new ResourceType('AWS::CloudFront::StreamingDistribution');
/**
 * Amazon CloudWatch Alarm.
 */
ResourceType.CLOUDWATCH_ALARM = new ResourceType('AWS::CloudWatch::Alarm');
/**
 * Amazon DynamoDB Table.
 */
ResourceType.DYNAMODB_TABLE = new ResourceType('AWS::DynamoDB::Table');
/**
 * Elastic Block Store (EBS) volume.
 */
ResourceType.EBS_VOLUME = new ResourceType('AWS::EC2::Volume');
/**
 * EC2 host.
 */
ResourceType.EC2_HOST = new ResourceType('AWS::EC2::Host');
/**
 * EC2 Elastic IP.
 */
ResourceType.EC2_EIP = new ResourceType('AWS::EC2::EIP');
/**
 * EC2 instance.
 */
ResourceType.EC2_INSTANCE = new ResourceType('AWS::EC2::Instance');
/**
 * EC2 security group.
 */
ResourceType.EC2_SECURITY_GROUP = new ResourceType('AWS::EC2::SecurityGroup');
/**
 * EC2 NAT gateway.
 */
ResourceType.EC2_NAT_GATEWAY = new ResourceType('AWS::EC2::NatGateway');
/**
 * EC2 Egress only internet gateway.
 */
ResourceType.EC2_EGRESS_ONLY_INTERNET_GATEWAY = new ResourceType('AWS::EC2::EgressOnlyInternetGateway');
/**
 * EC2 flow log.
 */
ResourceType.EC2_FLOW_LOG = new ResourceType('AWS::EC2::FlowLog');
/**
 * EC2 VPC endpoint.
 */
ResourceType.EC2_VPC_ENDPOINT = new ResourceType('AWS::EC2::VPCEndpoint');
/**
 * EC2 VPC endpoint service.
 */
ResourceType.EC2_VPC_ENDPOINT_SERVICE = new ResourceType('AWS::EC2::VPCEndpointService');
/**
 * EC2 VPC peering connection.
 */
ResourceType.EC2_VPC_PEERING_CONNECTION = new ResourceType('AWS::EC2::VPCPeeringConnection');
/**
 * Amazon ElasticSearch domain.
 */
ResourceType.ELASTICSEARCH_DOMAIN = new ResourceType('AWS::Elasticsearch::Domain');
/**
 * Amazon QLDB ledger.
 */
ResourceType.QLDB_LEDGER = new ResourceType('AWS::QLDB::Ledger');
/**
 * Amazon Redshift cluster.
 */
ResourceType.REDSHIFT_CLUSTER = new ResourceType('AWS::Redshift::Cluster');
/**
 * Amazon Redshift cluster parameter group.
 */
ResourceType.REDSHIFT_CLUSTER_PARAMETER_GROUP = new ResourceType('AWS::Redshift::ClusterParameterGroup');
/**
 * Amazon Redshift cluster security group.
 */
ResourceType.REDSHIFT_CLUSTER_SECURITY_GROUP = new ResourceType('AWS::Redshift::ClusterSecurityGroup');
/**
 * Amazon Redshift cluster snapshot.
 */
ResourceType.REDSHIFT_CLUSTER_SNAPSHOT = new ResourceType('AWS::Redshift::ClusterSnapshot');
/**
 * Amazon Redshift cluster subnet group.
 */
ResourceType.REDSHIFT_CLUSTER_SUBNET_GROUP = new ResourceType('AWS::Redshift::ClusterSubnetGroup');
/**
 * Amazon Redshift event subscription.
 */
ResourceType.REDSHIFT_EVENT_SUBSCRIPTION = new ResourceType('AWS::Redshift::EventSubscription');
/**
 * Amazon RDS database instance.
 */
ResourceType.RDS_DB_INSTANCE = new ResourceType('AWS::RDS::DBInstance');
/**
 * Amazon RDS database security group.
 */
ResourceType.RDS_DB_SECURITY_GROUP = new ResourceType('AWS::RDS::DBSecurityGroup');
/**
 * Amazon RDS database snapshot.
 */
ResourceType.RDS_DB_SNAPSHOT = new ResourceType('AWS::RDS::DBSnapshot');
/**
 * Amazon RDS database subnet group.
 */
ResourceType.RDS_DB_SUBNET_GROUP = new ResourceType('AWS::RDS::DBSubnetGroup');
/**
 * Amazon RDS event subscription.
 */
ResourceType.RDS_EVENT_SUBSCRIPTION = new ResourceType('AWS::RDS::EventSubscription');
/**
 * Amazon RDS database cluster.
 */
ResourceType.RDS_DB_CLUSTER = new ResourceType('AWS::RDS::DBCluster');
/**
 * Amazon RDS database cluster snapshot.
 */
ResourceType.RDS_DB_CLUSTER_SNAPSHOT = new ResourceType('AWS::RDS::DBClusterSnapshot');
/**
 * Amazon SQS queue.
 */
ResourceType.SQS_QUEUE = new ResourceType('AWS::SQS::Queue');
/**
 * Amazon SNS topic.
 */
ResourceType.SNS_TOPIC = new ResourceType('AWS::SNS::Topic');
/**
 * Amazon S3 bucket.
 */
ResourceType.S3_BUCKET = new ResourceType('AWS::S3::Bucket');
/**
 * Amazon S3 account public access block.
 */
ResourceType.S3_ACCOUNT_PUBLIC_ACCESS_BLOCK = new ResourceType('AWS::S3::AccountPublicAccessBlock');
/**
 * Amazon EC2 customer gateway.
 */
ResourceType.EC2_CUSTOMER_GATEWAY = new ResourceType('AWS::EC2::CustomerGateway');
/**
 * Amazon EC2 internet gateway.
 */
ResourceType.EC2_INTERNET_GATEWAY = new ResourceType('AWS::EC2::CustomerGateway');
/**
 * Amazon EC2 network ACL.
 */
ResourceType.EC2_NETWORK_ACL = new ResourceType('AWS::EC2::NetworkAcl');
/**
 * Amazon EC2 route table.
 */
ResourceType.EC2_ROUTE_TABLE = new ResourceType('AWS::EC2::RouteTable');
/**
 * Amazon EC2 subnet table.
 */
ResourceType.EC2_SUBNET = new ResourceType('AWS::EC2::Subnet');
/**
 * Amazon EC2 VPC.
 */
ResourceType.EC2_VPC = new ResourceType('AWS::EC2::VPC');
/**
 * Amazon EC2 VPN connection.
 */
ResourceType.EC2_VPN_CONNECTION = new ResourceType('AWS::EC2::VPNConnection');
/**
 * Amazon EC2 VPN gateway.
 */
ResourceType.EC2_VPN_GATEWAY = new ResourceType('AWS::EC2::VPNGateway');
/**
 * AWS Auto Scaling group.
 */
ResourceType.AUTO_SCALING_GROUP = new ResourceType('AWS::AutoScaling::AutoScalingGroup');
/**
 * AWS Auto Scaling launch configuration.
 */
ResourceType.AUTO_SCALING_LAUNCH_CONFIGURATION = new ResourceType('AWS::AutoScaling::LaunchConfiguration');
/**
 * AWS Auto Scaling policy.
 */
ResourceType.AUTO_SCALING_POLICY = new ResourceType('AWS::AutoScaling::ScalingPolicy');
/**
 * AWS Auto Scaling scheduled action.
 */
ResourceType.AUTO_SCALING_SCHEDULED_ACTION = new ResourceType('AWS::AutoScaling::ScheduledAction');
/**
 * AWS Certificate manager certificate.
 */
ResourceType.ACM_CERTIFICATE = new ResourceType('AWS::ACM::Certificate');
/**
 * AWS CloudFormation stack.
 */
ResourceType.CLOUDFORMATION_STACK = new ResourceType('AWS::CloudFormation::Stack');
/**
 * AWS CloudTrail trail.
 */
ResourceType.CLOUDTRAIL_TRAIL = new ResourceType('AWS::CloudTrail::Trail');
/**
 * AWS CodeBuild project.
 */
ResourceType.CODEBUILD_PROJECT = new ResourceType('AWS::CodeBuild::Project');
/**
 * AWS CodePipeline pipeline.
 */
ResourceType.CODEPIPELINE_PIPELINE = new ResourceType('AWS::CodePipeline::Pipeline');
/**
 * AWS Elastic Beanstalk (EB) application.
 */
ResourceType.ELASTIC_BEANSTALK_APPLICATION = new ResourceType('AWS::ElasticBeanstalk::Application');
/**
 * AWS Elastic Beanstalk (EB) application version.
 */
ResourceType.ELASTIC_BEANSTALK_APPLICATION_VERSION = new ResourceType('AWS::ElasticBeanstalk::ApplicationVersion');
/**
 * AWS Elastic Beanstalk (EB) environment.
 */
ResourceType.ELASTIC_BEANSTALK_ENVIRONMENT = new ResourceType('AWS::ElasticBeanstalk::Environment');
/**
 * AWS IAM user.
 */
ResourceType.IAM_USER = new ResourceType('AWS::IAM::User');
/**
 * AWS IAM group.
 */
ResourceType.IAM_GROUP = new ResourceType('AWS::IAM::Group');
/**
 * AWS IAM role.
 */
ResourceType.IAM_ROLE = new ResourceType('AWS::IAM::Role');
/**
 * AWS IAM policy.
 */
ResourceType.IAM_POLICY = new ResourceType('AWS::IAM::Policy');
/**
 * AWS KMS Key.
 */
ResourceType.KMS_KEY = new ResourceType('AWS::KMS::Key');
/**
 * AWS Lambda function.
 */
ResourceType.LAMBDA_FUNCTION = new ResourceType('AWS::Lambda::Function');
/**
 * AWS Secrets Manager secret.
 */
ResourceType.SECRETS_MANAGER_SECRET = new ResourceType('AWS::SecretsManager::Secret');
/**
 * AWS Service Catalog CloudFormation product.
 */
ResourceType.SERVICE_CATALOG_CLOUDFORMATION_PRODUCT = new ResourceType('AWS::ServiceCatalog::CloudFormationProduct');
/**
 * AWS Service Catalog CloudFormation provisioned product.
 */
ResourceType.SERVICE_CATALOG_CLOUDFORMATION_PROVISIONED_PRODUCT = new ResourceType('AWS::ServiceCatalog::CloudFormationProvisionedProduct');
/**
 * AWS Service Catalog portfolio.
 */
ResourceType.SERVICE_CATALOG_PORTFOLIO = new ResourceType('AWS::ServiceCatalog::Portfolio');
/**
 * AWS Shield protection.
 */
ResourceType.SHIELD_PROTECTION = new ResourceType('AWS::Shield::Protection');
/**
 * AWS Shield regional protection.
 */
ResourceType.SHIELD_REGIONAL_PROTECTION = new ResourceType('AWS::ShieldRegional::Protection');
/**
 * AWS Systems Manager managed instance inventory.
 */
ResourceType.SYSTEMS_MANAGER_MANAGED_INSTANCE_INVENTORY = new ResourceType('AWS::SSM::ManagedInstanceInventory');
/**
 * AWS Systems Manager patch compliance.
 */
ResourceType.SYSTEMS_MANAGER_PATCH_COMPLIANCE = new ResourceType('AWS::SSM::PatchCompliance');
/**
 * AWS Systems Manager association compliance.
 */
ResourceType.SYSTEMS_MANAGER_ASSOCIATION_COMPLIANCE = new ResourceType('AWS::SSM::AssociationCompliance');
/**
 * AWS Systems Manager file data.
 */
ResourceType.SYSTEMS_MANAGER_FILE_DATA = new ResourceType('AWS::SSM::FileData');
/**
 * AWS WAF rate based rule.
 */
ResourceType.WAF_RATE_BASED_RULE = new ResourceType('AWS::WAF::RateBasedRule');
/**
 * AWS WAF rule.
 */
ResourceType.WAF_RULE = new ResourceType('AWS::WAF::Rule');
/**
 * AWS WAF web ACL.
 */
ResourceType.WAF_WEB_ACL = new ResourceType('AWS::WAF::WebACL');
/**
 * AWS WAF rule group.
 */
ResourceType.WAF_RULE_GROUP = new ResourceType('AWS::WAF::RuleGroup');
/**
 * AWS WAF regional rate based rule.
 */
ResourceType.WAF_REGIONAL_RATE_BASED_RULE = new ResourceType('AWS::WAFRegional::RateBasedRule');
/**
 * AWS WAF regional rule.
 */
ResourceType.WAF_REGIONAL_RULE = new ResourceType('AWS::WAFRegional::Rule');
/**
 * AWS WAF web ACL.
 */
ResourceType.WAF_REGIONAL_WEB_ACL = new ResourceType('AWS::WAFRegional::WebACL');
/**
 * AWS WAF regional rule group.
 */
ResourceType.WAF_REGIONAL_RULE_GROUP = new ResourceType('AWS::WAFRegional::RuleGroup');
/**
 * AWS WAFv2 web ACL.
 */
ResourceType.WAFV2_WEB_ACL = new ResourceType('AWS::WAFv2::WebACL');
/**
 * AWS WAFv2 rule group.
 */
ResourceType.WAFV2_RULE_GROUP = new ResourceType('AWS::WAFv2::RuleGroup');
/**
 * AWS WAFv2 managed rule set.
 */
ResourceType.WAFV2_MANAGED_RULE_SET = new ResourceType('AWS::WAFv2::ManagedRuleSet');
/**
 * AWS X-Ray encryption configuration.
 */
ResourceType.XRAY_ENCRYPTION_CONFIGURATION = new ResourceType('AWS::XRay::EncryptionConfig');
/**
 * AWS ELB classic load balancer.
 */
ResourceType.ELB_LOAD_BALANCER = new ResourceType('AWS::ElasticLoadBalancing::LoadBalancer');
/**
 * AWS ELBv2 network load balancer or AWS ELBv2 application load balancer.
 */
ResourceType.ELBV2_LOAD_BALANCER = new ResourceType('AWS::ElasticLoadBalancingV2::LoadBalancer');
function renderScope(ruleScope) {
    var _a;
    return ruleScope ? {
        complianceResourceId: ruleScope.resourceId,
        complianceResourceTypes: (_a = ruleScope.resourceTypes) === null || _a === void 0 ? void 0 : _a.map(resource => resource.complianceResourceType),
        tagKey: ruleScope.key,
        tagValue: ruleScope.value,
    } : undefined;
}
//# sourceMappingURL=data:application/json;base64,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