"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackNotificationCheck = exports.CloudFormationStackDriftDetectionCheck = exports.AccessKeysRotated = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: rule_1.ManagedRuleIdentifiers.ACCESS_KEYS_ROTATED,
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: rule_1.ManagedRuleIdentifiers.CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK,
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.ruleScope = rule_1.RuleScope.fromResource(rule_1.ResourceType.CLOUDFORMATION_STACK, props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to a SNS topic.
 *
 * Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: rule_1.ManagedRuleIdentifiers.CLOUDFORMATION_STACK_NOTIFICATION_CHECK,
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
            ruleScope: rule_1.RuleScope.fromResources([rule_1.ResourceType.CLOUDFORMATION_STACK]),
        });
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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