# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['domify']

package_data = \
{'': ['*']}

extras_require = \
{':python_version < "3.8"': ['typing-extensions>=3.7,<4.0']}

setup_kwargs = {
    'name': 'domify',
    'version': '0.1.0',
    'description': 'HTML generator using pure Python',
    'long_description': '# domify\n\nHTML generator using pure Python.\n\n[![PyPI](https://img.shields.io/pypi/v/domify)](https://pypi.org/project/domify/)\n[![PyPI - Status](https://img.shields.io/pypi/status/dominate)](https://pypi.org/project/domify/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/domify)](https://pypi.org/project/domify/)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Tests status](https://img.shields.io/github/workflow/status/Parnassius/domify/main/main?event=push&label=tests)](https://github.com/Parnassius/domify/actions?query=workflow%3Amain)\n[![Codecov](https://img.shields.io/codecov/c/gh/Parnassius/dominate/main?token=RYDAXOWCUS)](https://codecov.io/gh/Parnassius/dominate)\n[![PyPI - License](https://img.shields.io/pypi/l/domify)](https://github.com/Parnassius/domify/blob/main/LICENSE)\n\nSimple example:\n```python\nfrom domify import html_elements as e\n\nhtml = e.Html(lang="en")\nwith html:\n    with e.Head():\n        e.Meta(charset="utf-8")\n        e.Title("Example page")\n\n    with e.Body():\n        e.H1("Hello world")\n        e.P("Lorem ipsum ", e.I("dolor sit amet"))\n\nprint(str(html))\n```\n\nHTML output (formatted for legibility):\n```html\n<html lang="en">\n  <head>\n    <meta charset="utf-8">\n    <title>Example page</title>\n  </head>\n  <body>\n    <h1>Hello world</h1>\n    <p>Lorem ipsum <i>dolor sit amet</i></p>\n  </body>\n</html>\n```\n\nNote: every example on this page assumes domify has already been imported:\n```python\nfrom domify import html_elements as e\n```\n\n`domify.html_elements` contains a class for each HTML element, with the first letter\nconverted to uppercase:\n```python\np = e.P("Lorem ipsum dolor sit amet")\nprint(str(p))\n```\n```html\n<p>Lorem ipsum dolor sit amet</p>\n```\n\nYou can pass strings or additional elements as positional arguments, and they will be\ntreated as children:\n```python\np = e.P("Lorem ipsum dolor sit amet", e.Br(), "consectetur adipiscing elit.")\nprint(str(p))\n```\n```html\n<p>\n  Lorem ipsum dolor sit amet\n  <br>\n  consectetur adipiscing elit.\n</p>\n```\n\nAttributes can be passed as keyword arguments, optionally appending an underscore to\nattributes which are reserved keywords in python (for example `class`). Integers and\nfloats are automatically converted to strings, while additional underscores are replaced\nwith dashes (especially useful for `data` and `aria` attributes):\n```python\nbtn = e.Button("Click!", type="submit", class_="someclass", data_order=155)\nprint(str(btn))\n```\n```html\n<button type="submit" class="someclass" data-order="155">Click!</button>\n```\n\nHTML5 boolean attributes accept a boolean value instead:\n```python\ntext_input = e.Input(type="text", required=True, disabled=False)\nprint(str(text_input))\n```\n```html\n<input type="text" required>\n```\n\nAttributes can be added or modified by subscripting an element object:\n```python\nhtml = e.Html()\nhtml["lang"] = "en"\nprint(str(html))\n```\n```html\n<html lang="en"></html>\n```\n\nChildren can be modified (but not added) by subscripting as well:\n```python\np = e.P("Text node 1", e.Br(), "Text node 2")\np[0] = "Text node 3"\np[1] = t.Hr()\nprint(str(p))\n```\n```html\n<p>\n  Text node 3\n  <hr>\n  Text node 2\n</p>\n```\n\nThe `del` keyword can be used to delete both attributes and children:\n```python\ndiv = e.Div("foo", e.Br(), "bar", id="someid", class_="someclass")\ndel div["class"]\ndel div[1]\nprint(str(div))\n```\n```html\n<div id="someid">foobar</div>\n```\n\n`add_class` and `remove_class` can be used to manage classes:\n```python\ndiv = e.Div(class_="some-class some-other-class")\ndiv.remove_class("some-class")\ndiv.add_class("third-class")\nprint(str(div))\n```\n```html\n<div class="some-other-class third-class"></div>\n```\n\nChildren can be added using the `add` method, which return the newly added element:\n```python\np = e.P()\nspan = p.add(e.Span())\nspan.add("First line")\nspan.add(e.Br())\nspan.add("Second line")\nprint(str(p))\n```\n```html\n<p>\n  <span>\n    First line\n    <br>\n    Second line\n  </span>\n</p>\n```\n\nContext managers can be used to add child elements too:\n```python\nwith e.Select() as select:\n    e.Option("Option 1", value=1)\n    e.Option("Option 2", value=2)\n    e.Option("Option 3", value=3)\nprint(str(select))\n```\n```html\n<select>\n  <option value="1">Option 1</option>\n  <option value="2">Option 2</option>\n  <option value="3">Option 3</option>\n</select>\n```\n\nThe `+` operator can be used to concatenate multiple elements and/or strings:\n```python\nusername = "Username: " + e.Input(type="text", name="username")\npassword = "Password: " + e.Input(type="password", name="password")\nlogin_form = e.Label(username) + e.Label(password)\nlogin_form += e.Button("Login", type="submit")\nprint(str(login_form))\n```\n```html\n<label>Username: <input type="text" name="username"></label>\n<label>Password: <input type="password" name="password"></label>\n<button type="submit">Login</button>\n```\n\nText nodes can be created using `TextNode` (this is usually not required, since strings\nare automatically coverted):\n```python\nwith e.P() as p:\n    e.TextNode("Lorem ipsum")\nprint(str(p))\n```\n```html\n<p>Lorem ipsum</p>\n```\n\n`RawTextNode` can be used to add strings without escaping:\n```python\nwith e.P() as p:\n    e.TextNode("<i>TextNode</i>")\n    e.Br()\n    e.RawTextNode("<i>RawTextNode</i>")\nprint(str(p))\n```\n```html\n<p>\n  &lt;i&gt;TextNode&lt;/i&gt;\n  <br>\n  <i>RawTextNode</i>\n</p>\n```\n',
    'author': 'Parnassius',
    'author_email': 'Parnassius@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Parnassius/domify',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
