from datetime import datetime
from pathlib import Path
from typing import Optional, Union, Dict, ClassVar, Literal

import yaml
from pydantic import BaseModel, AnyUrl, ValidationError

from .leaderboard import PublicationEntry
from ..misc import load_obj

BenchmarkList = Literal["sLM21", "abxLS", "abx17", "abx15", "tde17", "tde15", ""]


class PublicationInfo(BaseModel):
    author_label: str = ""
    authors: str
    paper_title: Optional[str]
    paper_url: Optional[str]
    publication_year: int
    bib_reference: Optional[str]
    DOI: Optional[str]
    institution: str
    team: Optional[str]


class ModelInfo(BaseModel):
    model_id: Optional[str]
    system_description: str
    train_set: str
    gpu_budget: Optional[str]


class MetaFile(BaseModel):
    username: Optional[str]
    benchmark_name: BenchmarkList
    model_info: ModelInfo
    publication: PublicationInfo
    open_source: bool
    code_url: Optional[Union[AnyUrl, str]]
    file_stem: ClassVar[str] = "meta.yaml"

    @classmethod
    def from_file(cls, file: Path, enforce: bool = False):
        if not file.is_file() and not enforce:
            return None
        return cls.parse_obj(load_obj(file))

    @classmethod
    def to_template(cls, benchmark_name: BenchmarkList):
        return cls(
            username="<optional(str>): username on the zerospeech.com platform>",
            benchmark_name=benchmark_name,
            model_info=ModelInfo(
                model_id="<auto-generated>",
                system_description="<required(str): a description of the system>",
                train_set="<required(str): the dateset used to train the system>",
                gpu_budget="<optional(str): the number of gpu hours used to train the system>",
            ),
            publication=PublicationInfo(
                author_label="<required(str): a short label used for reference (ex: author1 et al.)>",
                authors="<required(str): the full names of the authors of the system>",
                paper_title="<optional(str): the title of the paper referencing the system/submission>",
                paper_url="<optional(str): A URL referencing the paper online (arxiv.org or other)>",
                publication_year=datetime.now().year,
                institution="<required(str): name of the institution (University, company, etc..)>",
                team="<optional(str): name of the team>"
            ),
            open_source=True,
            code_url="<optional(str): a url to a github or other with the code used to create this system>"
        )

    def get_publication_info(self) -> PublicationEntry:
        pub = self.publication
        paper_ref = None
        if pub.authors and pub.paper_title:
            paper_ref = f"{pub.authors} ({pub.publication_year} {pub.paper_title})"

        return PublicationEntry(
            author_short=pub.author_label,
            authors=pub.authors,
            paper_title=pub.paper_title,
            paper_ref=paper_ref,
            bib_ref=pub.bib_reference,
            paper_url=pub.paper_url,
            pub_year=pub.publication_year,
            team_name=pub.team,
            institution=pub.institution,
            code=self.code_url,
            DOI=pub.DOI,
            open_science=self.open_source
        )

    def to_yaml(self, file: Path, excluded: Dict):
        with file.open("w") as fp:
            yaml.dump(dict(self._iter(to_dict=True, exclude=excluded)), fp)

    @classmethod
    def benchmark_from_submission(cls, location: Path) -> Optional[BenchmarkList]:
        """ Extract the benchmark name from a given submission """
        meta_file = location / cls.file_stem
        if not meta_file.is_file():
            return None

        with meta_file.open() as fp:
            meta_obj = yaml.load(fp, Loader=yaml.FullLoader)
        try:
            meta = cls.parse_obj(meta_obj)
            return meta.benchmark_name
        except ValidationError:
            return None
